package com.bxm.localnews.activity.controller;

import java.util.List;

import com.bxm.localnews.activity.dto.CalendarSignDTO;
import com.bxm.localnews.activity.dto.SignDTO;
import com.bxm.localnews.activity.dto.SignLeaderBoard;
import com.bxm.localnews.activity.service.sign.DailySignService;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.Json;
import com.bxm.newidea.component.vo.Message;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;

@Api(tags = "2-01 每日签到接口", description = "签到必须登录后才可以访问")
@RestController
@RequestMapping("api/sign/public")
public class DailySignController {

    @Resource
    private DailySignService dailySignService;

    @ApiOperation(value = "2-01-1 获取用户的签到记录", notes = "签到页面相关信息：除了日历签到列表信息，其余相关信息都在此接口返回")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "用户ID", required = true),
            @ApiImplicitParam(name = "areaCode", value = "地区编号", required = true)
    })
    @GetMapping(value = "signRecord")
    public Json<SignDTO> signRecord(@RequestParam(value = "userId") Long userId,
                                    @RequestParam(value = "areaCode") String areaCode) {
        return ResultUtil.genSuccessResult(dailySignService.signRecord(userId,areaCode));
    }

    @ApiOperation(value = "2-01-2 获取用户对应签到的月历", notes = "签到日历及其相关信息")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "用户ID", required = true)
    })
    @GetMapping(value = "listSignRecord")
    public Json<List<CalendarSignDTO>> listSignRecord(@RequestParam(value = "userId") Long userId) {
        return ResultUtil.genSuccessResult(dailySignService.listSignRecord(userId));
    }

    @ApiOperation(value = "2-01-3 用户签到", notes = "用户进行签到---只针对当天签到")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "用户ID", required = true),
            @ApiImplicitParam(name = "areaCode", value = "地区编号", required = true)
    })
    @PostMapping()
    public Json<Boolean> userSign(@RequestParam(value = "userId") Long userId,
                                  @RequestParam(value = "areaCode") String areaCode) {
        Message message = dailySignService.executeUserSign(userId,areaCode);
        if (!message.isSuccess()) {
            return ResultUtil.genFailedResult(message.getLastMessage());
        }
        return ResultUtil.genSuccessResult(message.isSuccess());
    }

    @ApiOperation(value = "2-01-4 用户补签", notes = "用户进行补签---只针对昨日签到")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "用户ID", required = true),
            @ApiImplicitParam(name = "areaCode", value = "地区编号", required = true)
    })
    @PostMapping("/fill")
    public Json<Boolean> userFillSign(@RequestParam(value = "userId") Long userId,@RequestParam(value = "areaCode") String areaCode) {
        Message message = dailySignService.executeUserFillSign(userId,areaCode);
        if (!message.isSuccess()) {
            return ResultUtil.genFailedResult(message.getLastMessage());
        }
        return ResultUtil.genSuccessResult(message.isSuccess());
    }

    @ApiOperation(value = "2-01-5 获取某个地区的签到排行榜以及自己的签到信息", notes = "" +
            "1)签到排行榜数据区分地区，根据最近一次签到时所选的地区决定，更换地区签到不影响连续签到天数\n" +
            "2)按照连续签到天数进行降序排名，取前20名，若天数相同，则id号小的排名靠前\n" +
            "3)前3名若无数据，则显示如右图所示，虚位以待\n" +
            "4)左上角“广德”仅供显示当前排行榜的所属地区，不可点击\n" +
            "5)页面底部置底悬浮显示当前访问用户的信息：排名、头像、昵称、连续签到天数。\n" +
            ">若当前用户已在当前地区签到但未进入前20名，则排名显示为：暂未上榜\n" +
            ">若当前用户未在当前地区签到，则连续签到天数显示为：还未在广德签到。地区名动态显示\n" +
            "6)排行榜数据需尽量实时更新")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "用户ID", required = true),
            @ApiImplicitParam(name = "areaCode", value = "地区编号", required = true)
    })
    @GetMapping("/ranking")
    public Json<SignLeaderBoard> listSignLeaderBoard(@RequestParam(value = "userId") Long userId,
                                                     @RequestParam(value = "areaCode") String areaCode) {
        return ResultUtil.genSuccessResult(dailySignService.getSignLeaderBoard(userId, areaCode));
    }
}
