package com.bxm.localnews.activity.controller;

import com.bxm.localnews.activity.common.config.ActivityProperties;
import com.bxm.localnews.base.service.BaseUrlFacadeService;
import com.bxm.localnews.common.config.BizConfigProperties;
import com.bxm.localnews.common.vo.Json;
import com.bxm.newidea.component.CodeCreaterFactory;
import com.bxm.newidea.component.controller.BaseController;
import com.bxm.newidea.component.oss.service.AliyunOSSService;
import com.bxm.newidea.component.param.QrCodeParam;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.StringUtils;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiOperation;
import lombok.AllArgsConstructor;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.Date;

import static com.bxm.localnews.common.constant.RedisConfig.USER_SHARE_IMAGES;
import static org.apache.commons.io.FileUtils.copyURLToFile;
import static org.apache.commons.io.FileUtils.openInputStream;

@Api(tags = "2-04 h5相关接口", description = "h5相关接口操作")

@AllArgsConstructor
@RestController
@RequestMapping("/api/h5")
public class H5Controller extends BaseController {

    private RedisHashMapAdapter redisHashMapAdapter;

    private BizConfigProperties bizConfigProperties;

    private AliyunOSSService aliyunOSSService;

    private ActivityProperties activityProperties;

    private BaseUrlFacadeService baseUrlFacadeService;

    @ApiOperation(value = "2-04-1 获取用户分享的图片", notes = "")
    @ApiImplicitParam(name = "userId", value = "分享的用户ID")
    @GetMapping("share/img/{userId}")
    public Json<String> getShareImge(@PathVariable String userId) {
        //判断缓存是否存在图片路径
        String shareImg = redisHashMapAdapter.get(USER_SHARE_IMAGES, userId, String.class);
        if (StringUtils.isBlank(shareImg)) {
            try {
                //生成图片
                shareImg = gen(userId);
                if (StringUtils.isNotBlank(shareImg)) {
                    redisHashMapAdapter.put(USER_SHARE_IMAGES, userId, shareImg);
                }
            } catch (IOException e) {
                logger.error(e.getMessage(), e);
            }
        }
        return Json.build(shareImg);
    }

    private String getUrl(String userId) {
        String h5Host = baseUrlFacadeService.getInnerH5BaseUrl();
        return h5Host + "/downloadWx.html?userId=" + userId + "&chnl=invitenew";
    }

    private String gen(String userId) throws IOException {
        String rootPath = bizConfigProperties.getTempDir() + File.separator;

        //获取下载路径
        String url = getUrl(userId);

        //创建二维码
        File qrCodeFile = new File(rootPath + userId + ".jpg");
        QrCodeParam param = new QrCodeParam();
        param.setWidth(236);
        param.setHeight(236);
        param.setImageFile(qrCodeFile);
        param.setMessage(url);
        param.setLogoPath(getIconFile().getPath());
        CodeCreaterFactory.createQrCode(param);

        String suffix = "png";
        //生成图片
        File sourceFile = getSourceFile();
        BufferedImage finalImgBuffer = watermark(sourceFile, qrCodeFile, 256, 923, 1);
        String finalFileName = userId + "_final." + suffix;
        File finalFile = new File(rootPath + finalFileName);
        ImageIO.write(finalImgBuffer, suffix, finalFile);

        //上传到云端
        String dateStr = DateUtils.PATTERN_NO_DELIMITER_FORMAT.get().format(new Date());
        String imageUrl = aliyunOSSService.upload(openInputStream(finalFile), "share/" + dateStr + "/" + userId + "." + suffix);

        //删除文件
        qrCodeFile.delete();
        finalFile.delete();

        return imageUrl;
    }

    private File getSourceFile() {
        return getAliyunOssFile(activityProperties.getShareSourceImgUrl());
    }

    private File getIconFile() {
        return getAliyunOssFile(baseUrlFacadeService.getAppIconUrl());
    }

    private File getAliyunOssFile(String aliyunFilePath) {
        String fileName = StringUtils.substringAfterLast(aliyunFilePath, "/");

        String separator = StringUtils.EMPTY;
        if (!this.bizConfigProperties.getTempDir().endsWith("/") &&
                !this.bizConfigProperties.getTempDir().endsWith("\\")) {
            separator = File.separator;
        }

        File appFile = new File(this.bizConfigProperties.getTempDir() + separator + fileName);

        if (appFile.exists()) {
            return appFile;
        }

        File tempDirFile = new File(this.bizConfigProperties.getTempDir());
        if (!tempDirFile.exists()) {
            boolean result = tempDirFile.mkdirs();
            if (!result) {
                this.logger.error("创建[{}]目录失败，可能没有权限", this.bizConfigProperties.getTempDir());
            }
        }

        try {
            copyURLToFile(new URL(aliyunFilePath), appFile);
        } catch (IOException e) {
            this.logger.error(aliyunFilePath + "配置的下载文件路径不正确", e);
        }

        return appFile;
    }

    /**
     * @param file      源文件(图片)
     * @param waterFile 水印文件(图片)
     * @param x         距离右下角的X偏移量
     * @param y         距离右下角的Y偏移量
     * @param alpha     透明度, 选择值从0.0~1.0: 完全透明~完全不透明
     * @return BufferedImage
     * @throws IOException
     */
    private BufferedImage watermark(File file, File waterFile, int x, int y, float alpha) throws IOException {
        // 获取底图
        BufferedImage buffImg = ImageIO.read(file);
        // 获取层图
        BufferedImage waterImg = ImageIO.read(waterFile);
        // 创建Graphics2D对象，用在底图对象上绘图
        Graphics2D g2d = buffImg.createGraphics();
        // 获取层图的宽度
        int waterImgWidth = waterImg.getWidth();
        // 获取层图的高度
        int waterImgHeight = waterImg.getHeight();
        // 在图形和图像中实现混合和透明效果
        g2d.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_ATOP, alpha));
        // 绘制
        g2d.drawImage(waterImg, x, y, waterImgWidth, waterImgHeight, null);
        g2d.dispose();// 释放图形上下文使用的系统资源
        return buffImg;
    }

}






















