package com.bxm.localnews.activity.controller;

import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.activity.common.constant.PrivilegeStatusEnum;
import com.bxm.localnews.activity.dto.*;
import com.bxm.localnews.activity.param.*;
import com.bxm.localnews.activity.service.MerchantService;
import com.bxm.localnews.activity.service.PrivilegeParticipantService;
import com.bxm.localnews.activity.service.PrivilegeService;
import com.bxm.localnews.base.service.LocationFacadeService;
import com.bxm.localnews.common.annotation.TouTiaoAuth;
import com.bxm.localnews.common.constant.AppConst;
import com.bxm.localnews.common.constant.RespCode;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.facade.SmsSupplyFeignService;
import com.bxm.localnews.mq.common.constant.SmsTemplateEnum;
import com.bxm.newidea.component.enums.FileTypeEnum;
import com.bxm.newidea.component.util.WebUtils;
import com.bxm.newidea.component.vo.PageWarper;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.net.URL;
import java.util.List;
import java.util.Map;

/**
 * Created by Arei on 2018/10/17.
 */
@Api(tags = "2-06 优惠活动相关接口", description = "包括活动的相关接口操作")
@RestController
@RequestMapping("/api/privileges")
@Slf4j
public class PrivilegeController {

    @Resource
    private PrivilegeService privilegeService;

    @Resource
    private PrivilegeParticipantService privilegeParticipantService;

    @Resource
    private MerchantService merchantService;

    @Resource
    private SmsSupplyFeignService smsSupplyFeignService;

    @Resource
    private LocationFacadeService locationFacadeService;

    @ApiOperation(value = "2-06-1 获得优惠活动列表", notes = "如果该城市开通VIP,用户可以在VIP栏目可以看到该城市VIP所支持的店铺")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "longitude", value = "经度", required = false),
            @ApiImplicitParam(name = "latitude", value = "纬度", required = false),
            @ApiImplicitParam(name = "userId", value = "用户ID", required = false, dataType = "long"),
            @ApiImplicitParam(name = "areaCode", value = "地区编码", required = true)
    })
    @GetMapping
    public Json<PageWarper<PrivilegeListDTO>> listPrivileges(@Validated PrivilegeParam privilegeParam, BasicParam basicParam, HttpServletRequest request) {

        //ios 2.3.0版本有bug，错误提交了pageNum字段，进行特殊处理
        if (AppConst.PLATFORM.IOS == basicParam.getPlatform() && "2.3.0".equals(basicParam.getCurVer())) {
            Map<String, String[]> paramMap = request.getParameterMap();

            if (paramMap != null) {
                for (Map.Entry<String, String[]> entry : paramMap.entrySet()) {
                    if (StringUtils.isNotBlank(entry.getKey()) && NumberUtils.isDigits(entry.getKey())) {
                        int paramKey = NumberUtils.toInt(entry.getKey());

                        privilegeParam.setPageNum(paramKey);
                        break;
                    }
                }
            }

        }
        return Json.build(privilegeService.listPrivilege(privilegeParam, basicParam));
    }

    @ApiOperation(value = "2-06-2 获得优惠活动详情", notes = "显示享受VIP折扣的店铺详情,其中包含店铺名、营业时间、地址、电话、折扣、详细内容等")
    @GetMapping("/detail")
    public Json<PrivilegeDTO> getPrivilegeById(@Validated PrivilegeDetailParam privilegeDetailParam) {
        PrivilegeDTO privilegeDTO = privilegeService.getPrivilegeDetail(privilegeDetailParam);
        if (privilegeDTO == null) {
            return Json.build(RespCode.BAD_REQUEST, "活动不存在");
        }

        return Json.build(privilegeDTO);
    }

    @ApiOperation(value = "2-06-3 获得活动抽奖结果名单", notes = "")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "privilegeId", value = "活动id", dataType = "long", required = true),
            @ApiImplicitParam(name = "userId", value = "用户id", dataType = "long", required = false)
    })
    @GetMapping("/result")
    public Json<List<WinnerDTO>> getPrivilegeWinnersById(@Validated PrivilegeWinnerParam privilegeWinnerParam) {
        boolean isDisplayCommodityCode = true;
        if (privilegeWinnerParam.getPlatform() == AppConst.PLATFORM.WEB) {
            isDisplayCommodityCode = false;
        }
        return Json.build(privilegeService.listWinnerByPrivilegeId(
                privilegeWinnerParam.getPrivilegeId(),
                privilegeWinnerParam.getUserId(),
                isDisplayCommodityCode));
    }

    @ApiOperation(value = "2-06-4 参加活动", notes = "")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "privilegeId", value = "活动id", required = true),
            @ApiImplicitParam(name = "userId", value = "用户ID", required = true),
            @ApiImplicitParam(name = "inviteUserId", value = "邀请人用户ID", required = false),
            @ApiImplicitParam(name = "areaCode", value = "地区编码", required = true)
    })
    @TouTiaoAuth
    @PostMapping("/participate")
    public Json<ParticipateResultDTO> participate(@Validated ParticipatePrivilegeParam participatePrivilegeParam,
                                                  HttpServletRequest httpServletRequest) {
        PrivilegeDetailParam privilegeDetailParam = new PrivilegeDetailParam();
        privilegeDetailParam.setUserId(participatePrivilegeParam.getUserId());
        privilegeDetailParam.setId(participatePrivilegeParam.getPrivilegeId());

        PrivilegeDTO privilegeDTO = privilegeService.getPrivilegeDetail(privilegeDetailParam);
        if (privilegeDTO == null) {
            return Json.build(new ParticipateResultDTO(1)).setCode(RespCode.BAD_REQUEST).setErrorMsg("活动不存在");
        }
        PrivilegeStatusEnum statusEnum = PrivilegeStatusEnum.getEnumByStatus(privilegeDTO.getStatus());
        switch (statusEnum) {
            case END:
                return Json.build(new ParticipateResultDTO(1)).setCode(RespCode.BAD_REQUEST).setErrorMsg("活动已结束");
            case CANCEL:
                return Json.build(new ParticipateResultDTO(1)).setCode(RespCode.BAD_REQUEST).setErrorMsg("活动已取消");
            case WAIT_TO_START:
                return Json.build(new ParticipateResultDTO(1)).setCode(RespCode.BAD_REQUEST).setErrorMsg("活动未开始");
            case PROCESSING:
                break;
            default:
                return Json.build(new ParticipateResultDTO(3)).setCode(RespCode.INTERNAL_SERVER_ERROR).setErrorMsg("活动错误");

        }
        //验证是否邀请自己
        if (participatePrivilegeParam.getUserId().equals(participatePrivilegeParam.getInviteUserId())) {
            return Json.build(new ParticipateResultDTO(1)).setCode(RespCode.BAD_REQUEST).setErrorMsg("不能邀请自己");
        }
        //是否参加过
        if (privilegeDTO.getParticipated()) {
            return Json.build(new ParticipateResultDTO(1)).setCode(RespCode.BAD_REQUEST).setErrorMsg("已经参加过");
        }
        //判断地域
        if (participatePrivilegeParam.getPlatform() != 3) {
            if (!privilegeDTO.getGlobalFlag()) {
                if (!locationFacadeService.compareLocationCode(participatePrivilegeParam.getAreaCode(), privilegeDTO.getAreaCodeList())) {
                    return Json.build(new ParticipateResultDTO(2)).setCode(RespCode.BAD_REQUEST).setErrorMsg("不能跨地域参加活动");
                }
            }
        }
        //记录参加
        String ip = WebUtils.getIpAddr(httpServletRequest);
        int result = privilegeParticipantService.saveParticipantPrivilege(
                participatePrivilegeParam.getUserId(),
                participatePrivilegeParam.getInviteUserId(),
                participatePrivilegeParam.getPrivilegeId(),
                ip
        );
        return Json.build(new ParticipateResultDTO(0));
    }

    @ApiOperation(value = "2-06-5 当前用户邀请的好友列表", notes = "")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "privilegeId", value = "活动id", dataType = "long", required = true),
            @ApiImplicitParam(name = "userId", value = "用户ID", dataType = "long", required = true)
    })
    @TouTiaoAuth
    @GetMapping("/invitees")
    public Json<PageWarper<PrivilegeParticipantDTO>> getInviteesByUserId(@Validated PrivilegeParticipantParam privilegeParticipantParam) {
        PageWarper<PrivilegeParticipantDTO> privilegeParticipantDTOPageWarper
                = privilegeService.listPrivilegeParticipantByInvite(privilegeParticipantParam);
        return Json.build(privilegeParticipantDTOPageWarper);
    }

    @ApiOperation(value = "2-06-6 获得推荐活动列表", notes = "获取全国和分享用户(邀请人)所在地的活动，支持去掉分享的当前活动和用户已参加的活动")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "longitude", value = "经度", required = false),
            @ApiImplicitParam(name = "latitude", value = "纬度", required = false),
            @ApiImplicitParam(name = "privilegeId", value = "当前活动ID", required = false, dataType = "long"),
            @ApiImplicitParam(name = "userId", value = "用户ID", required = false, dataType = "long"),
            @ApiImplicitParam(name = "areaCode", value = "地区编码", required = false)
    })
    @GetMapping("/recommend")
    public Json<PageWarper<PrivilegeListDTO>> listRecommendPrivileges(PrivilegeRecommendParam privilegeRecommendParam) {
        return Json.build(privilegeService.listRecommendPrivilege(privilegeRecommendParam));
    }

    @ApiOperation(value = "2-06-7 商家报名", notes = "")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "name", value = "商家名字（公司名）", required = true),
            @ApiImplicitParam(name = "type", value = "商家类型", required = true),
            @ApiImplicitParam(name = "userId", value = "用户ID", required = true, dataType = "long"),
            @ApiImplicitParam(name = "contractor", value = "联系人姓名", required = true),
            @ApiImplicitParam(name = "code", value = "短信验证码", required = true),
            @ApiImplicitParam(name = "phone", value = "联系电话", required = true)
    })
    @PostMapping("/merchant")
    public Json newMerchant(@Validated MerchantParam merchantParam) {

        boolean result = smsSupplyFeignService.verifySmsCode(merchantParam.getPhone(), merchantParam.getCode(), SmsTemplateEnum.NORMAL_V_CODE);
        if (!result) {
            return Json.build(RespCode.BAD_REQUEST, "验证码错误");
        }
        log.debug("商户报名传入参数:[{}]", JSONObject.toJSONString(merchantParam));
        if (merchantParam.getPlatform() != 3 && merchantParam.getUserId() == null) {
            return Json.build(RespCode.BAD_REQUEST, "参数错误");
        }
        int insertResult = merchantService.saveMerchant(merchantParam);

        return Json.build("成功");
    }

    @ApiOperation(value = "2-06-8 获取优惠活动图片的文件流", notes = "")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "活动id", required = true, dataType = "long"),
            @ApiImplicitParam(name = "userId", value = "用户ID", required = false, dataType = "long"),
    })
    @GetMapping("/detail/img")
    public void getPrivilegeImg(@RequestParam("id") Long id, @RequestParam(name = "userId", required = false) Long userId, HttpServletResponse response) throws IOException {
        BufferedInputStream dis = null;
        BufferedOutputStream fos = null;
        PrivilegeDetailParam privilegeDetailParam = new PrivilegeDetailParam();
        privilegeDetailParam.setUserId(userId);
        privilegeDetailParam.setId(id);
        PrivilegeDTO privilegeDTO = this.privilegeService.getPrivilegeDetail(privilegeDetailParam);
        if (privilegeDTO == null) {
            return;
        }

        String urlString = privilegeDTO.getShareBgUrl();
        String fileName = urlString.substring(urlString.lastIndexOf('/') + 1);
        try {

            URL url = new URL(urlString);
            response.reset();//避免空行
            String contentType = FileTypeEnum.getContentType(urlString.substring(urlString.lastIndexOf('.') + 1));
            response.setContentType(contentType);
            response.setHeader("Content-disposition", "attachment; filename=" + new String(fileName.getBytes("utf-8"), "ISO8859-1"));
            response.setHeader("Content-Length", String.valueOf(url.openConnection().getContentLength()));

            dis = new BufferedInputStream(url.openStream());
            fos = new BufferedOutputStream(response.getOutputStream());

            byte[] buff = new byte[2048];
            int bytesRead;
            while (-1 != (bytesRead = dis.read(buff, 0, buff.length))) {
                fos.write(buff, 0, bytesRead);
            }

        } catch (Exception e) {
            log.error(e.getMessage(), e);
        } finally {
            if (dis != null) {
                dis.close();
            }
            if (fos != null) {
                fos.close();
            }
        }

    }

    @ApiOperation(value = "2-06-9 获得某地区内的所有优惠活动", notes = "当用户在开通VIP时,会进入开通VIP页面，其中会显示该地区的所有优惠活动以吸引用户")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "longitude", value = "经度", required = false),
            @ApiImplicitParam(name = "latitude", value = "纬度", required = false),
            @ApiImplicitParam(name = "areaCode", value = "地区编码", required = true)
    })
    @GetMapping("/all")
    public Json<List<PrivilegeOverviewDTO>> listAllPrivileges(@Validated PrivilegeOverviewParam privilegeParam) {
        return Json.build(privilegeService.listAllPrivilege(privilegeParam));
    }
}
