package com.bxm.localnews.activity.rank.impl;

import com.bxm.localnews.activity.common.config.ActivityProperties;
import com.bxm.localnews.activity.common.constant.RankEnum;
import com.bxm.localnews.activity.dto.*;
import com.bxm.localnews.activity.param.RankParam;
import com.bxm.localnews.activity.rank.RankSelfService;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.dto.InteractRankInfo;
import com.bxm.localnews.dto.UserInfoDTO;
import com.bxm.localnews.dto.UserInviteOrderVO;
import com.bxm.localnews.integration.ForumPostIntegrationService;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.vo.ForumPostRankForUserVo;
import com.bxm.localnews.vo.ForumPostTitleInfoVo;
import com.bxm.localnews.vo.ForumRankVo;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.fasterxml.jackson.core.type.TypeReference;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 组装榜单 （发帖榜，收徒榜，互动榜）
 *
 * @Author: pf.w
 * @Date: 2020/6/4 9:35
 */
@Service
@Log4j2
public class RankSelfServiceImpl implements RankSelfService {

    @Autowired
    UserIntegrationService userIntegrationService;

    @Autowired
    ForumPostIntegrationService forumPostIntegrationService;

    @Autowired
    RedisHashMapAdapter redisHashMapAdapter;

    @Autowired
    private ActivityProperties activityProperties;

    private static Integer RANK_TOP_MSG_LENGHT = 3;

    @Override
    public RankDetailDTO rankForumList(RankParam param) {
        RankDetailDTO rankDetailDTO = new RankDetailDTO();
        rankDetailDTO.setRankTitle(RankEnum.FORUM.getDesc());
        rankDetailDTO.setRankCode(RankEnum.FORUM.name());

        List<ForumRankVo> forumRankVoList = this.loadForum(param.getAreaCode());

        List<RankUserCompositeDTO> rankUsers = forumRankVoList.stream().map(this::convertToForum).collect(Collectors
                .toList());

        if (rankUsers.size() > 0) {
            //填充“关注”状态
            rankUsers.forEach(s -> s.getForumRankUser().setFollow(userIntegrationService.isFollow(param
                    .getUserId().longValue(), s.getForumRankUser().getRankUserId())));
        }
        rankDetailDTO.setRankUsers(rankUsers);
        return rankDetailDTO;
    }

    @Override
    public RankDetailDTO rankInteractList(RankParam param) {
        RankDetailDTO rankDetailDTO = new RankDetailDTO();
        rankDetailDTO.setRankTitle(RankEnum.INTERACT.getDesc());
        rankDetailDTO.setRankCode(RankEnum.INTERACT.name());

        List<InteractRankInfo> interactRankInfos = this.loadInteract(param.getAreaCode());

        List<RankUserCompositeDTO> rankUsers = interactRankInfos.stream().map(this::convertToInteract).collect(Collectors
                .toList());
        if (rankUsers.size() > 0) {
            //填充“关注”状态
            rankUsers.forEach(s -> s.getInteractRankUser().setFollow(userIntegrationService.isFollow(param
                    .getUserId().longValue(), s.getInteractRankUser().getRankUserId())));
        }

        rankDetailDTO.setRankUsers(rankUsers);
        return rankDetailDTO;
    }

    @Override
    public RankDetailDTO rankInviteList(RankParam param) {
        RankDetailDTO rankDetailDTO = new RankDetailDTO();
        rankDetailDTO.setRankTitle(RankEnum.INVITE.getDesc());
        rankDetailDTO.setRankCode(RankEnum.INVITE.name());

        List<UserInviteOrderVO> inviteRankInfos = this.loadInvite(param.getAreaCode());

        List<RankUserCompositeDTO> rankUsers = inviteRankInfos.stream().map(this::convertToInvite).collect
                (Collectors
                        .toList());

        if (rankUsers.size() > 0) {
            //填充“关注”状态
            rankUsers.forEach(s -> s.getInviteRankUser().setFollow(userIntegrationService.isFollow(param
                    .getUserId().longValue(), s.getInviteRankUser().getRankUserId())));
        }

        rankDetailDTO.setRankUsers(rankUsers);
        return rankDetailDTO;
    }

    @Override
    public RankBriefMocDto buildTopUserInfo(RankParam param) {
        String areaCode = param.getAreaCode();
        RankBriefMocDto result = new RankBriefMocDto();
        List<RankBriefDTO> rankBrief = new ArrayList<>();
        Map<String, Integer> rankForUser = new HashMap<>();
        //组装发帖榜
        if (activityProperties.getRankType().get(RankEnum.FORUM.name().toLowerCase())) {
            RankBriefDTO forumBrief = new RankBriefDTO();
            forumBrief.setRankTitle(RankEnum.FORUM.getDesc());
            forumBrief.setRankCode(RankEnum.FORUM.name());

            List<ForumRankVo> forumByRdList = this.loadForum(areaCode);

            for (int i = 0; i < forumByRdList.size(); i++) {
                if (forumByRdList.get(i).getUserId().toString().equals(param.getUserId().toString())) {
                    rankForUser.put(RankEnum.FORUM.name(), i + 1);
                    break;
                }
            }

            //截取前三位
            forumByRdList = forumByRdList.subList(0, forumByRdList.size() >= RANK_TOP_MSG_LENGHT
                    ? RANK_TOP_MSG_LENGHT
                    : forumByRdList.size());

            List<BaseRankTopUserDTO> topUser = forumByRdList.stream().map(item -> {
                BaseRankTopUserDTO baseRankTopUserDTO = new BaseRankTopUserDTO();
                baseRankTopUserDTO.setRankUserId(item.getUserId());
                baseRankTopUserDTO.setRankHeadImg(item.getUserImg());
                baseRankTopUserDTO.setRankNickName(item.getUserName());
                return baseRankTopUserDTO;
            }).collect(Collectors
                    .toList());

            forumBrief.setTopUsers(topUser);

            rankBrief.add(forumBrief);
        }
        //组装评论榜
        if (activityProperties.getRankType().get(RankEnum.INTERACT.name().toLowerCase())) {
            RankBriefDTO interactBrief = new RankBriefDTO();
            interactBrief.setRankTitle(RankEnum.INTERACT.getDesc());
            interactBrief.setRankCode(RankEnum.INTERACT.name());

            List<InteractRankInfo> interactByRdList = this.loadInteract(areaCode);

            for (int i = 0; i < interactByRdList.size(); i++) {
                if (interactByRdList.get(i).getId().toString().equals(param.getUserId().toString())) {
                    rankForUser.put(RankEnum.INTERACT.name(), i + 1);
                    break;
                }
            }

            //截取前三位
            interactByRdList = interactByRdList.subList(0, interactByRdList.size() >= RANK_TOP_MSG_LENGHT ?
                    RANK_TOP_MSG_LENGHT : interactByRdList
                    .size());

            List<BaseRankTopUserDTO> topUser = interactByRdList.stream().map(item -> {
                BaseRankTopUserDTO baseRankTopUserDTO = new BaseRankTopUserDTO();
                baseRankTopUserDTO.setRankUserId(item.getId());
                baseRankTopUserDTO.setRankHeadImg(item.getHeadImg());
                baseRankTopUserDTO.setRankNickName(item.getNickName());
                return baseRankTopUserDTO;
            }).collect(Collectors.toList());

            interactBrief.setTopUsers(topUser);

            rankBrief.add(interactBrief);
        }

        //组装收徒榜
        if (activityProperties.getRankType().get(RankEnum.INVITE.name().toLowerCase())) {
            RankBriefDTO inviteBrief = new RankBriefDTO();
            inviteBrief.setRankTitle(RankEnum.INVITE.getDesc());
            inviteBrief.setRankCode(RankEnum.INVITE.name());

            List<UserInviteOrderVO> inviteOrderVOS = this.loadInvite(areaCode);

            for (int i = 0; i < inviteOrderVOS.size(); i++) {
                if (inviteOrderVOS.get(i).getUserId().toString().equals(param.getUserId().toString())) {
                    rankForUser.put(RankEnum.INVITE.name(), i + 1);
                    break;
                }
            }

            inviteOrderVOS = inviteOrderVOS.subList(0, inviteOrderVOS.size() >= RANK_TOP_MSG_LENGHT
                    ? RANK_TOP_MSG_LENGHT
                    : inviteOrderVOS.size());

            List<BaseRankTopUserDTO> topUser = inviteOrderVOS.stream().map(item -> {
                BaseRankTopUserDTO baseRankTopUserDTO = new BaseRankTopUserDTO();
                baseRankTopUserDTO.setRankUserId(item.getUserId());
                baseRankTopUserDTO.setRankHeadImg(item.getHeadImg());
                baseRankTopUserDTO.setRankNickName(item.getNickname());
                return baseRankTopUserDTO;
            }).collect(Collectors.toList());

            inviteBrief.setTopUsers(topUser);

            rankBrief.add(inviteBrief);
        }

        result.setRankBrief(rankBrief);
        result.setRankForUser(rankForUser);
        return result;
    }


    private RankUserCompositeDTO convertToForum(ForumRankVo source) {
        ForumRankUserDTO forumRankUser = new ForumRankUserDTO();
        forumRankUser.setRankUserId(source.getUserId());
        forumRankUser.setRankNickName(source.getUserName());
        forumRankUser.setRankHeadImg(source.getUserImg());
        forumRankUser.setContentQualityScore(source.getContentQualityScore());
        forumRankUser.setLikeTotal(source.getLikeTotal());
        List<ForumPostTitleInfoVo> postTitleInfo = source.getPostTitleInfo();
        forumRankUser.setPostList(postTitleInfo.stream().map(e -> {
            ForumPostDTO forumPostDTO = new ForumPostDTO();
            forumPostDTO.setId(e.getId());
            forumPostDTO.setTitle(e.getTitle());
            return forumPostDTO;
        }).collect(Collectors.toList()));
        return new RankUserCompositeDTO(forumRankUser);
    }

    private RankUserCompositeDTO convertToInteract(InteractRankInfo source) {
        InteractRankUserDTO interactRankUser = new InteractRankUserDTO();
        interactRankUser.setRankUserId(source.getId());
        interactRankUser.setRankHeadImg(source.getHeadImg());
        interactRankUser.setRankNickName(source.getNickName());
        interactRankUser.setCommentTotal(source.getReplyNum());
        return new RankUserCompositeDTO(interactRankUser);
    }

    private RankUserCompositeDTO convertToInvite(UserInviteOrderVO source) {
        InviteRankUserDTO inviteRankUser = new InviteRankUserDTO();
        inviteRankUser.setRankUserId(source.getUserId());
        inviteRankUser.setRankNickName(source.getNickname());
        inviteRankUser.setRankHeadImg(source.getHeadImg());
        inviteRankUser.setInviteTotal(source.getInviteNum());
        return new RankUserCompositeDTO(inviteRankUser);
    }

    private List<ForumRankVo> loadForum(String areaCode) {
        TypeReference<List<ForumRankVo>> typeReference = new TypeReference<List<ForumRankVo>>() {
        };
        List<ForumRankVo> forumRankVoList = redisHashMapAdapter.get(buildKey(RankEnum.FORUM), areaCode, typeReference);
        if (!CollectionUtils.isEmpty(forumRankVoList)) {

            //黑名单
            forumRankVoList.removeIf(s -> activityProperties.getForumRankBlack().contains(s.getUserId()));

            return forumRankVoList;
        }
        return this.loadForumRankFromDb(areaCode);
    }

    @Override
    public List<ForumRankVo> loadForumRankFromDb(String areaCode) {
        List<ForumRankVo> forumRankVoListFromDb = forumPostIntegrationService.forumRank(areaCode,activityProperties
                .getForumRankBlack());
        if (forumRankVoListFromDb.size() > 0) {
            redisHashMapAdapter.put(buildKey(RankEnum.FORUM), areaCode, forumRankVoListFromDb);
            redisHashMapAdapter.expire(buildKey(RankEnum.FORUM), DateUtils.DAY_MILLISECOND / 1000);
        }
        return forumRankVoListFromDb;
    }


    private List<InteractRankInfo> loadInteract(String areaCode) {
        TypeReference<List<InteractRankInfo>> typeReference = new TypeReference<List<InteractRankInfo>>() {
        };
        List<InteractRankInfo> interactRankInfos = redisHashMapAdapter.get(buildKey(RankEnum.INTERACT), areaCode, typeReference);

        if (!CollectionUtils.isEmpty(interactRankInfos)) {

            //过滤互动榜黑名单
            interactRankInfos.removeIf(s -> activityProperties.getInteractRankBlack().contains(s.getId()));

            return interactRankInfos;
        }
        return this.loadInteractRankFromDb(areaCode);
    }

    @Override
    public List<InteractRankInfo> loadInteractRankFromDb(String areaCode) {
        List<InteractRankInfo> interactRankInfosFromDb = forumPostIntegrationService.getInteractRankInfo(areaCode,
                10, activityProperties.getInteractRankBlack());
        if (interactRankInfosFromDb.size() > 0) {
            redisHashMapAdapter.put(buildKey(RankEnum.INTERACT), areaCode, interactRankInfosFromDb);
            redisHashMapAdapter.expire(buildKey(RankEnum.INTERACT), DateUtils.DAY_MILLISECOND / 1000);
        }
        return interactRankInfosFromDb;
    }

    private List<UserInviteOrderVO> loadInvite(String areaCode) {
        TypeReference<List<UserInviteOrderVO>> typeReference = new TypeReference<List<UserInviteOrderVO>>() {
        };
        List<UserInviteOrderVO> inviteRankInfos = redisHashMapAdapter.get(buildKey(RankEnum.INVITE), areaCode,
                typeReference);

        if (!CollectionUtils.isEmpty(inviteRankInfos)) {

            //黑名单
            inviteRankInfos.removeIf(s -> activityProperties.getInviteRankBlack().contains(s.getUserId()));

            return inviteRankInfos;
        }

        return this.loadInviteRankFromDb(areaCode);
    }

    @Override
    public List<UserInviteOrderVO> loadInviteRankFromDb(String areaCode) {
        List<UserInviteOrderVO> userInviteOrderVOSFromDb = userIntegrationService.getOrderTen(10, areaCode,
                activityProperties.getInviteRankBlack());
        if (userInviteOrderVOSFromDb.size() > 0) {
            redisHashMapAdapter.put(buildKey(RankEnum.INVITE), areaCode, userInviteOrderVOSFromDb);
            redisHashMapAdapter.expire(buildKey(RankEnum.INVITE), DateUtils.DAY_MILLISECOND / 1000);
        }
        return userInviteOrderVOSFromDb;
    }

    @Override
    public CurrentUserRankDTO doCurrentUser(RankParam param) {

        CurrentUserRankDTO currentUserRankDTO = new CurrentUserRankDTO();

        UserInfoDTO userInfoDTO = userIntegrationService.getUserFromRedisDB(param.getUserId().longValue());

        currentUserRankDTO.setUserId(param.getUserId().longValue());
        currentUserRankDTO.setNickName(Objects.isNull(userInfoDTO) ? "" : userInfoDTO.getNickname());
        currentUserRankDTO.setHeadImg(Objects.isNull(userInfoDTO) ? "" : userInfoDTO.getHeadImg());
        currentUserRankDTO.setLikeNum(Objects.isNull(userInfoDTO) ? 0 : userInfoDTO.getLikeNum());


        ForumPostRankForUserVo forumPostRankForUserVo = forumPostIntegrationService.forumRankForUser(param.getUserId
                ().longValue());

        if (Objects.nonNull(forumPostRankForUserVo)) {
            currentUserRankDTO.setPostNum(forumPostRankForUserVo.getForumPostNum());
            currentUserRankDTO.setContentQualityScore(forumPostRankForUserVo.getContentQualityScore());
        } else {
            log.error("  cannot find forum info  for userId : {}", param.getUserId());
        }
        currentUserRankDTO.setCommentTotal(forumPostIntegrationService.replyNumForUser(param.getUserId().longValue()));
        currentUserRankDTO.setInviteTotal(userIntegrationService.getInviteNumByUserId(param.getUserId().longValue()).getInviteNum());
        return currentUserRankDTO;
    }

    @Override
    public Map<String, Integer> rankForUser(Long userId) {

        return null;
    }

    /**
     * 缓存key值
     *
     * @param rankEnum
     * @return
     */
    private KeyGenerator buildKey(RankEnum rankEnum) {
        return RedisConfig.RANK_LIST.copy().appendKey(rankEnum.name().toLowerCase());
    }

}
