package com.bxm.localnews.activity.service.impl;

import com.bxm.localnews.activity.common.config.HelpInfoProperties;
import com.bxm.localnews.activity.config.BaConfig;
import com.bxm.localnews.activity.domain.HelpInfoMapper;
import com.bxm.localnews.activity.param.HelpInfoQueryParam;
import com.bxm.localnews.activity.service.HelpInfoFacadeService;
import com.bxm.localnews.activity.service.HelpInfoService;
import com.bxm.localnews.activity.common.utils.UploadBase64Img;
import com.bxm.localnews.activity.vo.*;
import com.bxm.localnews.base.service.BaseUrlFacadeService;
import com.bxm.localnews.base.service.ShortLinkService;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.dto.UserInfoDTO;
import com.bxm.localnews.integration.ForumPostIntegrationService;
import com.bxm.localnews.integration.PushMsgIntegrationService;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.mq.common.constant.PushMessageEnum;
import com.bxm.localnews.mq.common.constant.TemplateTypeEnum;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.mq.common.model.dto.PushPayloadInfo;
import com.bxm.localnews.mq.common.model.dto.PushReceiveScope;
import com.bxm.localnews.vo.ForumBasicVo;
import com.bxm.localnews.vo.PostImgVo;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import com.bxm.newidea.component.vo.PageWarper;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.io.UnsupportedEncodingException;
import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;

import static org.apache.commons.lang3.StringUtils.join;

@Service
public class HelpInfoServiceImpl implements HelpInfoService, HelpInfoFacadeService {
    private static final Logger logger = LoggerFactory.getLogger(HelpInfoServiceImpl.class);

    @Resource
    private HelpInfoMapper helpInfoMapper;

    @Resource
    private HelpInfoProperties helpInfoProperties;

    @Resource
    private SequenceCreater sequenceCreater;

    @Resource
    private UserIntegrationService userIntegrationService;

    @Resource
    private ForumPostIntegrationService forumPostIntegrationService;

    @Resource
    private PushMsgIntegrationService pushMsgIntegrationService;

    @Resource
    private RedisStringAdapter redisStringAdapter;

    @Resource
    private ShortLinkService shortLinkService;

    @Resource
    private DistributedLock distributedLock;

    @Resource
    private UploadBase64Img uploadBase64Img;

    @Resource
    private BaConfig baConfig;

    @Resource
    private BaseUrlFacadeService baseUrlFacadeService;

    @Override
    public Boolean isActiveArea(String areaCode) {
        if (StringUtils.isEmpty(helpInfoProperties.getActiveArea())) {
            return false;
        }

        //全国
        if ("0".equals(helpInfoProperties.getActiveArea())) {
            return true;
        }

        return helpInfoProperties.getActiveArea().contains(areaCode);
    }

    @Override
    public HelpIndexPageInfo getIndexPageInfo(Integer size, Long userId, Integer pageType) {
        HelpIndexPageInfo helpIndexPageInfo = new HelpIndexPageInfo();

        //获取用户助力信息
        HelpRankInfo helpRankInfo = getUserRank(userId);
        helpIndexPageInfo.setUserHelpInfo(helpRankInfo);

        //获取轮播信息
        helpIndexPageInfo.setRoundList(getRoundList(pageType, size, userId, helpRankInfo));

        //获取进度信息
        Integer helpedNum = getHelpedNum();
        helpIndexPageInfo.setHelpedNum(helpedNum);
        helpIndexPageInfo.setHelpRate(helpInfoProperties.getHelpRate());
        helpIndexPageInfo.setHelpTotalAmount(addComma(helpInfoProperties.getHelpRate()
                .multiply(new BigDecimal(helpInfoProperties.getHelpTotalLimit())).intValue()));
        helpIndexPageInfo.setHelpedAmount(addComma(helpInfoProperties.getHelpRate().multiply(new BigDecimal(helpedNum)).intValue()));
        Integer helpPercent = helpedNum * 100 / helpInfoProperties.getHelpTotalLimit();
        if (helpPercent < 1) {
            helpPercent = 1;
        }
        helpIndexPageInfo.setHelpPercent(helpPercent.toString());
        if (helpInfoProperties.getHelpPoint() <= helpedNum) {
            helpIndexPageInfo.setHelpPoint(helpInfoProperties.getHelpRate()
                    .multiply(new BigDecimal(helpInfoProperties.getHelpPoint())).intValue());
        }

        helpIndexPageInfo.setHelpedUserImgList(getHelpedUserImgList(userId, helpRankInfo));

        //获取榜单信息
        HelpInfoQueryParam param = new HelpInfoQueryParam();
        param.setPageNum(1);
        param.setPageSize(3);
        param.setType((byte) 1);

        helpIndexPageInfo.setPersonalRankList(getRankingList(param).getList());
        param.setType((byte) 2);
        helpIndexPageInfo.setTeamRankList(getRankingList(param).getList());

        //获取活动状态
        if (0 == helpInfoProperties.getState()) {
            helpIndexPageInfo.setActivityStatus(1);
        } else if (isActivityValid()) {
            helpIndexPageInfo.setActivityStatus(0);
        } else {
            helpIndexPageInfo.setActivityStatus(2);
        }

        return helpIndexPageInfo;
    }

    @Override
    public PageWarper<HelpRankInfo> getRankingList(HelpInfoQueryParam param) {
        PageWarper<HelpRankInfo> resultPage = new PageWarper<>(helpInfoMapper.getRankingList(param));

        if (CollectionUtils.isNotEmpty(resultPage.getList())) {
            int index = (param.getPageNum() - 1) * param.getPageSize() + 1;
            for (HelpRankInfo helpRankInfo : resultPage.getList()) {
                helpRankInfo.setRank(index);
                index++;

                //获取邀请人id
                if ((byte) 2 == helpRankInfo.getType()) {
                    helpRankInfo.setInviteUserId(helpInfoMapper.getTeamCreatorById(helpRankInfo.getHelperId()));
                } else {
                    helpRankInfo.setInviteUserId(helpRankInfo.getHelperId());
                }
            }
        }

        return resultPage;
    }

    @Override
    public HelpRankInfo getUserRank(Long userId) {
        HelpRankInfo helpRankInfo = getUserRankInfo(userId);
        if (helpRankInfo != null && (null == helpRankInfo.getInviteNum() || 0 == helpRankInfo.getInviteNum())) {
            helpRankInfo.setRank(999);
        }

        //团队助力获取creator
        if (null != helpRankInfo && null != helpRankInfo.getType() && (byte) 2 == helpRankInfo.getType()) {
            Long creator = helpInfoMapper.getTeamCreatorById(helpRankInfo.getHelperId());
            //团队创建人不是当前用户，则不返回创建人
            if (userId.equals(creator)) {
                helpRankInfo.setCreator(creator);
            }
        }

        return helpRankInfo;
    }

    @Override
    @Transactional
    public Message help(String areaCode, String areaName, Long userId, Byte type, String teamName, Long inviteUserId) {
        if ((byte) 1 != type && (byte) 2 != type && (byte) 3 != type) {
            return Message.build(false).setMessage("参数错误");
        }

        if (!isActivityValid()) {
            return Message.build(false).setMessage("活动结束，点击了解更多");
        }

        UserInfoDTO userInfoDTO = userIntegrationService.getUserFromRedisDB(userId);
        if (null == userInfoDTO || StringUtils.isEmpty(userInfoDTO.getNickname())) {
            return Message.build(false).setMessage("该用户不存在");
        }

        Long inviteHelperId = null;
        Byte inviteHelperType = (byte) 0;
        if ((byte) 3 == type) {
            //邀请助力邀请用户校验
            HelpBaseInfo inviteUserHelpInfo = getUserHelpBaseInfo(inviteUserId);
            if (null == inviteUserHelpInfo || null == inviteUserHelpInfo.getHelperId()) {
                return Message.build(false).setMessage("邀请用户无助力信息");
            }

            inviteHelperId = inviteUserHelpInfo.getHelperId();
            inviteHelperType = inviteUserHelpInfo.getType();
        }

        if (helpInfoMapper.countUserHelpInfo(userId) > 0) {
            return Message.build(false).setMessage("您已助力过该活动");
        }

        if ((byte) 2 == type && helpInfoMapper.countHelpTeamByName(teamName) > 0) {
            return Message.build(false).setMessage("已存在该团队名");
        }

        //控制并发
        if (distributedLock.lock(userId.toString(), sequenceCreater.nextStringId(), 3, TimeUnit.SECONDS)
                && ((byte) 2 != type || (byte) 2 == type && distributedLock.lock(teamName, sequenceCreater.nextStringId(), 3, TimeUnit.SECONDS))) {
            Long helpedNum = redisStringAdapter.increment(RedisConfig.BLUE_SKY_HELPED_NUM.copy());
            if (helpedNum > helpInfoProperties.getHelpTotalLimit()) {
                redisStringAdapter.decrement(RedisConfig.BLUE_SKY_HELPED_NUM.copy());
                return Message.build(false).setMessage("广德蓝天救援队助力目标达成，感谢参与");
            } else if (helpedNum.equals(helpInfoProperties.getHelpTotalLimit().longValue())) {
                redisStringAdapter.set(RedisConfig.BLUE_SKY_HELP_COMPLETE.copy(), true);
            } else {
                redisStringAdapter.remove(RedisConfig.BLUE_SKY_HELP_COMPLETE.copy());
            }

            doHelp(areaCode, areaName, type, teamName, userInfoDTO, inviteUserId, inviteHelperId, inviteHelperType);
        } else {
            return Message.build(false).setMessage("系统繁忙，请稍后再试");
        }

        return Message.build(true);
    }

    @Override
    public Message updateTeamImg(Long userId, String teamImg) {
        int update = helpInfoMapper.updateTeamImgByCreator(userId, teamImg);

        HelpBaseInfo helpBaseInfo = getUserHelpBaseInfo(userId);
        if (null != helpBaseInfo && (byte) 2 == helpBaseInfo.getType()) {
            helpInfoMapper.updateHelpTeamImg(helpBaseInfo.getHelperId(), teamImg);
        }

        if (update > 0) {
            return Message.build(true);
        } else {
            return Message.build(false).setMessage("该用户非团队发起人");
        }
    }

    @Override
    public HelpCertInfo getUserCert(Long userId) {
        HelpCertInfo helpCertInfo = new HelpCertInfo();
        Date createTime = helpInfoMapper.getUserHelpTime(userId);
        if (null == createTime) {
            return null;
        }

        Integer countUserRank = helpInfoMapper.countUserRank(createTime, userId);
        if (null == countUserRank) {
            return null;
        }
        helpCertInfo.setRank(countUserRank + 1);

        UserInfoDTO userInfoDTO = userIntegrationService.getUserFromRedisDB(userId);
        if (userInfoDTO != null) {
            helpCertInfo.setHelperId(userId);
            helpCertInfo.setHelperName(userInfoDTO.getNickname());
            helpCertInfo.setHelperImg(userInfoDTO.getHeadImg());
        }

        helpCertInfo.setRankRate((99 - helpCertInfo.getRank() * 100 / 450000) + "");
        helpCertInfo.setJumpAddress(shortLinkService.generateShortUrl(join(
                baseUrlFacadeService.getContentViewSceneBaseUrl(),
                baConfig.getJumpAddress(), "?shareUserId=", userId, "&type=", helpCertInfo.getType(), "&source=poster")));
        helpCertInfo.setCreatePost(!redisStringAdapter.hasKey(RedisConfig.BLUE_SKY_CREATE_POST.copy().appendKey(userId)));
        helpCertInfo.setCertDate(DateUtils.formatAtWill(new Date(), DateUtils.DATE_FORMAT_YEAR_MONTH_DAY_FORMAT));

        return helpCertInfo;
    }

    @Override
    public Message shareCreatePost(Long userId, String areaCode, String base64Str) {
        Date createTime = helpInfoMapper.getUserHelpTime(userId);
        if (null == createTime) {
            return Message.build(false).setMessage("用户未助力");
        }

        KeyGenerator key = RedisConfig.BLUE_SKY_CREATE_POST.copy().appendKey(userId);
        if (redisStringAdapter.hasKey(key)) {
            return Message.build(false).setMessage("已创建过帖子");
        }
        redisStringAdapter.set(key, true);

        Integer countUserRank = helpInfoMapper.countUserRank(createTime, userId);
        UserInfoDTO userInfoDTO = userIntegrationService.getUserFromRedisDB(userId);
        if (countUserRank != null && userInfoDTO != null) {
            //调用新闻模块创建帖子
            ForumBasicVo forumBasicVo = new ForumBasicVo();
            forumBasicVo.setTextField("我是" + userInfoDTO.getNickname() + "，我是第" + (countUserRank + 1) +
                    "位为广德蓝天救援队助力的广德人！让我们一起致敬生命的救援者！");
            forumBasicVo.setForumId(helpInfoProperties.getForumId());
            forumBasicVo.setAreaCode(areaCode);
            forumBasicVo.setUserId(userId);
            String postAddress = join(baseUrlFacadeService.getContentViewSceneBaseUrl(), baConfig.getPostAddress(),
                    "?userId=", userId,
                    "&areaName=广德&areaCode=341822000000&hasShareBtn=1&shareDialogWay=1&source=poster");
            try {
                postAddress = java.net.URLEncoder.encode(postAddress, "UTF-8");
            } catch (UnsupportedEncodingException e) {
                logger.error("url encode 失败，url：" + postAddress);
            }
            forumBasicVo.setExtraContent("<a href=\"wst://web/webDetail?url=" + postAddress + "\">点击马上参与助力</a>");

            if (StringUtils.isNotBlank(base64Str)) {
                String url = uploadBase64Img.uploadBase64Img(base64Str);
                if (StringUtils.isNotBlank(url)) {
                    List<PostImgVo> imgList = new ArrayList<>();
                    PostImgVo postImgVo = new PostImgVo();
                    postImgVo.setImgUrl(url);
                    postImgVo.setType("IMG");
                    imgList.add(postImgVo);
                    forumBasicVo.setPostImgList(imgList);
                }
            }

            forumPostIntegrationService.createPost(forumBasicVo);
        }

        return Message.build(true);
    }

    @Override
    public HelpBaseInfo getUserHelpBaseInfo(Long userId) {
        if (null == userId) {
            return null;
        }

        HelpBaseInfo helpBaseInfo = helpInfoMapper.getPersonalInfoByUserId(userId);
        if (null == helpBaseInfo) {
            helpBaseInfo = helpInfoMapper.getTeamInfoByUserId(userId);
        }
        return helpBaseInfo;
    }

    @Override
    public String getHomeWindowUrl(Long userId, String areaCode, String areaName) {
        String homeWindowUrl = join(baseUrlFacadeService.getContentViewSceneBaseUrl(), baConfig.getHomeWindowUrl());
        if (StringUtils.isBlank(homeWindowUrl)) {
            return null;
        }

        //每天只弹一次
        KeyGenerator key = RedisConfig.BLUE_SKY_HOME_WINDOW.copy().appendKey(DateUtils.formatDate(new Date())).appendKey(userId);
        if (redisStringAdapter.hasKey(key)) {
            return null;
        }

        HelpWindowInfo helpWindowInfo = getBaseHelpWindowInfo(areaCode, userId);
        if (null == helpWindowInfo) {
            return null;
        }

        redisStringAdapter.set(key, true, DateUtils.getCurSeconds());
        return homeWindowUrl + "?source=dialog&userId=" + userId + "&areaCode=" + areaCode +
                "&areaName=" + areaName + "&inviteUserId=" + helpWindowInfo.getInviteUserId() +
                "&userName=" + helpWindowInfo.getUserName() + "&userImg=" + helpWindowInfo.getUserImg();

    }

    @Override
    public HelpWindowInfo getHelpWindowInfo(String areaCode, Long userId) {
        //每天只弹一次
        KeyGenerator key = RedisConfig.BLUE_SKY_H5_HOME_WINDOW.copy().appendKey(DateUtils.formatDate(new Date())).appendKey(userId);
        if (redisStringAdapter.hasKey(key)) {
            return null;
        }

        HelpWindowInfo helpWindowInfo = getBaseHelpWindowInfo(areaCode, userId);
        if (null == helpWindowInfo) {
            return null;
        }

        redisStringAdapter.set(key, true, DateUtils.getCurSeconds());
        return helpWindowInfo;
    }

    @Override
    public Message createHelpUserInvite(Long userId, Long inviteUserId) {
        if (userId.equals(inviteUserId)) {
            return Message.build(false).setMessage("邀请人与被邀请人id相同");
        }

        int insertNum = helpInfoMapper.createHelpUserInvite(userId, inviteUserId);

        if (insertNum > 0) {
            return Message.build(true);
        } else {
            return Message.build(false).setMessage("创建失败");
        }
    }

    private List<HelpBaseInfo> getRoundList(Integer pageType, Integer size, Long userId, HelpRankInfo helpRankInfo) {
        if (2 == pageType && null != helpRankInfo) {
            if (1 == helpRankInfo.getType()) {
                return helpInfoMapper.getRoundList(size, userId);
            } else {
                List<Long> userIdList = helpInfoMapper.getRoundUserIdList(size, helpRankInfo.getHelperId());
                List<HelpBaseInfo> roundList = new ArrayList<>();
                if (CollectionUtils.isNotEmpty(userIdList)) {
                    for (Long id : userIdList) {
                        UserInfoDTO userInfoDTO = userIntegrationService.getUserFromRedisDB(id);
                        if (null == userInfoDTO) {
                            continue;
                        }
                        HelpBaseInfo helpBaseInfo = new HelpBaseInfo();
                        helpBaseInfo.setHelperId(id);
                        helpBaseInfo.setHelperName(userInfoDTO.getNickname());
                        helpBaseInfo.setHelperImg(userInfoDTO.getHeadImg());
                        roundList.add(helpBaseInfo);
                    }
                }
                return roundList;
            }
        } else {
            return helpInfoMapper.getRoundList(size, null);
        }
    }

    private List<String> getHelpedUserImgList(Long userId, HelpRankInfo helpRankInfo) {
        List<String> userImgList = new ArrayList<>();
        UserInfoDTO userInfoDTO = userIntegrationService.getUserFromRedisDB(userId);
        List<Long> helpedUserIds;
        if (null != helpRankInfo && null != userInfoDTO) {
            userImgList.add(userInfoDTO.getHeadImg());
            helpedUserIds = helpInfoMapper.getHelpedUserIds(userId, 4);
        } else {
            helpedUserIds = helpInfoMapper.getHelpedUserIds(userId, 5);
        }
        if (null != helpedUserIds) {
            for (Long id : helpedUserIds) {
                UserInfoDTO userInfo = userIntegrationService.getUserFromRedisDB(id);
                if (null == userInfo) {
                    continue;
                }
                userImgList.add(userInfo.getHeadImg());
            }
        }

        return userImgList;
    }

    private HelpRankInfo getUserRankInfo(Long userId) {
        HelpBaseInfo helpBaseInfo = getUserHelpBaseInfo(userId);

        if (null == helpBaseInfo) {
            return null;
        }

        return helpInfoMapper.getRankInfoByHelperIdAndType(helpBaseInfo.getHelperId(), helpBaseInfo.getType());
    }

    private void doHelp(String areaCode, String areaName, Byte type, String teamName, UserInfoDTO userInfoDTO,
                        Long inviteUserId, Long inviteHelperId, Byte inviteHelperType) {
        HelpBaseInfo helpBaseInfo = new HelpBaseInfo();
        helpBaseInfo.setId(sequenceCreater.nextLongId());
        helpBaseInfo.setType(type);

        if ((byte) 2 == type) {
            //团队助力
            helpBaseInfo.setHelperName(teamName);
            helpBaseInfo.setCreator(userInfoDTO.getId());
            helpBaseInfo.setCreatorName(userInfoDTO.getNickname());
            helpBaseInfo.setCreatorImg(userInfoDTO.getHeadImg());
            helpBaseInfo.setHelperId(sequenceCreater.nextLongId());
            helpBaseInfo.setHelperImg(helpInfoProperties.getDefaultTeamImg());

            helpInfoMapper.insertTeamInfo(helpBaseInfo);
            helpInfoMapper.insertTeamUserRelation(helpBaseInfo.getHelperId(), helpBaseInfo.getCreator());
            helpInfoMapper.insertHelpInfo(helpBaseInfo);
        } else if ((byte) 1 == type) {
            //个人助力
            helpBaseInfo.setHelperId(userInfoDTO.getId());
            helpBaseInfo.setHelperName(userInfoDTO.getNickname());
            helpBaseInfo.setHelperImg(userInfoDTO.getHeadImg());
            helpInfoMapper.insertHelpInfo(helpBaseInfo);
        } else if ((byte) 3 == type) {
            if ((byte) 1 == inviteHelperType) {
                //创建助力信息
                helpBaseInfo.setHelperId(userInfoDTO.getId());
                helpBaseInfo.setHelperName(userInfoDTO.getNickname());
                helpBaseInfo.setHelperImg(userInfoDTO.getHeadImg());
                helpBaseInfo.setType(inviteHelperType);
                helpBaseInfo.setInviteUserId(inviteUserId);
                helpInfoMapper.insertHelpInfo(helpBaseInfo);
            } else if ((byte) 2 == inviteHelperType) {
                helpInfoMapper.insertTeamUserRelation(inviteHelperId, userInfoDTO.getId());
            }

            if (null != inviteHelperId) {
                helpInfoMapper.increaseInviteNum(inviteHelperId, inviteHelperType);
            }

            sendMessage(areaCode, areaName, userInfoDTO, inviteUserId);
        }
    }

    //发送助力成功消息
    private void sendMessage(String areaCode, String areaName, UserInfoDTO userInfoDTO, Long sendUserId) {
        //邀请助力成功推送
        try {
            String title = userInfoDTO.getNickname() + "用户已经帮你助力成功，奉献了1份爱心，快来看看吧";
            PushPayloadInfo info = PushPayloadInfo.build(PushMessageEnum.OPEN_URL);
            info.addExtend("url", join(
                    baseUrlFacadeService.getContentViewSceneBaseUrl(),
                    baConfig.getSuccessAddress(), "?userId=", sendUserId
                    ,"&areaCode=", areaCode, "&areaName=", areaName));

            PushMessage message = PushMessage.build();
            message.setTitle("通知消息");
            message.setContent(title);
            message.setType(TemplateTypeEnum.NOTIFCTION);
            message.setPushReceiveScope(PushReceiveScope.pushSignle(sendUserId));
            message.setPayloadInfo(info);

            this.pushMsgIntegrationService.pushMsg(message);
        } catch (Exception e) {
            logger.error("发送消息失败：", e);
        }
    }

    private HelpWindowInfo getBaseHelpWindowInfo(String areaCode, Long userId) {
        HelpWindowInfo helpWindowInfo = new HelpWindowInfo();
        //活动未开启或非活动区域或活动过期
        if (0 == helpInfoProperties.getState() || !isActiveArea(areaCode) || !isActivityValid()) {
            return null;
        }
        //活动助力已满
        Integer helpedNum = getHelpedNum();
        if (helpedNum >= helpInfoProperties.getHelpTotalLimit()) {
            return null;
        }
        //无用户信息
        UserInfoDTO user = userIntegrationService.getUserFromRedisDB(userId);
        if (null == user) {
            return null;
        }
        //有助力信息
        HelpBaseInfo userHelpInfo = getUserHelpBaseInfo(userId);
        if (null != userHelpInfo) {
            return null;
        }
        //用户邀请人
        Long inviteUserId = helpInfoMapper.getInviteUserIdByUserId(userId);
        HelpBaseInfo inviteHelpInfo = getUserHelpBaseInfo(inviteUserId);
        if (null == inviteHelpInfo) {
            return null;
        }

        helpWindowInfo.setInviteUserId(inviteUserId);
        helpWindowInfo.setUserName(inviteHelpInfo.getHelperName());
        helpWindowInfo.setUserImg(inviteHelpInfo.getHelperImg());

        return helpWindowInfo;
    }

    //判断活动是否有效
    private boolean isActivityValid() {
        Date startTime = DateUtils.parse(helpInfoProperties.getStartTime());
        Date endTime = DateUtils.parse(helpInfoProperties.getEndTime());
        Date now = new Date();
        return now.after(startTime) && now.before(endTime);
    }

    //获取已助力人数
    private Integer getHelpedNum() {
        Integer helpedNum = redisStringAdapter.get(RedisConfig.BLUE_SKY_HELPED_NUM.copy(), Integer.class);

        if (null == helpedNum) {
            helpedNum = helpInfoMapper.getHelpedNum();
            if (null == helpedNum) {
                helpedNum = 0;
            }
            redisStringAdapter.set(RedisConfig.BLUE_SKY_HELPED_NUM.copy(), helpedNum);
        }

        return helpedNum;
    }

    private String addComma(Integer i) {
        DecimalFormat df = new DecimalFormat("#,###");
        return df.format(i);
    }

}
