package com.bxm.localnews.activity.service.impl;

import com.bxm.localnews.activity.common.constant.CashEnum;
import com.bxm.localnews.activity.domain.LotteryDrawMapper;
import com.bxm.localnews.activity.param.CashDetailParam;
import com.bxm.localnews.activity.service.LotteryDrawService;
import com.bxm.localnews.activity.vo.WechatUserInfoVo;
import com.bxm.localnews.base.service.BizLogService;
import com.bxm.localnews.dto.UserInfoDTO;
import com.bxm.localnews.integration.InviteRecordIntegrationService;
import com.bxm.localnews.integration.UserAccountIntegrationService;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.param.AccountCashParam;
import com.bxm.localnews.thirdparty.service.PushMessageService;
import com.bxm.localnews.vo.InviteRecord;
import com.bxm.newidea.component.vo.PageWarper;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

@Service
public class LotteryDrawServiceImpl implements LotteryDrawService {

    private LotteryDrawMapper lotteryDrawMapper;

    private UserIntegrationService userIntegrationService;

    private UserAccountIntegrationService userAccountIntegrationService;

    private InviteRecordIntegrationService inviteRecordIntegrationService;

    private PushMessageService pushMessageService;

    private BizLogService bizLogService;

    @Autowired
    public LotteryDrawServiceImpl(LotteryDrawMapper lotteryDrawMapper,
                                  UserIntegrationService userIntegrationService,
                                  UserAccountIntegrationService userAccountIntegrationService,
                                  InviteRecordIntegrationService inviteRecordIntegrationService,
                                  PushMessageService pushMessageService,
                                  BizLogService bizLogService) {
        this.lotteryDrawMapper = lotteryDrawMapper;
        this.userIntegrationService = userIntegrationService;
        this.userAccountIntegrationService = userAccountIntegrationService;
        this.inviteRecordIntegrationService = inviteRecordIntegrationService;
        this.pushMessageService = pushMessageService;
        this.bizLogService = bizLogService;
    }

    @Override
    public WechatUserInfoVo getUserInfo(Long userId, String city, String district) {
        WechatUserInfoVo user = new WechatUserInfoVo();

        if (StringUtils.isBlank(city) && StringUtils.isBlank(district)) {
            user.setActivityArea(true);
        } else {
            String areaCode = null;
            if (StringUtils.isNotBlank(district)) {
                areaCode = lotteryDrawMapper.getCodeByName(district);
                user.setShareArea(district.substring(0, district.length() - 1));
            }
            if (StringUtils.isBlank(areaCode) && StringUtils.isNotBlank(city)) {
                areaCode = lotteryDrawMapper.getCodeByName(city);
                user.setShareArea(city.substring(0, city.length() - 1));
            }
            user.setAreaCode(areaCode);

            if (StringUtils.isNotBlank(user.getAreaCode())) {
                user.setActivityArea(true);

                if (userId != null) {
                    bizLogService.conformDrawAreaUser(userId, user.getAreaCode());
                }
            } else {
                user.setActivityArea(false);
            }

            getWechatUserInfo(userId, user);
        }

        return user;
    }

    @Override
    public List<BigDecimal> getPrizeList() {
        List<BigDecimal> prizeList = new ArrayList<>();
        List<Map<String, Object>> prizeMapList = lotteryDrawMapper.getPrizeList();

        if (CollectionUtils.isNotEmpty(prizeMapList)) {
            Map<String, Object> prizeMap = prizeMapList.get(0);
            for (int i = 1; i < 9; i++) {
                BigDecimal prize = new BigDecimal(prizeMap.get("prize_" + i).toString());
                prizeList.add(prize);
            }
        }

        return prizeList;
    }

    @Override
    public List<WechatUserInfoVo> getWinUserList() {
        return lotteryDrawMapper.getWinUserList();
    }

    @Override
    public String lotteryDraw(Long userId, Integer platform, String areaCode) {
        List<Map<String, Object>> prizeMapList = this.lotteryDrawMapper.getPrizeList();
        if (CollectionUtils.isEmpty(prizeMapList) || lotteryDrawMapper.updateDrawTimes(userId) <= 0) {
            return "0";
        }

        //根据是否临时用户返回不同的赏金及类型
        Boolean isTempUser = userIntegrationService.isTempUser(userId);
        Integer hasDrawTimes = lotteryDrawMapper.getHasDrawTimes(userId);
        String winPrize = getWinPrize(prizeMapList, hasDrawTimes, isTempUser);

        //更新邀请关系金额，及用户账户金额
        if (!"0".equals(winPrize)) {
            BigDecimal winCash = new BigDecimal(winPrize);
            winPrize = winCash.stripTrailingZeros().toPlainString();

            AccountCashParam param = new AccountCashParam();
            param.setAddTotal(true);
            param.setCashType(CashEnum.TEMP_CASH.name());
            param.setCash(winCash);

            //增加邀请人金额
            addInviteUserCash(userId, winCash, param);

            //判断用户自己中奖金额状态
            if (Boolean.FALSE.equals(isTempUser)) {
                param.setCashType(CashEnum.DRAWABLEL_CASH.name());
            }
            param.setInviteType(null);
            param.setInvitedUserName(null);
            param.setCashFlowType("SELF_BOUNTY");
            param.setUserId(userId);
            userAccountIntegrationService.addCash(param);
            lotteryDrawMapper.disableUserRedPacket(userId);

            bizLogService.lotteryDraw(userId, hasDrawTimes, winCash, platform, areaCode);
        }

        return winPrize;
    }

    @Override
    public Boolean share(Long userId) {
        if (null == userId) {
            return false;
        }

        List<Map<String, Object>> prizeMapList = lotteryDrawMapper.getPrizeList();
        if (CollectionUtils.isNotEmpty(prizeMapList)) {
            Integer totalTimes = Integer.valueOf(prizeMapList.get(0).get("total_times").toString());
            return lotteryDrawMapper.addRemainTimes(userId, totalTimes) > 0;
        }

        return false;
    }

    @Override
    public PageWarper<WechatUserInfoVo> getDetailPage(CashDetailParam param) {
        List<WechatUserInfoVo> resultList = lotteryDrawMapper.getDetailList(param);

        //补充本人信息
        if (CollectionUtils.isNotEmpty(resultList)) {
            UserInfoDTO user = userIntegrationService.getUserFromRedisDB(param.getUserId());
            for (WechatUserInfoVo userInfo : resultList) {
                if (1 == userInfo.getType()) {
                    if (null != user) {
                        userInfo.setNickname(user.getNickname());
                        userInfo.setHeadImg(user.getHeadImg());
                    }
                } else {
                    InviteRecord inviteRecord = inviteRecordIntegrationService.getInviteRecordById(
                            param.getUserId(), userInfo.getRelationId());
                    if (null != inviteRecord) {
                        Long invitedUserId = inviteRecord.getInvitedUserId();
                        UserInfoDTO invitedUser = userIntegrationService.getUserFromRedisDB(invitedUserId);
                        if (null != invitedUser) {
                            userInfo.setNickname(invitedUser.getNickname());
                            userInfo.setHeadImg(invitedUser.getHeadImg());
                        }
                    }
                }
            }
        }

        return new PageWarper<>(resultList);
    }

    @Override
    public Boolean isActiveArea(String city, String district) {
        if (StringUtils.isBlank(city) && StringUtils.isBlank(district)) {
            return true;
        } else {
            String areaCode = lotteryDrawMapper.getCodeByName(district);
            if (StringUtils.isBlank(areaCode)) {
                areaCode = lotteryDrawMapper.getCodeByName(city);
            }
            return StringUtils.isNotBlank(areaCode);
        }
    }

    private void getWechatUserInfo(Long userId, WechatUserInfoVo user) {
        if (userId != null) {
            UserInfoDTO userInfoDTO = userIntegrationService.getUserFromRedisDB(userId);
            if (userInfoDTO != null) {
                user.setHeadImg(userInfoDTO.getHeadImg());
                user.setNickname(userInfoDTO.getNickname());
            }

            user.setTempUser(userIntegrationService.isTempUser(userId));
            user.setWalletCash(lotteryDrawMapper.getWalletCash(userId));
            Integer remainTimes = lotteryDrawMapper.getRemainTimes(userId);
            user.setRemainTimes(remainTimes);

            List<Map<String, Object>> prizeMapList = this.lotteryDrawMapper.getPrizeList();
            if (CollectionUtils.isNotEmpty(prizeMapList)) {
                Integer totalTimes = Integer.valueOf(prizeMapList.get(0).get("total_times").toString());

                Integer hasDrawTimes = lotteryDrawMapper.getHasDrawTimes(userId);
                user.setHasDrawTimes(hasDrawTimes);
                if (remainTimes != 0 || hasDrawTimes >= totalTimes) {
                    user.setCanAddTimes(false);
                } else {
                    user.setCanAddTimes(true);
                }
            }
        }
    }

    private String getWinPrize(List<Map<String, Object>> prizeMapList, Integer hasDrawTimes, Boolean isTempUser) {
        String winPrize = "0";
        if (Boolean.TRUE.equals(isTempUser)) {
            Object prize = prizeMapList.get(0).get("win_prize_" + hasDrawTimes);
            if (null != prize) {
                winPrize = prize.toString();
            }
        } else {
            Map<String, Object> prizeMap = prizeMapList.get(0);
            BigDecimal minPrize = new BigDecimal(prizeMap.get("prize_1").toString());
            for (int i = 2; i < 9; i++) {
                BigDecimal prize = new BigDecimal(prizeMap.get("prize_" + i).toString());
                if (minPrize.compareTo(prize) > 0) {
                    minPrize = prize;
                }
            }

            winPrize = minPrize.stripTrailingZeros().toPlainString();
        }
        return winPrize;
    }

    private void addInviteUserCash(Long userId, BigDecimal winCash, AccountCashParam param) {
        Long inviteUserId = lotteryDrawMapper.getInviteUserByUserId(userId);
        if (inviteUserId != null) {
            Long id = inviteRecordIntegrationService.updateInviteCash(inviteUserId, userId, winCash);
            param.setRelationId(id);
            param.setUserId(inviteUserId);
            param.setCashFlowType("APPLET_BOUNTY");

            InviteRecord inviteRecord = inviteRecordIntegrationService.getInviteRecordByInvitedUser(inviteUserId, userId);
            if (null != inviteRecord) {
                param.setInviteType(inviteRecord.getType());
                param.setInvitedUserName(inviteRecord.getInvitedUserName());
                //判断邀请金额状态
                if ("LOGIN_APP".equals(inviteRecord.getInviteState())) {
                    param.setCashType(CashEnum.STABILIZE_CASH.name());
                } else if ("SUCCESS_WITHDRAW".equals(inviteRecord.getInviteState())) {
                    param.setCashType(CashEnum.DRAWABLEL_CASH.name());
                }

                userAccountIntegrationService.addCash(param);
                pushMessageService.pushMessage(winCash, inviteUserId);
            }
        }
    }
}
