package com.bxm.localnews.activity.service.impl;

import com.alibaba.fastjson.JSONArray;
import com.bxm.localnews.activity.common.constant.MerchantSortTypeEnum;
import com.bxm.localnews.activity.domain.MerchantMapper;
import com.bxm.localnews.activity.dto.MerchantDetailDTO;
import com.bxm.localnews.activity.dto.MerchantListDTO;
import com.bxm.localnews.activity.param.MerchantDetailBasicParam;
import com.bxm.localnews.activity.param.MerchantDetailParam;
import com.bxm.localnews.activity.param.MerchantListParam;
import com.bxm.localnews.activity.param.MerchantParam;
import com.bxm.localnews.activity.service.MerchantService;
import com.bxm.localnews.activity.vo.MerchantVO;
import com.bxm.localnews.activity.vo.UserForMerchantInfo;
import com.bxm.localnews.base.service.LocationFacadeService;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.common.dto.LocationDetailDTO;
import com.bxm.localnews.dto.UserInfoDTO;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.newidea.component.geo.dto.Coordinate;
import com.bxm.newidea.component.geo.service.GeoService;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.StringUtils;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.stereotype.Service;
import com.bxm.localnews.common.vo.BasicParam;
import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * Created by Arei on 2018/10/25.
 */
@Service("merchantService")
@Log4j2
public class MerchantServiceImpl implements MerchantService {

    @Resource
    private MerchantMapper merchantMapper;

    @Resource
    private RedisStringAdapter redisStringAdapter;

    @Resource
    private GeoService geoService;

    @Autowired
    private UserIntegrationService userIntegrationService;

    @Autowired
    LocationFacadeService locationFacadeService;

    @Override
    public int saveMerchant(MerchantParam merchantParam) {
        MerchantVO merchantVO = new MerchantVO();
        BeanUtils.copyProperties(merchantParam, merchantVO);
        merchantVO.setStatus(0);
        merchantVO.setDeleteFlag(0);
        if (merchantVO.getUserId() == null) {
            merchantVO.setUserId(0L);
        }
        return merchantMapper.saveMerchant(merchantVO);
    }

    @Override
    public List<MerchantListDTO> getMerchantList(MerchantListParam param) {

        log.info("MerchantListParam : {}",param);

        List<MerchantListDTO> listInfo = getInfoList(param);

        return sorted(listInfo,param);
    }

    private List<MerchantListDTO> sorted( List<MerchantListDTO> listInfo,MerchantListParam param){

        //检索- 未过期的置顶商家
        List<MerchantListDTO> listForSort = listInfo.stream().filter(p -> p.getSort() != null && (DateUtils.before(p
                .getSortStartTime()) && DateUtils.after(p.getSortEndTime())))
                .collect
                (Collectors
                .toList
                ());

        //检索-非置顶商家 和 已过期置顶商家
        List<MerchantListDTO> listOther = listInfo.stream().filter(p -> p.getSort() == null || DateUtils.before(p
                .getSortEndTime() )|| DateUtils.after(p.getSortStartTime())).collect
                (Collectors
                .toList());

        //置顶的商家  置顶相同时 按距离排序
        Collections.sort(listForSort, new Comparator<MerchantListDTO>() {
            @Override
            public int compare(MerchantListDTO o1, MerchantListDTO o2) {
               if(o1.getSort().equals( o2.getSort()) && o1.getDistanceNum() != null && o2.getDistanceNum()!= null){
                   if(o1.getDistanceNum() < o2.getDistanceNum()){
                       return -1;
                   }else{
                       return 1;
                   }
               }else{
                   if(o1.getSort()<o2.getSort()){
                       return -1;
                   }else{
                       return 1;
                   }
               }
            }
        });

        //对（非置顶商家 和 已过期置顶商家） 进行逻辑排序
        if(param.getSortType().equals(MerchantSortTypeEnum.RECOMMEND_SORT.getCode())){
            //权重排序   下一版本
        }else if(param.getSortType().equals(MerchantSortTypeEnum.DISTANCE_SORT.getCode())){
            //按距离排序   距离相等时按照折扣力度排序
            Collections.sort(listOther, new Comparator<MerchantListDTO>() {
                @Override
                public int compare(MerchantListDTO o1, MerchantListDTO o2) {
                    if(o1.getDistanceNum() != null && o2.getDistanceNum()!= null && !o1.getDistanceNum()
                            .equals(o2.getDistanceNum())){
                        if(o1.getDistanceNum() < o2.getDistanceNum()){
                            return -1;
                        }else{
                            return 1;
                        }
                    }else{
                        if(o1.getDiscount() != null  && o2.getDiscount() != null
                                 && o1.getDiscount().compareTo(o2.getDiscount()) > 0){
                            return -1;
                        }else{
                            return 1;
                        }
                    }
                }
            });
        }
        //重新组装list
        listForSort.addAll(listOther);
        return listForSort;
    }


    private List<MerchantListDTO> getInfoList(MerchantListParam param){

        //优先从缓存获取
        List<MerchantListDTO> list = new ArrayList<>();
        String listInfoStr = redisStringAdapter.getString(getListKey(param));
        if(StringUtils.isNotBlank(listInfoStr)){
            list = JSONArray.parseArray(listInfoStr,MerchantListDTO.class);
        }else{
            list = getFromDb(param);
            if(list.size()>0) {
                this.redisStringAdapter.set(this.getListKey(param), JSONArray.toJSONString(list));
            }
        }
        //填充距离字段
        distanceCalculation(param,list);
        return list;
    }

    /**
     * 填充“距离”数据
     * @param param
     */
    private void distanceCalculation(MerchantListParam param,List<MerchantListDTO> list){
        if(param.getLatitude()!=null && param.getLongitude() != null){
            Coordinate userLngAndLat = new Coordinate();
            userLngAndLat.setLongitude(param.getLongitude());
            userLngAndLat.setLatitude(param.getLatitude());

            for (MerchantListDTO merchantListDTO : list) {
                if(merchantListDTO.getGeoLat() != null && merchantListDTO.getGeoLng()!=null){
                    Long distance = fillInstance(userLngAndLat,merchantListDTO.getGeoLat(),merchantListDTO.getGeoLng());
                    merchantListDTO.setDistance((Math.round(distance / 100d) / 10d) < 1 ? "<1km" : String.valueOf(Math
                            .round
                            (distance / 100d) / 10d)+"km");
                    merchantListDTO.setDistanceNum(distance);
                }
            }
        }
    }

    private List<MerchantListDTO> getFromDb(MerchantListParam param){
        LocationDetailDTO locationDetailDTO = null;
        if(StringUtils.isNotBlank(param.getAreaCode())){
            BasicParam basicParam = new BasicParam();
            BeanUtils.copyProperties(param,basicParam);
            locationDetailDTO = locationFacadeService.getInternalLocationByGeocode(param.getAreaCode(),
                    basicParam);
        }
        List<MerchantVO> list = merchantMapper.selectForList(Objects.nonNull(locationDetailDTO) ? locationDetailDTO
                .getCode() : null);
        return  list.stream().map(this::convert).collect(Collectors.toList());
    }

    private MerchantListDTO convert(MerchantVO merchantVO){
        MerchantListDTO merchantListDTO = MerchantListDTO.builder()
                .merchantId(merchantVO.getId().intValue())
                .merchantName(merchantVO.getName())
                .facadeUrl(StringUtils.isNotBlank(merchantVO.getFacadeUrl()) ? Arrays.asList(merchantVO.getFacadeUrl().split
                        (",")) : new ArrayList<>())
                .merchantTypeDesc(StringUtils.isNotBlank(merchantVO.getType()) ? merchantVO.getType() : "")
                .build();
        BeanUtils.copyProperties(merchantVO,merchantListDTO);
        return merchantListDTO;
    }

    private KeyGenerator getListKey(MerchantListParam param) {

        return RedisConfig.MERCHANT_LIST.copy().appendKey(StringUtils.isNotBlank(param.getAreaCode()) ? param
                .getAreaCode() : "all");
    }

    @Override
    public MerchantDetailDTO getMerchantDetail(MerchantDetailParam param) {
        MerchantVO merchantVO = merchantMapper.selectByPrimaryKey(param.getMerchantId());
        MerchantDetailDTO merchantDetailDTO = convertToDetail(merchantVO);

        distanceCalculationDetail(param,merchantDetailDTO);
        return merchantDetailDTO;
    }

    private MerchantDetailDTO convertToDetail(MerchantVO merchantVO){
        MerchantDetailDTO detail = new MerchantDetailDTO();
        detail.setFacadeUrl(StringUtils.isNotBlank(merchantVO.getFacadeUrl()) ? Arrays.asList(merchantVO.getFacadeUrl()
                .split(",")) : new ArrayList<>());
        detail.setImgUrls(StringUtils.isNotBlank(merchantVO.getImgUrls()) ? Arrays.asList(merchantVO.getImgUrls().split
                (",")) : new ArrayList<>());
        detail.setMerchantName(merchantVO.getName());
        detail.setMerchantTypeDesc(StringUtils.isNotBlank(merchantVO.getType()) ? merchantVO.getType() : "");
        BeanUtils.copyProperties(merchantVO,detail);

        //填充user信息
       if(merchantVO.getRelationUserId()!=null){
           UserInfoDTO userInfoDTO =  userIntegrationService.getUserFromRedisDB((long)merchantVO.getRelationUserId());
           if(Objects.nonNull(userInfoDTO)){
               UserForMerchantInfo userInfo = UserForMerchantInfo.builder()
                       .userId(merchantVO.getRelationUserId())
                       .headimgurl(StringUtils.isNotBlank(userInfoDTO.getHeadImg()) ? userInfoDTO.getHeadImg() : "")
                       .nickName(StringUtils.isNotBlank(userInfoDTO.getNickname()) ? userInfoDTO.getNickname() : "")
                       .userJobDesc(StringUtils.isNotBlank(userInfoDTO.getPersonalProfile()) ? userInfoDTO
                               .getPersonalProfile().startsWith("一句话介绍下自己") ? "暂时没有介绍自己" : userInfoDTO.getPersonalProfile() : "暂时没有介绍自己")
                       .build();
               detail.setUserForMerchantInfo(userInfo);
           }
        }
        return detail;
    }

    @Override
    public MerchantDetailDTO getDetailForShareOutSite(MerchantDetailBasicParam param) {
        MerchantVO merchantVO = merchantMapper.selectByPrimaryKey(param.getMerchantId());
        MerchantDetailDTO detail = convertToDetail(merchantVO);
        return detail;
    }

    private void distanceCalculationDetail(MerchantDetailParam param,MerchantDetailDTO merchantDetailDTO){
        if(param.getLatitude()!=null && param.getLongitude() != null){
            Coordinate userLngAndLat = new Coordinate();
            userLngAndLat.setLongitude(param.getLongitude());
            userLngAndLat.setLatitude(param.getLatitude());
            if(merchantDetailDTO.getGeoLat() != null && merchantDetailDTO.getGeoLng()!=null){
                Long distance = fillInstance(userLngAndLat,merchantDetailDTO.getGeoLat(),merchantDetailDTO
                        .getGeoLng());
                merchantDetailDTO.setDistance((Math.round(distance / 100d) / 10d) < 1 ? "<1km" : String.valueOf(Math
                        .round
                        (distance / 100d) / 10d)+"km");
                }
            }
    }

    /**
     * 根据经纬度计算距离
     * @param userLngAndLat
     * @param geoLat
     * @param geoLng
     * @return
     */
    private Long fillInstance(Coordinate userLngAndLat,Double geoLat,Double geoLng){
        Coordinate lngAndLat = new Coordinate();
        lngAndLat.setLongitude(geoLng);
        lngAndLat.setLatitude(geoLat);
         return geoService.getDistance(userLngAndLat, lngAndLat);
    }
}
