package com.bxm.localnews.activity.service.impl;

import com.bxm.localnews.activity.common.constant.CommodityCategoryEnum;
import com.bxm.localnews.activity.common.constant.VipStatusEnum;
import com.bxm.localnews.activity.domain.CommodityCodeMapper;
import com.bxm.localnews.activity.domain.CommodityMapper;
import com.bxm.localnews.activity.domain.MerchantMapper;
import com.bxm.localnews.activity.domain.WinnerMapper;
import com.bxm.localnews.activity.dto.MovieDTO;
import com.bxm.localnews.activity.dto.NoticeDTO;
import com.bxm.localnews.activity.param.MovieParam;
import com.bxm.localnews.activity.param.PrivilegeParam;
import com.bxm.localnews.activity.service.MovieService;
import com.bxm.localnews.activity.service.PrivilegeService;
import com.bxm.localnews.activity.service.VipService;
import com.bxm.localnews.activity.vo.*;
import com.bxm.localnews.base.service.LocationFacadeService;
import com.bxm.localnews.common.dto.LocationDetailDTO;
import com.bxm.newidea.component.geo.dto.Coordinate;
import com.bxm.newidea.component.geo.service.GeoService;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.tools.StringUtils;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Optional;

@Service
public class MovieServiceImpl extends BaseService implements MovieService {

    @Autowired
    private CommodityMapper commodityMapper;

    @Autowired
    private WinnerMapper winnerMapper;

    @Autowired
    private MerchantMapper merchantMapper;

    @Autowired
    private CommodityCodeMapper commodityCodeMapper;

    @Autowired
    private DistributedLock distributedLock;

    @Autowired
    private GeoService geoService;

    @Autowired
    private PrivilegeService privilegeService;

    @Autowired
    private LocationFacadeService locationFacadeService;

    private VipService vipService;

    @Override
    public boolean isHasMovie(Long userId) {
        Integer count = commodityMapper.countVipMovie(userId);
        return count > 0;
    }

    private VipService getVipService() {
        if (null == vipService) {
            vipService = SpringContextHolder.getBean(VipService.class);
        }
        return vipService;
    }

    @Override
    public MovieDTO getMovieDetail(MovieParam movieParam) {
        //查询电影票信息
        Vip vip = getVipService().getVipByUser(movieParam.getUserId());
        if (null != vip && VipStatusEnum.ACTIVATED.getType().equals(vip.getStatus())) {
            List<WinnerVO> winnerList = winnerMapper.getVipMovie(movieParam.getUserId());
            if (CollectionUtils.isNotEmpty(winnerList)) {
                WinnerVO winnerVO = winnerList.get(0);
                //查询商家信息
                MerchantVO merchantVO = merchantMapper.selectByPrimaryKey(winnerVO.getMerchantId());
                if (null != merchantVO) {
                    MovieDTO movieDTO = new MovieDTO();
                    movieDTO.setTitle(merchantVO.getName());
                    movieDTO.setBusinessArea(merchantVO.getBusinessArea());
                    movieDTO.setBusinessEndTime(merchantVO.getBusinessEndTime());
                    movieDTO.setBusinessStartTime(merchantVO.getBusinessStartTime());
                    movieDTO.setDisplayAddress(merchantVO.getDisplayAddress());
                    movieDTO.setGeoLat(merchantVO.getGeoLat());
                    movieDTO.setGeoLng(merchantVO.getGeoLng());
                    movieDTO.setHeadImg(merchantVO.getHeadImg());
                    movieDTO.setMerchantName(merchantVO.getName());
                    movieDTO.setServicePhone(merchantVO.getServicePhone());
                    movieDTO.setCode(winnerVO.getCommodityCode());
                    movieDTO.setExpireTime(winnerVO.getRedeemTime());
                    movieDTO.setDistance(getDistance(movieParam, merchantVO));
                    if (StringUtils.isNotEmpty(vip.getActivationAreaCode())) {
                        movieDTO.setActivationAreaCode(vip.getActivationAreaCode());
                    }
                    return movieDTO;
                }
            }
        }
        return null;
    }

    /**
     * 得到距离
     * @param movieParam
     * @param merchantVO
     * @return
     */
    private String getDistance(MovieParam movieParam, MerchantVO merchantVO) {
        PrivilegeParam privilegeParam = new PrivilegeParam();
        BeanUtils.copyProperties(movieParam, privilegeParam);
        privilegeParam.setUserId(movieParam.getUserId());

        Coordinate userLngAndLat = privilegeService.getUserCoordinate(privilegeParam);

        if (null != userLngAndLat) {
            Coordinate merchantLngAndLat = new Coordinate();
            if (merchantVO.getGeoLng() != null && merchantVO.getGeoLat() != null) {
                merchantLngAndLat.setLongitude(merchantVO.getGeoLng());
                merchantLngAndLat.setLatitude(merchantVO.getGeoLat());
                Long distance = geoService.getDistance(userLngAndLat, merchantLngAndLat);
                String distancePreStr = "距您";
                return distancePreStr + Math.round(distance / 100d) / 10d + "km";
            }
        }
        return "";
    }

    @Override
    public NoticeDTO getMovieVipNotice(Long userId) {
        List<WinnerVO> winnerList = winnerMapper.getVipMovie(userId);
        if (CollectionUtils.isNotEmpty(winnerList)) {
            WinnerVO winnerVO = winnerList.get(0);
            if (winnerVO.getPopup() == 1) {
                NoticeDTO noticeDTO = new NoticeDTO();
                noticeDTO.setName(CommodityCategoryEnum.MOVIE.getDesc());
                noticeDTO.setPopup(true);
                return noticeDTO;

            }
        }
        return null;
    }

    @Override
    public void updateMovieVipNotice(Long userId) {
        List<WinnerVO> winnerList = winnerMapper.getVipMovie(userId);
        if (CollectionUtils.isNotEmpty(winnerList)) {
            WinnerVO winnerVO = winnerList.get(0);
            winnerMapper.updateWinnerPopup(winnerVO.getId());
        }
    }

    @Override
    public void doGeneratorMovie(Long userId, String areaCode) {
        int count = 0;
        retry(userId, count, areaCode);
    }

    private void retry(Long userId, Integer count, String areaCode) {
        if (count >= 3) {
            logger.error("获取锁达三次以上无法进行操作");
            return;
        }
        List<CommodityVO> commodityList = commodityMapper.selectByCategory(CommodityCategoryEnum.MOVIE.getType());

        if (CollectionUtils.isNotEmpty(commodityList)) {
            List<WinnerVO> winnerList = winnerMapper.getVipMovie(userId);
            if (CollectionUtils.isNotEmpty(winnerList)) {
                return;
            }
            String requestId = String.valueOf(nextSequence());

            CommodityVO commodityVO = getCommodity(commodityList, areaCode);
            if (null == commodityVO) {
                return;
            }
            //得到电影券
            List<CommodityCode> commodityCodeList = commodityCodeMapper.getUnusedCommodities(commodityVO.getId(), 1);
            if (CollectionUtils.isEmpty(commodityCodeList)) {
                return;
            }
            String code = commodityCodeList.get(0).getCode();
            //上锁
            if (distributedLock.lock(code, requestId)) {
                //更新商品编码表中的当前编码为<已使用>
                commodityCodeMapper.updateToUsed(code, commodityVO.getId());

                //插入奖品用户关系表
                WinnerVO winnerVO = new WinnerVO();
                winnerVO.setAddTime(new Date());
                winnerVO.setCommodityCode(code);
                winnerVO.setId(nextSequence());
                winnerVO.setUserId(userId);

                Calendar calendar = Calendar.getInstance();
                calendar.setTime(new Date());
                calendar.add(Calendar.MONTH, 2);
                winnerVO.setRedeemTime(calendar.getTime());
                winnerMapper.insertSelective(winnerVO);

                //商品库存减1
                //commodityMapper.updateCommodityNum(commodityVO.getId());

                //解锁
                distributedLock.unlock(code, requestId);
            } else {
                try {
                    Thread.sleep(100);
                } catch (InterruptedException e) {
                    e.printStackTrace();
                }
                // 休息 100ms
                retry(userId, count++, areaCode);
            }

        }

    }


    private CommodityVO getCommodity(List<CommodityVO> commodityList, String areaCode) {
        Optional<CommodityVO> commodityVO = commodityList.stream().filter(x -> isCurrentArea(x, areaCode)).findFirst();
        return commodityVO.orElse(null);
    }

    private boolean isCurrentArea(CommodityVO commodityVO, String areaCode) {
        MerchantVO merchantVO = merchantMapper.selectByPrimaryKey(commodityVO.getMerchantId());
        String cityCode = merchantVO.getCityCode();
        String countyCode = merchantVO.getCountyCode();

        LocationDetailDTO locationDTO = locationFacadeService.getLocationDetailByCode(countyCode);
        if (null != locationDTO) {
            return areaCode.equals(countyCode);
        }
        LocationDetailDTO locationByCityCode = locationFacadeService.getLocationDetailByCode(cityCode);
        if (null != locationByCityCode) {
            return areaCode.equals(cityCode);
        }
        return false;
    }
}
