package com.bxm.localnews.activity.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.activity.common.config.SignProperties;
import com.bxm.localnews.activity.domain.SignConfigMapper;
import com.bxm.localnews.activity.domain.SignRecordMapper;
import com.bxm.localnews.activity.dto.SignLeaderBoard;
import com.bxm.localnews.activity.service.SignFacadeService;
import com.bxm.localnews.activity.vo.SignConfig;
import com.bxm.localnews.activity.vo.SignRecord;
import com.bxm.localnews.base.service.LocationFacadeService;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.common.dto.LocationDetailDTO;
import com.bxm.localnews.common.vo.Location;
import com.bxm.localnews.dto.UserInfoDTO;
import com.bxm.localnews.integration.PushMsgIntegrationService;
import com.bxm.localnews.integration.UserAccountIntegrationService;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.redis.RedisZSetAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.StringUtils;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.redis.core.DefaultTypedTuple;
import org.springframework.data.redis.core.ZSetOperations;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

import static com.bxm.localnews.common.constant.RedisConfig.SIGN_CACHE_KEY;
import static com.bxm.newidea.component.tools.DateUtils.addField;

@Service
public class SignServiceImpl extends BaseService implements SignFacadeService {

    private static final Logger logger = LoggerFactory.getLogger(SignServiceImpl.class);

    /**
     * 用户签到数据过期时间，防止非活跃用户长期占据内存
     */
    private static final int EXPIRE_TIME = 3600 * 24 * 2;

    /**
     * 签到列表长度
     */
    private static final int SIGN_LIST_LENGTH = 8;

    @Resource
    private SignConfigMapper signConfigMapper;

    @Resource
    private SignRecordMapper signRecordMapper;

    @Resource
    private UserAccountIntegrationService userAccountIntegrationService;

    @Resource
    private RedisStringAdapter redisStringAdapter;

    @Resource
    private RedisZSetAdapter redisZSetAdapter;

    @Resource
    private RedisSetAdapter redisSetAdapter;

    @Resource
    private UserIntegrationService userIntegrationService;

    @Resource
    private PushMsgIntegrationService pushMsgIntegrationService;

    @Resource
    private LocationFacadeService locationFacadeService;

    @Resource
    private SignProperties signProperties;

//    @Override
//    public com.bxm.localnews.activity.dto.SignWarper listSignRecord(Long userId, int platform) {
//        com.bxm.localnews.activity.dto.SignWarper warper = new com.bxm.localnews.activity.dto.SignWarper();
//        //获取签到记录
//        List<SignRecord> signRecords = this.listUserSignRecord(userId);
//
//        User user = userService.selectByPrimaryKey(userId);
//        if (user != null && 1 == user.getState()) {
//            Date today = new Date();
//
//            for (SignRecord record : signRecords) {
//                if (DateUtils.isSameDay(record.getSignDate(), today) && record.getSignFlag()) {
//                    warper.setTodaySignState(true);
//                    break;
//                }
//            }
//
//            String requestId = String.valueOf(nextSequence());
//            if (!warper.isTodaySignState()) {
//                if (distributedLock.lock(String.valueOf(userId), requestId)) {
//                    logger.warn("签到逻辑-今日未签到，自动进行签到,user:[{}]", userId);
//                    signRecords = this.getLastSignRecordsByDoDailySign(userId, signRecords);
//                    //用于标记是否弹窗签到成功页面
//                    warper.setAutoSign(false);
//                    warper.setTodaySignState(true);
//                    //解锁
//                    distributedLock.unlock(String.valueOf(userId), requestId);
//                }else{
//                    logger.warn("签到逻辑-获取不到锁，进行短暂休眠，重新走一遍签到逻辑,user:[{}]",userId);
//                    try {
//                        Thread.sleep(200);
//                    } catch (InterruptedException e) {
//                        logger.error("分布式锁休眠时间:200,错误:{}",e.getMessage());
//                    }
//                    return this.listSignRecord(userId,platform);
//                }
//            } else {
//                warper.setAutoSign(true);
//            }
//            //获取最新一天的签到金币数
//            Long signReward = signRecords.stream()
//                    .filter(SignRecord::getSignFlag)
//                    .sorted(Comparator.comparing(SignRecord::getSignDate).reversed())
//                    .findFirst().get().getSignReward();
//
//            warper.setSignReward(signReward);
//
//            warper.setTotalGold(userAmountIntegrationService.selectGoldBalanceByUserId(userId));
//        }
//        warper.setRecords(signRecords);
//        warper.setCount(this.getSignCount(userId));
//        return warper;
//    }

    @Override
    public com.bxm.localnews.activity.dto.SignWarper listSignRecord(Long userId, int platform) {
        com.bxm.localnews.activity.dto.SignWarper warper = new com.bxm.localnews.activity.dto.SignWarper();
        //获取签到记录
        Date today = new Date();
        List<SignRecord> signRecords = this.buildSignList(0, today, userId, true);

        warper.setAutoSign(true);
        warper.setTodaySignState(true);
        warper.setSignReward(0L);
        warper.setTotalGold(BigDecimal.valueOf(userAccountIntegrationService.getUserUsableGold(userId)));
        warper.setRecords(signRecords);
        warper.setCount(this.getSignCount(userId));
        return warper;
    }

    /**
     * 获取用户签到次数的key
     *
     * @param userId
     * @return
     */
    private KeyGenerator getSignCountKey(Long userId) {
        return RedisConfig.SIGN_COUNT_KEY.copy().appendKey(userId.toString());
    }

    /**
     * 获取用户签到列表的key
     *
     * @param userId
     * @return
     */
    private KeyGenerator getSignListKey(Long userId) {
        return RedisConfig.SIGN_LIST_KEY.copy().appendKey(userId.toString());
    }

    /**
     * 获取用户签到次数
     *
     * @param userId
     * @return
     */
    private int getSignCount(Long userId) {
        Integer count = this.redisStringAdapter.getInt(this.getSignCountKey(userId));
        return count == null ? 1 : count;
    }

    /**
     * 构建签到列表
     *
     * @param startIndex 开始签到的天数
     * @param startDate  开始签到的日期
     * @param newList    是否为开始的新列表(新列表的已签到标签全部为false)
     * @return 签到列表
     */
    private List<SignRecord> buildSignList(int startIndex, Date startDate, Long userId, boolean newList) {
        List<SignRecord> records = Lists.newArrayList();

        List<SignConfig> signConfigs = this.getSignConfigInfo();

        int startNum = startIndex > SIGN_LIST_LENGTH ? ((startIndex - 1) / SIGN_LIST_LENGTH) * SIGN_LIST_LENGTH : 0;

        int actualStartIndex = startIndex % SIGN_LIST_LENGTH;
        if (startIndex > 0 && actualStartIndex == 0) {
            actualStartIndex = SIGN_LIST_LENGTH;
        } else if (startIndex == 0) {
            actualStartIndex = 1;
        }

        SignRecord record;
        for (int i = 0; i < SIGN_LIST_LENGTH; i++) {
            record = new SignRecord();
            int signDay = i + 1 + startNum;
            record.setSignDay(signDay);
            if (startIndex == 0) {
                record.setSignDate(addField(startDate, Calendar.DATE, i));
            } else {
                record.setSignDate(addField(startDate, Calendar.DATE, i + 1 - actualStartIndex));
            }
            if (newList) {
                record.setSignFlag(false);
            } else {
                record.setSignFlag(startIndex >= signDay);
            }

            if (signDay >= SIGN_LIST_LENGTH) {
                record.setSignReward(signConfigs.get(SIGN_LIST_LENGTH - 1).getReward());
            } else {
                record.setSignReward(signConfigs.get(i).getReward());
            }

            records.add(record);
        }

        this.redisStringAdapter.set(this.getSignListKey(userId), records, EXPIRE_TIME);

        return records;
    }

    /**
     * 获取签到奖励配置信息
     *
     * @return
     */
    @Override
    public List<SignConfig> getSignConfigInfo() {
        TypeReference<List<SignConfig>> typeReference = new TypeReference<List<SignConfig>>() {
        };
        List<SignConfig> redisRet = this.redisStringAdapter.get(SIGN_CACHE_KEY, typeReference);
        if (CollectionUtils.isNotEmpty(redisRet)) {
            return redisRet;
        }

        List<SignConfig> list = this.signConfigMapper.listAll();
        this.redisStringAdapter.set(SIGN_CACHE_KEY, list, 3600 * 24 * 7);
        return list;
    }

    @Override
    public void syncSignRanking() {
        //获得全表签到信息
        String tableName = "b_sign_record_";
        List<SignRecord> signRecordList = new ArrayList<>();
        for (int i = 0; i < 10; i++) {
            signRecordList.addAll(signRecordMapper.listSignRecordByUser(tableName + i));
        }
        //根据签到地区分组
        Map<String, List<SignRecord>> signRecordMap = signRecordList.stream()
                .filter(e -> StringUtils.isNotEmpty(e.getAreaCode()))
                .collect(Collectors.groupingBy(SignRecord::getAreaCode));
        //按地区遍历之后，将其存入redis
        signRecordMap.forEach((k, v) -> {
            v = v.stream().sorted(Comparator.comparing(SignRecord::getSignDay).reversed()).collect(Collectors.toList());

            logger.info("扫描各地区排行榜信息->[{}]该地区签到榜信息:[{}]", k, JSONObject.toJSONString(v));

            LocationDetailDTO locationOptional = locationFacadeService.getLocationDetailByCode(k);
            if (null != locationOptional) {

                //判断地区是否已经开通了签到排行榜
                if (!checkEnableSignBoard(locationOptional)) {
                    //如果没有开通签到排行榜，则判断当前地区是否满足可以将其开通（当前地区排行榜第20名连续签到天数不少于5天）
                    if (checkIsSatisfiedEnableSignBoard(v)) {
                        Location location = new Location();
                        location.setEnableSignBoard(1);
                        location.setCode(locationOptional.getCode());
                        locationFacadeService.update(location);
                    } else {
                        return;
                    }
                }

                //重置地区签到排行榜
                KeyGenerator signRecordKey = RedisConfig.SIGN_RANKING_KEY.copy().appendKey(k);
                Set<ZSetOperations.TypedTuple> tuples = new HashSet<>();
                v.forEach(x -> {
                    SignLeaderBoard.LeaderBoard leaderBoard = new SignLeaderBoard.LeaderBoard();
                    UserInfoDTO userInfoDTO = userIntegrationService.getUserFromRedisDB(x.getUserId());
                    if (null != userInfoDTO) {
                        leaderBoard.setNickname(userInfoDTO.getNickname() == null ? "" : userInfoDTO.getNickname())
                                .setHeadImg(StringUtils.isEmpty(userInfoDTO.getHeadImg()) ? userIntegrationService.getDefaultHeadImgUrl() : userInfoDTO.getHeadImg())
                                .setUserId(x.getUserId())
                                .setCount(x.getSignDay());
                        ZSetOperations.TypedTuple<SignLeaderBoard.LeaderBoard> tuple = new DefaultTypedTuple<>(leaderBoard, (double) leaderBoard.getCount());
                        tuples.add(tuple);
                    }
                });
                redisZSetAdapter.remove(signRecordKey);
                redisZSetAdapter.add(signRecordKey, tuples);
            }
        });
    }

    private boolean checkEnableSignBoard(LocationDetailDTO locationOptional) {
        return null != locationOptional.getEnableSignBoard() && 1 == locationOptional.getEnableSignBoard();
    }

    /**
     * 判断是否达到了开通签到排行榜的要求
     *
     * @param signRecordList
     * @return
     */
    private boolean checkIsSatisfiedEnableSignBoard(List<SignRecord> signRecordList) {
        return signRecordList.size() >= signProperties.getOpenDemandSize()
                && signRecordList.get(signProperties.getOpenDemandSize() - 1).getSignDay() >= signProperties.getOpenMinRequirement();
    }

    @Override
    public void noticeUserSign() {
        Date date = new Date();
        String today = DateUtils.formatDate(date);
        String yesterday = DateUtils.formatDate(DateUtils.addField(date, Calendar.DATE, -1));

        KeyGenerator signRecordKey = RedisConfig.SIGN_NOTIFICATION_KEY.copy().appendKey(today);
        KeyGenerator yesterdaySignRecordKey = RedisConfig.SIGN_NOTIFICATION_KEY.copy().appendKey(yesterday);
        //得到在昨天签到表中，但是不在今天签到表中的用户
        Set<Long> userIdSet = redisSetAdapter.difference(yesterdaySignRecordKey, Long.class, signRecordKey);

        pushMsgIntegrationService.pushSignNotificationMsg(userIdSet);
    }

}
