package com.bxm.localnews.activity.service.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.activity.common.config.VipRankProperties;
import com.bxm.localnews.activity.common.config.WechatCustomerServiceProperties;
import com.bxm.localnews.activity.common.constant.VipStatusEnum;
import com.bxm.localnews.activity.domain.VipMapper;
import com.bxm.localnews.activity.domain.VipViewRecordMapper;
import com.bxm.localnews.activity.dto.*;
import com.bxm.localnews.activity.param.RecordWechatParam;
import com.bxm.localnews.activity.service.MovieService;
import com.bxm.localnews.activity.service.VipFacadeService;
import com.bxm.localnews.activity.service.VipService;
import com.bxm.localnews.activity.vo.Vip;
import com.bxm.localnews.activity.vo.VipViewRecord;
import com.bxm.localnews.base.service.BizLogService;
import com.bxm.localnews.base.service.LocationFacadeService;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.common.dto.LocationDetailDTO;
import com.bxm.localnews.dto.UserInfoDTO;
import com.bxm.localnews.integration.PushMsgIntegrationService;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.vo.User;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author zhaoyadong 2018/12/24 18:02
 * @desc
 */
@Service
public class VipServiceImpl extends BaseService implements VipService, VipFacadeService {

    @Resource
    private WechatCustomerServiceProperties wechatCustomerServiceProperties;

    @Resource
    private VipRankProperties vipRankProperties;

    @Resource
    private SequenceCreater sequenceCreater;

    @Resource
    private VipMapper vipMapper;

    @Resource
    private VipViewRecordMapper vipViewRecordMapper;

    @Resource
    private RedisSetAdapter redisSetAdapter;

    @Resource
    private RedisStringAdapter redisStringAdapter;

    @Resource
    private LocationFacadeService locationFacadeService;

    @Resource
    private UserIntegrationService userIntegrationService;

    @Resource
    private PushMsgIntegrationService pushMsgIntegrationService;

    @Resource
    private MovieService movieService;

    @Resource
    private BizLogService bizLogService;

    @Resource
    private DistributedLock distributedLock;

    @Override
    public Message saveVipViewRecord(Long userId, String unionId, Integer platform) {
        //确定用户与好友的绑定关系
        confirmUserRelation(unionId, userId, platform);
        //更新本人的vip信息
        updateUserVip(userId, unionId);
        //当前用户在app内已存在----更新为不查看好友vip信息
        updateVipViewRecord(unionId);
        return Message.build();
    }

    /**
     * 通过支付生成VIP
     *
     * @param userId
     * @return
     */
    @Override
    public Long addUserVipByPay(Long userId) {
        Vip vip = vipMapper.selectVipByUser(userId, null);
        Boolean isUpdate = Boolean.FALSE;
        if (vip == null) {
            isUpdate = Boolean.TRUE;
            User user = userIntegrationService.selectByPrimaryKey(userId);
            vip = generateVip(userIntegrationService.getLocationByUser(user).getCode(), user.getWeixin(), userId);
        }
        if (VipStatusEnum.ACTIVATED.getType().equals(vip.getStatus())) {
            return null;
        }
        vip.setSource((byte) 2);
        Date now = new Date();
        vip.setModifyTime(now);
        vip.setActivationDate(now);
        vip.setExpiredDate(DateUtils.addField(now, Calendar.YEAR, 1));
        vip.setStatus(VipStatusEnum.ACTIVATED.getType());
        vip.setShowGuide((byte) 1);
        if (isUpdate) {
            saveUserVip(vip);
        } else {
            vipMapper.updateUserVip(vip);
        }
        logger.info("用户[{}]购买vip成功", userId);

        movieService.doGeneratorMovie(userId, vip.getActivationAreaCode());
        bizLogService.buyVip(userId);
        pushMsgIntegrationService.pushActivationVipSuccessMessage(userId);

        //vip激活存入redis缓存
//        saveUserToCache(vip.getUserId());

        removeUserLayer(vip.getUserId());

        return vip.getId();
    }

    @Override
    public Message updateVipViewRecord(String unionId) {
        vipViewRecordMapper.updateViewRecordByUser(unionId);
        return Message.build();
    }

    @Override
    public Integer countVipByUser(Long userId) {
        return vipMapper.countVipByUser(userId);
    }

    @Override
    public WechatVipInfoDTO addWechatUserVip(String areaCode, String wechatUid, String wechatName, String wechatImg) {
        Vip vip = vipMapper.selectVipByUser(null, wechatUid);

        if (vip == null) {
            vip = generateVip(areaCode, wechatUid, null);
            saveUserVip(vip);
            saveUserToCache(wechatName, wechatImg);
        } else {
            if (!redisStringAdapter.hasKey(getUserReceiveVipKey(vip.getWechatUid()))) {
                Date now = new Date();
                Long time = DateUtils.addField(now, Calendar.MINUTE, 30).getTime();
                redisStringAdapter.set(getUserReceiveVipKey(vip.getWechatUid()), time.toString(), 1800L);
            }
        }

        return convertVip(vip, areaCode);
    }

    /**
     * 保存vip----插入数据库
     *
     * @param vip
     */
    private void saveUserVip(Vip vip) {
        String randomId = sequenceCreater.nextStringId();
        if (distributedLock.lock(vip.getWechatUid(), randomId, 1, TimeUnit.SECONDS)) {
            if (vipMapper.selectVipByUser(null, vip.getWechatUid()) == null) {
                vipMapper.insertSelective(vip);
            }
        }
    }

    /**
     * 封装成功助力的好友列表
     *
     * @param userId
     * @return
     */
    @Override
    public UserSuccessFriendDTO getSuccessFriend(Long userId) {
        UserSuccessFriendDTO userSuccessFriendDTO = new UserSuccessFriendDTO();
        List<WechatUserInfoDTO> list = getWechatUserList(userId, (byte) 1, null);
        //获取助力列表
        userSuccessFriendDTO.setHelpFriendList(list);

        Vip vip = vipMapper.selectVipByUser(userId, null);
        if (null == vip) {
            vip = new Vip();
        }
        if (null == vip.getInvitedCount()) {
            vip.setInvitedCount(3);
        }
        userSuccessFriendDTO.setInvitedCount(Long.valueOf(vip.getInvitedCount()));
        if (CollectionUtils.isNotEmpty(list) && list.size() > vip.getInvitedCount()) {
            List<WechatUserInfoDTO> userInfoDTOList = Lists.newArrayList();
            for (int i = 0; i < userSuccessFriendDTO.getInvitedCount(); i++) {
                userInfoDTOList.add(list.get(i));
            }
            userSuccessFriendDTO.setHelpFriendList(userInfoDTOList);
        }

        //获取查看人数
        List<WechatUserInfoDTO> viewList = getWechatUserList(userId, null, (byte) 1);
        userSuccessFriendDTO.setViewCount((long) viewList.size());
        return userSuccessFriendDTO;
    }

    @Override
    public List<WechatUserInfoDTO> getInviteFriendRecord(Long userId) {
        return getWechatUserList(userId, null, (byte) 1);
    }

    @Override
    public Message addViewRecord(RecordWechatParam recordWechatParam) {
        Message message = Message.build();
        logger.debug("添加vip查看记录, 邀请人：[{}], 受邀人[{}]", recordWechatParam.getUserId(),
                recordWechatParam.getWechatUid());
        // 存在记录---更新
        VipViewRecord vipViewRecord = vipViewRecordMapper.getViewRecordByUserAndWechatUid(recordWechatParam);
        if (vipViewRecord != null) {
            logger.debug("邀请记录已存在: 数据：[{}]", JSON.toJSONString(vipViewRecord));
            if (vipViewRecord.getDisplayFlag() == 0) {
                return message;
            }
            vipViewRecord.setViewTime(new Date());
            vipViewRecordMapper.updateByPrimaryKey(vipViewRecord);
            return message;
        }

        //微信用户已存在---不记录
//        if (userIntegrationService.checkUnionIdIsUser(recordWechatParam.getWechatUid())) {
//            return message;
//        }

        // 本人查看---不记录
        UserInfoDTO userInfoDTO = userIntegrationService.getUserFromRedisDB(recordWechatParam.getUserId());
        if (recordWechatParam.getWechatUid().equals(userInfoDTO.getWeixin())) {
            logger.debug("本人查看不记录");
            return message;
        }

        //设置默认头像
        if (StringUtils.isEmpty(recordWechatParam.getWechatImg())) {
            recordWechatParam.setWechatImg(userIntegrationService.getDefaultHeadImgUrl());
        }

        VipViewRecord viewRecord = new VipViewRecord();
        viewRecord.setId(sequenceCreater.nextLongId());
        BeanUtils.copyProperties(recordWechatParam, viewRecord);
        viewRecord.setDisplayFlag((byte) 1);
        viewRecord.setViewTime(new Date());
        vipViewRecordMapper.insertSelective(viewRecord);

        return Message.build();
    }

    @Override
    public CustomerServiceDTO getCustomerService() {
        CustomerServiceDTO customerServiceDTO = new CustomerServiceDTO();
        customerServiceDTO.setCustomerServiceName(wechatCustomerServiceProperties.getCustomerServiceName());
        customerServiceDTO.setCustomerServiceImg(wechatCustomerServiceProperties.getCustomerServiceImg());

        return customerServiceDTO;
    }

    @Override
    public Message modifyUserGuid(Long userId) {
        vipMapper.updateUserGuide(userId);
        return Message.build();
    }

    @Override
    public UserVipInfoDTO modifyAndGetUserVip(Long userId, String areaCode) {
        Vip vip = vipMapper.selectVipByUser(userId, null);

        if (vip == null) {
            UserInfoDTO userInfoDTO = userIntegrationService.getUserFromRedisDB(userId);
            if (StringUtils.isNotEmpty(userInfoDTO.getWeixin())) {
                vip = vipMapper.selectVipByUser(null, userInfoDTO.getWeixin());
                if (vip == null) {
                    vip = generateVip(areaCode, userInfoDTO.getWeixin(), userId);
                    saveUserVip(vip);
                }
            } else {
//                userService.updateUserBaseInfo(userInfoDTO.getId());
                vip = generateVip(areaCode, null, userId);
            }

        }

        if (StringUtils.isEmpty(vip.getWechatUid())) {
            UserInfoDTO userInfoDTO = userIntegrationService.getUserFromRedisDB(userId);
            if (StringUtils.isNotEmpty(userInfoDTO.getWeixin())) {
                vip.setWechatUid(userInfoDTO.getWeixin());
                vipMapper.updateUserVip(vip);
            }
        }

        if (vip.getUserId() == null) {
            updateUserVip(userId, vip.getWechatUid());
        }

        return convertUserVip(vip);
    }

    @Override
    public boolean checkUserVip(Long userId) {
        Vip vip = vipMapper.selectVipByUser(userId, null);
        return null != vip && VipStatusEnum.ACTIVATED.getType().equals(vip.getStatus());
    }

    @Override
    public WechatVipInfoDTO getWechatVip(String unionId, String areaCode) {
        Vip vip = vipMapper.selectVipByUser(null, unionId);
        return convertVip(vip, areaCode);
    }

    @Override
    public List<WechatUserInfoDTO> getVipRank() {
        return new ArrayList<>(getUserVipRank());
    }

    @Override
    public Message modifyUserShowLayer(Long userId) {
        redisStringAdapter.set(getUserShowLayerKey(userId), 1);
        return Message.build();
    }

    @Override
    public Vip getVipByUser(Long userId) {
        return vipMapper.selectVipByUser(userId, null);
    }

    @Override
    public List<WechatUserInfoDTO> getAllSuccessWechatUserList(Long userId) {
        return getWechatUserList(userId, (byte) 1, null);
    }

    /**
     * 转换VIP信息
     *
     * @param vip
     * @return
     */
    private WechatVipInfoDTO convertVip(Vip vip, String areaCode) {
        WechatVipInfoDTO wechatVipInfoDTO = new WechatVipInfoDTO();

        if (vip == null) {
            wechatVipInfoDTO.setActivationVip(VipStatusEnum.RECEIVE_NOT_LOGIN.getType());
            return wechatVipInfoDTO;
        }

        wechatVipInfoDTO.setNum(vip.getNum());
        wechatVipInfoDTO.setSource(vip.getSource());

        if (vip.getUserId() == null) {
            wechatVipInfoDTO.setActivationVip(VipStatusEnum.RECEIVE_NOT_LOGIN.getType());

            if (redisStringAdapter.hasKey(getUserReceiveVipKey(vip.getWechatUid()))) {
                wechatVipInfoDTO.setRemainDate(redisStringAdapter.getLong(getUserReceiveVipKey(vip.getWechatUid())));
            }
        }

        LocationDetailDTO locationDetailDTO = locationFacadeService.getLocationDetailByCode(areaCode);
        if (locationDetailDTO != null) {
            wechatVipInfoDTO.setApplyArea(locationDetailDTO.getFullName());
            wechatVipInfoDTO.setAreaName(locationDetailDTO.getName());
        }
        if (vip.getUserId() != null) {
            wechatVipInfoDTO.setActivationVip(vip.getStatus());
        }

        return wechatVipInfoDTO;
    }

    /**
     * 转换VIP信息
     *
     * @param vip
     * @return
     */
    private UserVipInfoDTO convertUserVip(Vip vip) {
        UserVipInfoDTO userVipInfoDTO = new UserVipInfoDTO();

        userVipInfoDTO.setNum(vip.getNum());
        userVipInfoDTO.setInvitedCount(vip.getInvitedCount() == null ? 3 : Long.valueOf(vip.getInvitedCount()));
        userVipInfoDTO.setShowGuide(vip.getShowGuide());
        userVipInfoDTO.setActivationVip(vip.getStatus());
        userVipInfoDTO.setSource(vip.getSource());

        if (VipStatusEnum.ACTIVATED.getType().equals(vip.getStatus())) {
            Date now = new Date();
            Integer remainDay = DateUtils.getDiffDays(vip.getExpiredDate(), now, Boolean.FALSE);
            if (remainDay <= 0) {
                remainDay = -1;
            }
            if (remainDay > 0) {
                //如果是未过期的VIP会员则去判断他是否有右上角VIP电影票的小红点
                NoticeDTO notice = movieService.getMovieVipNotice(vip.getUserId());
                if (null != notice) {
                    userVipInfoDTO.setNotice(notice);
                }
            }
            userVipInfoDTO.setRemainDay((long) (remainDay + 1));
            userVipInfoDTO.setActivationStatus((byte) 2);
        }

        //设置用户信息
        UserInfoDTO userInfoDTO = userIntegrationService.getUserFromRedisDB(vip.getUserId());
        if (userInfoDTO != null) {
            WechatUserInfoDTO wechatUserInfoDTO = new WechatUserInfoDTO();
            wechatUserInfoDTO.setWechatName(userInfoDTO.getNickname());
            wechatUserInfoDTO.setWechatImg(userInfoDTO.getHeadImg());
            userVipInfoDTO.setWechatUserInfoDTO(wechatUserInfoDTO);
        }

        //设置用户区域信息
        LocationDetailDTO locationIntegrationDetailDTO = userIntegrationService.getLocationByUser(userIntegrationService.selectByPrimaryKey(vip.getUserId()));
        if (locationIntegrationDetailDTO != null) {
            userVipInfoDTO.setAreaName(locationIntegrationDetailDTO.getName());
            userVipInfoDTO.setApplyArea(locationIntegrationDetailDTO.getFullName());
            userVipInfoDTO.setReductionAmount(locationIntegrationDetailDTO.getReductionAmount());
        }

        //是否显示浮层--默认不显示
        if (!VipStatusEnum.ACTIVATED.getType().equals(vip.getStatus())) {
            userVipInfoDTO.setShowLayer(getUserShowLayer(vip.getUserId(), vip.getStatus()));
            if (userVipInfoDTO.getShowLayer() == 1) {
                List<WechatUserInfoDTO> list = getWechatUserList(vip.getUserId(), (byte) 1, null);
                userVipInfoDTO.setHelpFriendList(list);
                if (CollectionUtils.isEmpty(list)) {
                    userVipInfoDTO.setShowLayer((byte) 0);
                }
                if (CollectionUtils.isNotEmpty(list) && list.size() < vip.getInvitedCount()) {
                    userVipInfoDTO.setShowLayer(expireUserShowLayer(vip.getUserId()));
                }
            }
            List<WechatUserInfoDTO> recordList = getWechatUserList(vip.getUserId(), null, null);
            if (CollectionUtils.isEmpty(recordList)) {
                userVipInfoDTO.setActivationStatus((byte) 0);
            } else {
                userVipInfoDTO.setActivationStatus((byte) 1);
            }
        }

        if (userVipInfoDTO.getShowGuide() == 1) {
            List<WechatUserInfoDTO> list = getWechatUserList(vip.getUserId(), (byte) 1, null);
            userVipInfoDTO.setHelpFriendList(list);
        }

        if (VipStatusEnum.ACTIVATED.getType().equals(vip.getStatus())) {
            userVipInfoDTO.setShowLayer((byte) 0);
        }


        return userVipInfoDTO;
    }

    /**
     * 生成vip信息
     *
     * @return
     */
    private Vip generateVip(String areaCode, String wechatUid, Long userId) {
        Vip vip = new Vip();

        vip.setId(sequenceCreater.nextLongId());
        vip.setNum(generateVipNum());
        if (userId != null) {
            vip.setUserId(userId);
        }

        LocationDetailDTO locationDTO = locationFacadeService.getLocationDetailByCode(areaCode);
        //防止查询对应地区未开通VIP 邀请人数为null的问题
        if (locationDTO == null || locationDTO.getInvitedCount() == null) {
            locationDTO = locationFacadeService.getAcitivateVipLocation();
            logger.error("用户id:{},用户微信:{},当前地区:{},经检测并无开通VIP,地区转换为:{}", userId, wechatUid, areaCode, locationDTO.getName());
        }

        vip.setActivationAreaCode(locationDTO.getCode());
        vip.setInvitedCount(locationDTO.getInvitedCount());
        vip.setWechatUid(wechatUid);
        Date now = new Date();
        vip.setCreateTime(now);
        vip.setWechatUid(wechatUid);
        vip.setSource((byte) 1);
        vip.setStatus(VipStatusEnum.INACTIVATE.getType());
        vip.setShowGuide((byte) 0);

        //半个小时过期
        Long time = DateUtils.addField(now, Calendar.MINUTE, 30).getTime();
        if (StringUtils.isNotEmpty(wechatUid)) {
            redisStringAdapter.set(getUserReceiveVipKey(wechatUid), time.toString(), 1800L);
        }
        return vip;
    }

    /**
     * 生成vip卡号
     *
     * @return
     */
    private String generateVipNum() {
        Set<Integer> set = redisSetAdapter.getAllMembers(getUserVipNumKey(), Integer.class);
        Integer num = getRandomIntInRangeWithExclude(10000000, 99999999, set);
        redisSetAdapter.add(getUserVipNumKey(), num);
        return String.valueOf(num);
    }

    /**
     * 获取用户微信相关信息返回给客户端
     *
     * @param vipViewRecord
     * @return
     */
    private WechatUserInfoDTO convertVipViewRecord(VipViewRecord vipViewRecord) {
        WechatUserInfoDTO wechatUserInfoDTO = new WechatUserInfoDTO();
        wechatUserInfoDTO.setWechatImg(vipViewRecord.getWechatImg());
        wechatUserInfoDTO.setWechatName(vipViewRecord.getWechatName());
        return wechatUserInfoDTO;
    }

    /**
     * 获取并转换信息
     *
     * @param userId 用户id
     * @param checked 是否确认邀请
     * @param displayFlag 是否显示
     * @return 受邀人列表
     */
    private List<WechatUserInfoDTO> getWechatUserList(Long userId, Byte checked, Byte displayFlag) {
        List<VipViewRecord> vipViewRecordList = vipViewRecordMapper.selectVipRecordList(userId, checked, displayFlag);
        return vipViewRecordList.stream().map(this::convertVipViewRecord).collect(Collectors.toList());
    }

    /**
     * 获取用户已激活vip记录的key
     *
     * @return
     */
    private KeyGenerator getUserVipRankKey() {
        return RedisConfig.USER_VIP_RANK.copy();
    }

    /**
     * 获取用户已激活vip记录的key
     *
     * @return
     */
    private KeyGenerator getUserVipRankImgKey() {
        return RedisConfig.USER_VIP_RANK_IMG.copy();
    }

    /**
     * 获取用户领取vip卡生成的编号记录key
     *
     * @return
     */
    private KeyGenerator getUserVipNumKey() {
        return RedisConfig.USER_VIP_NUM_RECORD.copy();
    }

    /**
     * 获取用户显示浮层的过期时间key
     *
     * @param userId
     * @return
     */
    private KeyGenerator getUserLayerExpireTimeKey(Long userId) {
        return RedisConfig.USER_LAYER_EXPIRE_TIME.copy().appendKey(userId);
    }

    /**
     * 获取用户领取vip剩余的时间
     *
     * @param wechatUid
     * @return
     */
    private KeyGenerator getUserReceiveVipKey(String wechatUid) {
        return RedisConfig.USER_RECEIVE_VIP_TIME.copy().appendKey(wechatUid);
    }

    /**
     * 用户不显示浮层key
     * 存在这个key则不显示浮层
     *
     * @param userId
     * @return
     */
    private KeyGenerator getUserShowLayerKey(Long userId) {
        return RedisConfig.USER_SHOW_LAYER.copy().appendKey(userId);
    }

    /**
     * 生成排除{@code exclude} 在内的随机数
     *
     * @param min
     * @param max
     * @param exclude
     * @return
     */
    private Integer getRandomIntInRangeWithExclude(int min, int max, Set<Integer> exclude) {
        Random random = new Random();

        return random.ints(min, (max + 1)).filter((r) -> !exclude.contains(r)).limit(1).findFirst().getAsInt();
    }

    /**
     * 获取轮播信息（从redis获取）
     *
     * @return
     */
    private Set<String> getUserVipRankByCache() {
        Set<String> set = Sets.newHashSet();
        if (redisSetAdapter.hasKey(getUserVipRankKey())) {
            set = redisSetAdapter.getAllMembers(getUserVipRankKey(), String.class);
        }

        if (CollectionUtils.isEmpty(set)) {
            List<String> list = vipRankProperties.getRank();
            set = new HashSet<>(list);
            Object[] objects = list.toArray();
            if (CollectionUtils.isNotEmpty(list)) {
                redisSetAdapter.add(getUserVipRankKey(), objects);
            }
        }

        return set;
    }

    /**
     * 获取所有的轮播信息
     *
     * @return
     */
    private Set<WechatUserInfoDTO> getUserVipRank() {
        Set<WechatUserInfoDTO> set = Sets.newHashSet();
        if (redisSetAdapter.hasKey(getUserVipRankKey())) {
            TypeReference<WechatUserInfoDTO> wechatUserInfoDTOTypeReference = new TypeReference<WechatUserInfoDTO>() {
            };
            set = redisSetAdapter.getAllMembers(getUserVipRankKey(), wechatUserInfoDTOTypeReference);
        }

        if (CollectionUtils.isEmpty(set)) {
            List<String> list = vipRankProperties.getRank();
            List<String> imgList = vipRankProperties.getRankImg();

            set = convertNameAndImg(list, imgList);

            Object[] objects = set.toArray();
            if (CollectionUtils.isNotEmpty(list)) {
                redisSetAdapter.add(getUserVipRankKey(), objects);
            }
        }

        return set;
    }

    /**
     * 转换信息
     *
     * @param list
     * @param imgList
     * @return
     */
    private Set<WechatUserInfoDTO> convertNameAndImg(List<String> list, List<String> imgList) {
        Set<WechatUserInfoDTO> wechatUserInfoDTOS = Sets.newHashSet();
        for (int i = 0, size = list.size(); i < size; i++) {
            WechatUserInfoDTO wechatUserInfoDTO = new WechatUserInfoDTO();
            wechatUserInfoDTO.setWechatName(list.get(i));
            wechatUserInfoDTO.setWechatImg(imgList.get(i));
            wechatUserInfoDTOS.add(wechatUserInfoDTO);
        }

        return wechatUserInfoDTOS;
    }

    /**
     * 确认用户和好友的绑定关系
     *
     * @param unionId
     */
    private void confirmUserRelation(String unionId, Long userId, Integer platform) {
        VipViewRecord vipViewRecord = vipViewRecordMapper.getCheckedByWechatUid(unionId);
        //该用户已帮助好友激活vip
        if (vipViewRecord != null) {
            return;
        }

        vipViewRecord = vipViewRecordMapper.getLastVipViewRecord(unionId);
        if (vipViewRecord == null) {
            return;
        }

        //确认VIP邀请关系
        Date now = new Date();
        vipViewRecord.setCheckTime(now);
        vipViewRecord.setDisplayFlag((byte) 0);
        vipViewRecord.setChecked((byte) 1);
        vipViewRecordMapper.updateByPrimaryKey(vipViewRecord);
        logger.debug("确认vip邀请关系：" + JSON.toJSONString(vipViewRecord));

        Vip vip = vipMapper.selectVipByUser(vipViewRecord.getUserId(), null);
        if (vip == null) {
            return;
        }

        //若邀请人未激活VIP，检测并激活VIP
        activationVip(vip, vipViewRecord.getWechatName(), platform);
        //记录邀请VIP激活成功信息
        bizLogService.inviteVip(userId, platform);
    }

    /**
     * 激活用户vip
     *
     * @param vip
     * @param wechatName
     */
    private void activationVip(Vip vip, String wechatName, Integer platform) {
        if (!VipStatusEnum.INACTIVATE.getType().equals(vip.getStatus())) {
            return;
        }

        List<WechatUserInfoDTO> list = getWechatUserList(vip.getUserId(), (byte) 1, null);
        if (CollectionUtils.isEmpty(list)) {
            return;
        }
        //移除好友助力24小时冷却时间
        redisStringAdapter.remove(getUserLayerExpireTimeKey(vip.getUserId()));
        if (list.size() < vip.getInvitedCount()) {
            //推送好友助力消息
            pushMsgIntegrationService.pushHelpMessage(vip.getUserId(), wechatName, vip.getInvitedCount() - list.size());
        } else {
            Date now = new Date();
            vip.setModifyTime(now);
            vip.setActivationDate(now);
            vip.setExpiredDate(DateUtils.addField(now, Calendar.YEAR, 1));
            vip.setStatus(VipStatusEnum.ACTIVATED.getType());
            vip.setShowGuide((byte) 1);
            vipMapper.updateUserVip(vip);
            logger.debug("激活邀请人vip：" + JSON.toJSONString(vip));

            //vip激活存入redis缓存
//            saveUserToCache(vip.getUserId());

            removeUserLayer(vip.getUserId());

            //当VIP用户生成时，系统赠送一张电影票
            movieService.doGeneratorMovie(vip.getUserId(), vip.getActivationAreaCode());

            //推送vip激活消息
            pushMsgIntegrationService.pushActivationVipSuccessMessage(vip.getUserId());

            bizLogService.activateVip(vip.getUserId(), platform);
        }
    }

    /**
     * 把已激活vip的用户存入缓存
     *
     * @param userId
     */
    private void saveUserToCache(Long userId) {
        UserInfoDTO userInfoDTO = userIntegrationService.getUserFromRedisDB(userId);

        if (redisSetAdapter.size(getUserVipRankKey()) >= 20) {
            redisSetAdapter.pop(getUserVipRankKey(), 1L, String.class);
        }

        redisSetAdapter.add(getUserVipRankKey(), userInfoDTO.getNickname());
    }

    /**
     * 把点击领取vip的用户存入缓存
     *
     * @param wechaName
     */
    private void saveUserToCache(String wechaName, String wechatImg) {
        TypeReference<WechatUserInfoDTO> wechatUserInfoDTOTypeReference = new TypeReference<WechatUserInfoDTO>() {};

        if (redisSetAdapter.size(getUserVipRankKey()) >= 50) {
            redisSetAdapter.pop(getUserVipRankKey(), 1L, wechatUserInfoDTOTypeReference);
        }

        WechatUserInfoDTO wechatUserInfoDTO = new WechatUserInfoDTO();
        wechatUserInfoDTO.setWechatImg(wechatImg);
        wechatUserInfoDTO.setWechatName(wechaName);
        List<WechatUserInfoDTO> wechatUserInfoDTOS = Lists.newArrayList(wechatUserInfoDTO);

        redisSetAdapter.add(getUserVipRankKey(), wechatUserInfoDTOS.toArray());
    }

    /**
     * 更新用户vip信息
     *
     * @param userId
     * @param unionId
     */
    private void updateUserVip(Long userId, String unionId) {
        Vip vip = vipMapper.selectVipByUser(null, unionId);
        if (vip == null || vip.getUserId() != null) {
            return;
        }
        vip.setUserId(userId);
        vipMapper.updateUserVip(vip);
    }

    /**
     * 根据用户是否激活vip来判断是否显示浮层
     *
     * @param userId
     * @param status
     * @return
     */
    private Byte getUserShowLayer(Long userId, Byte status) {
        //用户已激活vip不显示浮层
        Boolean isSuccessVip = VipStatusEnum.ACTIVATED.getType().equals(status);
        if (isSuccessVip) {
            return 0;
        }

        if (redisStringAdapter.hasKey(getUserShowLayerKey(userId))) {
            return 0;
        }
        return 1;
    }

    /**
     * 根据用户上一次浮层显示的时间来判断是否显示浮层
     *
     * @param userId
     * @return
     */
    private Byte expireUserShowLayer(Long userId) {
        //24小时内已显示过浮层 不显示浮层
        if (redisStringAdapter.hasKey(getUserLayerExpireTimeKey(userId))) {
            return 0;
        }
        redisStringAdapter.set(getUserLayerExpireTimeKey(userId), DateUtils.getCurrentDateTime(), 3600 * 24L);
        return 1;
    }

    /**
     * 移除用户是否显示浮层key
     *
     * @param userId
     */
    private void removeUserLayer(Long userId) {
        redisStringAdapter.remove(getUserShowLayerKey(userId));
    }

}
