package com.bxm.localnews.activity.service.redpacket.command.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.bxm.component.httpclient.service.OKHttpService;
import com.bxm.localnews.activity.common.constant.*;
import com.bxm.localnews.activity.common.utils.AmountUtil;
import com.bxm.localnews.activity.common.utils.DateUtil;
import com.bxm.localnews.activity.domain.*;
import com.bxm.localnews.activity.dto.*;
import com.bxm.localnews.activity.param.ObtainHistoryParam;
import com.bxm.localnews.activity.param.RedPacketAddParam;
import com.bxm.localnews.activity.param.RedPacketUpdateParam;
import com.bxm.localnews.activity.param.WallRedPacketParam;
import com.bxm.localnews.activity.service.DingDingMsgService;
import com.bxm.localnews.activity.service.RedPacketFacadeService;
import com.bxm.localnews.activity.service.redpacket.command.*;
import com.bxm.localnews.activity.vo.*;
import com.bxm.localnews.base.service.BizLogService;
import com.bxm.localnews.base.service.LocationFacadeService;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.common.exception.BusinessException;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.payment.dto.OrderStatusDTO;
import com.bxm.localnews.payment.facde.service.PayService;
import com.bxm.localnews.vo.User;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.NumberUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.PageWarper;
import com.github.pagehelper.PageHelper;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * <p></p>
 * @Author: JandMin
 * @since: 1.0.0
 * @Date: 2019/4/18
 */
@Service
public class RedPacketServiceImpl extends BaseService implements RedPacketService, RedPacketFacadeService {
    private final static Logger LOG = LoggerFactory.getLogger(RedPacketServiceImpl.class);

    private static final Map<String, String> HEADER_MAP = Maps.newHashMap();

    @Autowired
    private BizLogService bizLogService;

    @Autowired
    private CommandGradeService commandGradeService;

    @Autowired
    private RedPacketObtainService redPacketObtainService;

    @Autowired
    private RedPacketObtainFailedHistoryService obtainFailedHistoryService;

    @Autowired
    private UserIntegrationService userIntegrationService;

    @Autowired
    private LocationFacadeService locationFacadeService;

    @Autowired
    private RedPacketStateService redPacketStateService;

    @Autowired
    private RedPacketMapper redPacketMapper;

    @Autowired
    private RedPacketObtainHistoryMapper obtainHistoryMapper;

    @Autowired
    private RedPacketContentMapper redPacketContentMapper;

    @Autowired
    private RedPacketItemMapper redPacketItemMapper;

    @Autowired
    private RedPacketShareMapper shareMapper;

    @Resource
    private SequenceCreater sequenceCreater;

    @Autowired
    private RedisStringAdapter redisStringAdapter;

    @Autowired
    private OKHttpService okHttpService;

    @Value("${dingding.robot.webhook}")
    private String webhook;

    @Autowired
    private PayService payService;

    @Autowired
    private DingDingMsgService dingMsgService;

    @Override
    public RedPacket selectOne(Long rpId) {
        return redPacketMapper.queryOne(rpId);
    }

    @Override
    public Boolean getHasAvailableRedPacket(Long userId, String areaCode) {
        // 获取当前地区的可用红包
        HashSet<Long> redPacketIds = redPacketMapper.queryByAreaCode(areaCode);
        if (CollectionUtils.isEmpty(redPacketIds)) {
            return Boolean.FALSE;
        }

        // 该地区的可用红包中哪些被用户领取过
        HashSet<Long> userGetRpIds = obtainHistoryMapper.queryRpIdsWhichUserGet(userId, redPacketIds);

        if (CollectionUtils.isEmpty(userGetRpIds)) {
            return Boolean.TRUE;
        }

        redPacketIds.removeAll(userGetRpIds);

        if (CollectionUtils.isEmpty(redPacketIds)) {
            return Boolean.FALSE;
        }
        return Boolean.TRUE;
    }

    @Override
    public void updateRedpacketActiveStatus(Long rpId, String paymentNum, Integer platform) {
        logger.info("订单支付成功状态个更新：订单编号[{}],红包id[{}]", paymentNum, rpId);

        RedPacket redPacket = selectOne(rpId);
        if (null == redPacket) {
            throw new BusinessException("红包不存在");
        }

        // 红包状态已修改 不做操作
        if (redPacket.getActiveStatus() != -1) {
            return;
        }
        updateActiveStatus(paymentNum, platform, redPacket);
    }

    /**
     * 获取该地区6个已抢完的红包
     * @param areaCode
     * @return
     */
    private List<WallRedPacketDTO> getObtainedAll(String areaCode) {
        return redPacketMapper.getObtainedAll(areaCode).stream().map(this::convertWallRedPacketDTO).collect(Collectors.toList());
    }

    @Override
    public PageWarper<WallRedPacketDTO> getReadPacketWall(WallRedPacketParam wallRedPacketParam) {
        // 查询当前区域 可领取或者已领取的红包墙红包
        List<RedPacketVo> redPackets = redPacketMapper.queryWallByPageSize(wallRedPacketParam);
        PageWarper pageWarper = new PageWarper<>(redPackets);

        List<WallRedPacketDTO> wallRedPacketDTOS = Lists.newArrayList();
        // 无可领取红包 返回6个已领完红包
        if (redPackets.isEmpty()) {
            List<WallRedPacketDTO> obtainedAllPackets = getObtainedAll(wallRedPacketParam.getAreaCode());
            if (CollectionUtils.isEmpty(obtainedAllPackets)) {
                pageWarper.setList(wallRedPacketDTOS);
                return pageWarper;
            }

            wallRedPacketDTOS.addAll(obtainedAllPackets);
            pageWarper.setList(wallRedPacketDTOS);
            pageWarper.setSize(obtainedAllPackets.size());
            pageWarper.setTotal(obtainedAllPackets.size());
            pageWarper.setHasNextPage(Boolean.FALSE);
            return pageWarper;
        }

        wallRedPacketDTOS.addAll(redPackets.stream().map(this::convertWallRedPacketDTO).collect(Collectors.toList()));

        // 没有下一页了 ,插入6个已领完红包
        if (!pageWarper.isHasNextPage()) {
            List<WallRedPacketDTO> obtainedAllPackets = getObtainedAll(wallRedPacketParam.getAreaCode());
            if (!CollectionUtils.isEmpty(obtainedAllPackets)) {
                wallRedPacketDTOS.addAll(obtainedAllPackets);
            }
        }
        pageWarper.setList(wallRedPacketDTOS);
        return pageWarper;
    }

    @Override
    public WallRedPacketDTO getSingleWallReadPacket(Long userId, Long rpId) {
        WallRedPacketDTO wallRedPacketDTO = new WallRedPacketDTO();
        RedPacket redPacket = selectOne(rpId);
        if (null == redPacket) {
            throw new BusinessException("红包id[" + rpId + "]无法查询到红包");
        }
        wallRedPacketDTO.setId(redPacket.getId());
        wallRedPacketDTO.setUserImg(redPacket.getUserImg());
        wallRedPacketDTO.setUserName(redPacket.getUserName());
        wallRedPacketDTO.setAmount(AmountUtil.halfUpScale2StringValue(redPacket.getAmount()));
        wallRedPacketDTO.setContent(redPacket.getContent());

        if (redPacket.getObtainStatus().equals(0)) { // 已领完
            wallRedPacketDTO.setStatus(0);
            return wallRedPacketDTO;
        }
        Boolean isObtainedByUser = redPacketStateService.isObtainedByUser(rpId, userId);
        if (isObtainedByUser) {// 已领取
            wallRedPacketDTO.setStatus(1);
            return wallRedPacketDTO;
        }
        // 可领取
        wallRedPacketDTO.setStatus(2);
        return wallRedPacketDTO;
    }

    private WallRedPacketDTO convertWallRedPacketDTO(RedPacketVo redPacket) {
        WallRedPacketDTO wallRedPacketDTO = new WallRedPacketDTO();
        wallRedPacketDTO.setId(redPacket.getId());
        wallRedPacketDTO.setContent(redPacket.getContent());
        wallRedPacketDTO.setUserImg(redPacket.getUserImg());
        wallRedPacketDTO.setUserName(redPacket.getUserName());
        wallRedPacketDTO.setStatus(redPacket.getStatus() == null ? 0 : redPacket.getStatus());
        wallRedPacketDTO.setAmount(AmountUtil.halfUpScale2StringValue(redPacket.getAmount()));
        return wallRedPacketDTO;
    }

    @Override
    public RedPacketDTO getReadPacketDetail(Long userId, Long rpId) {
        RedPacket redPacket = selectOne(rpId);
        if (null == redPacket) {
            throw new BusinessException("redpacket is not exists");
        }
        if (redPacket.getOnlineStatus().equals(OnlineStatesEnum.OFFLINE.getType())) {
            return new RedPacketDTO(OnlineStatesEnum.OFFLINE.getType());// 红包下线
        }
        ObtainStatesEnum obtainStatusEnum = redPacketStateService.rpObtainStatus(redPacket.getAmount(), redPacket.getSurplusAmount());// 红包领取状态
        UserObtainRpStatesDTO userObtainRpStatesDTO = redPacketStateService.userObtainRpStates(redPacket.getSurplusAmount(), userId, rpId);// 当前用户领取状态
        return convertRedPacketDTO(redPacket, obtainStatusEnum, userObtainRpStatesDTO);
    }

    @Override
    public RedPacketPublicDTO getPublicReadPacketDetail(Long userId, Long shareUserId, Long rpId) {
        RedPacket redPacket = selectOne(rpId);
        if (null == redPacket) {
            throw new BusinessException("redpacket is not exists");
        }
        if (redPacket.getOnlineStatus().equals(OnlineStatesEnum.OFFLINE.getType())) {
            return new RedPacketPublicDTO(OnlineStatesEnum.OFFLINE.getType());// 红包下线
        }
        ObtainStatesEnum obtainStatusEnum = redPacketStateService.rpObtainStatus(redPacket.getAmount(), redPacket.getSurplusAmount());// 红包领取状态
        UserObtainRpStatesDTO userObtainRpStatesDTO = redPacketStateService.userObtainRpStates(redPacket.getSurplusAmount(), userId, rpId);// 当前用户领取状态
        // 分享人领取信息
        RedPacketObtainHistory obtainHistory = obtainHistoryMapper.queryByUserAndRp(shareUserId, rpId);

        return convertRedPacketPublicDTO(redPacket, obtainStatusEnum, userObtainRpStatesDTO, obtainHistory);
    }

    private RedPacketPublicDTO convertRedPacketPublicDTO(RedPacket redPacket, ObtainStatesEnum obtainStatusEnum, UserObtainRpStatesDTO userObtainRpStatesDTO, RedPacketObtainHistory obtainHistory) {
        RedPacketPublicDTO redPacketPublicDTO = new RedPacketPublicDTO();
        redPacketPublicDTO.setType(redPacket.getPacketType());
        redPacketPublicDTO.setOwnerUserName(redPacket.getUserName());
        redPacketPublicDTO.setOwnerUserImg(redPacket.getUserImg());
        redPacketPublicDTO.setContent(redPacket.getContent());
        redPacketPublicDTO.setContentImg(redPacket.getPromotionImg() == null ? Lists.newArrayList() : JSONObject.parseObject(redPacket.getPromotionImg(), List.class));
        redPacketPublicDTO.setPromotionContent(redPacket.getPromotionContent());
        BigDecimal obtainAmount = redPacket.getAmount().subtract(redPacket.getSurplusAmount());// 红包被领取总金额
        redPacketPublicDTO.setAmount(AmountUtil.halfUpScale2StringValue(redPacket.getAmount()));
        redPacketPublicDTO.setObtainAmount(AmountUtil.halfUpScale2StringValue(obtainAmount));
        redPacketPublicDTO.setObtainStatus(obtainStatusEnum.getType());
        redPacketPublicDTO.setRpStatus(userObtainRpStatesDTO.getUserObtainRpStatesEnum().getType());
        redPacketPublicDTO.setUserObtainAmount(AmountUtil.halfUpScale2StringValue(userObtainRpStatesDTO.getUserObtainAmount()));

        // 分享人信息
        if (obtainHistory != null) {
            redPacketPublicDTO.setShareUserName(obtainHistory.getUserName());
            redPacketPublicDTO.setShareUserImg(obtainHistory.getUserImg());
            redPacketPublicDTO.setShareAmount(AmountUtil.halfUpScale2StringValue(obtainHistory.getObtainAmount()));
            redPacketPublicDTO.setShareScore(String.valueOf(obtainHistory.getScore()));
        }
        return redPacketPublicDTO;
    }

    private RedPacketDTO convertRedPacketDTO(RedPacket redPacket, ObtainStatesEnum obtainStatusEnum, UserObtainRpStatesDTO userObtainRpStatesDTO) {
        RedPacketDTO redPacketDTO = new RedPacketDTO();
        redPacketDTO.setType(redPacket.getPacketType());
        redPacketDTO.setOwnerUserName(redPacket.getUserName());
        redPacketDTO.setOwnerUserImg(redPacket.getUserImg());
        redPacketDTO.setContent(redPacket.getContent());
        redPacketDTO.setContentImg(redPacket.getPromotionImg() == null ? Lists.newArrayList() : JSONObject.parseObject(redPacket.getPromotionImg(), List.class));
        redPacketDTO.setPromotionContent(redPacket.getPromotionContent());
        BigDecimal obtainAmount = redPacket.getAmount().subtract(redPacket.getSurplusAmount());// 红包被领取总金额
        redPacketDTO.setAmount(AmountUtil.halfUpScale2StringValue(redPacket.getAmount()));
        redPacketDTO.setObtainAmount(AmountUtil.halfUpScale2StringValue(obtainAmount));
        redPacketDTO.setObtainStatus(obtainStatusEnum.getType());
        redPacketDTO.setRpStatus(userObtainRpStatesDTO.getUserObtainRpStatesEnum().getType());
        redPacketDTO.setUserObtainAmount(AmountUtil.halfUpScale2StringValue(userObtainRpStatesDTO.getUserObtainAmount()));
        return redPacketDTO;
    }

    @Override
    public RedPacketTotalVo getTotalRecordByUser(Integer owner, Long userId) {
        Map<String, Object> totalInfo = null;

        if (RedPacketOwnerEnum.SENT.getType() == owner) {// 我发的
            totalInfo = redPacketMapper.getTotalSentRedPacket(userId);
        }
        if (RedPacketOwnerEnum.GOTTEN.getType() == owner) {// 我领的
            totalInfo = obtainHistoryMapper.getTotalGottenRedPacket(userId);
        }

        if (null == totalInfo || totalInfo.size() == 0) {
            return new RedPacketTotalVo("0.00", 0);
        }

        return convertToRedPacketTotalVo(totalInfo);
    }

    private RedPacketTotalVo convertToRedPacketTotalVo(Map<String, Object> totalInfo) {
        RedPacketTotalVo redPacketTotalVo = new RedPacketTotalVo();
        // 数据转换成前端显示格式
        Object num = totalInfo.get("total_num");
        Object amount = totalInfo.get("total_amount");
        redPacketTotalVo.setTotalNum(null == num || !StringUtils.isNumeric(String.valueOf(num)) ? 0 : Integer.valueOf(String.valueOf(num)));
        redPacketTotalVo.setTotalAmount(null == amount ? "0.00" : NumberUtils.format(new BigDecimal(String.valueOf(amount))));
        return redPacketTotalVo;
    }

    @Override
    public PageWarper<RedPacketSentDTO> listSentRecord(Long userId, Integer current, Integer pageSize) {
        // 分页
        PageHelper.startPage(null == current ? 1 : current, null == pageSize ? 10 : pageSize);
        List<RedPacket> redPackets = redPacketMapper.listRedPacketByUser(userId);
        PageWarper pageWarper = new PageWarper(redPackets);
        List<RedPacketSentDTO> list = Lists.newArrayList();

        // 获取前端显示字段
        if (null != redPackets && redPackets.size() != 0) {
            for (RedPacket redPacket : redPackets) {
                RedPacketSentDTO redPacketSentDTO = new RedPacketSentDTO();
                redPacketSentDTO.setAmount(NumberUtils.format(redPacket.getAmount()));
                redPacketSentDTO.setContent(redPacket.getContent());
                redPacketSentDTO.setPublishTime(DateUtil.getDateLog(redPacket.getCreateTime()));
                redPacketSentDTO.setRpId(redPacket.getId());
                Integer statue = getDisplayStatus(redPacket.getActiveStatus(), redPacket.getSurplusAmount());
                redPacketSentDTO.setStatus(statue);
                redPacketSentDTO.setStatusStr(null == statue ? "" : RedPacketDisplayStatesEnum.getValue(statue));
                redPacketSentDTO.setSurplusAmount(NumberUtils.format(redPacket.getSurplusAmount()));
                redPacketSentDTO.setType(redPacket.getPacketType());
                list.add(redPacketSentDTO);
            }
        }

        pageWarper.setList(list);
        return pageWarper;
    }

    /**
     * 根据红包状态和余额获取前端展示状态
     * @param activeStatus  红包状态
     * @param surplusAmount 红包余额
     * @return java.lang.Integer
     * @throws
     * @date: 2019/4/18
     */
    private Integer getDisplayStatus(Integer activeStatus, BigDecimal surplusAmount) {
        if (null != activeStatus && RedPacketDisplayStatesEnum.PASS_AUDIT.getCode() == activeStatus.intValue()) {
            if (null != surplusAmount && 0.00F == surplusAmount.floatValue()) {
                return RedPacketDisplayStatesEnum.BROUGHT_OUT.getCode();
            }
        }
        return activeStatus;
    }

    @Override
    public PageWarper<RedPacketGottenDTO> listGottenRecord(Long userId, Integer current, Integer pageSize) {
        if (null == userId) {
            return new PageWarper();
        }
        // 分页数据
        PageHelper.startPage(null == current ? 1 : current, null == pageSize ? 10 : pageSize);
        List<RedPacketObtainHistory> obtainHistories = obtainHistoryMapper.listRedPacketByUser(userId);
        PageWarper pageWarper = new PageWarper(obtainHistories);
        // 获取展示数据
        List<RedPacketGottenDTO> list = obtainHistories.stream().map(this::convertToRedPacketGottenDTO).collect(Collectors.toList());
        pageWarper.setList(list);
        return pageWarper;
    }

    private RedPacketGottenDTO convertToRedPacketGottenDTO(RedPacketObtainHistory obtainHistory) {
        RedPacketGottenDTO gottenDTO = new RedPacketGottenDTO();
        gottenDTO.setAmount(NumberUtils.format(obtainHistory.getObtainAmount()));
        gottenDTO.setObtainTime(DateUtil.getDateLog(obtainHistory.getCreateTime()));
        gottenDTO.setRphId(obtainHistory.getRpId());
        gottenDTO.setScore(String.valueOf(obtainHistory.getScore()));
        gottenDTO.setUserName(obtainHistory.getOwnerUserName());
        return gottenDTO;
    }

    @Override
    public Boolean updateRedPacket(RedPacketUpdateParam redPacketUpdateParam, Integer platform) {
        RedPacket redPacket = selectOne(redPacketUpdateParam.getRpId());
        if (null == redPacket) {
            throw new BusinessException("redpacket is not exists");
        }
        convertRedPacket(redPacket, redPacketUpdateParam);

        Integer result = redPacketMapper.update(redPacket);

        if (result == 0) {
            return Boolean.FALSE;
        }

        // 待审核红包发送钉钉待审核消息
        if (redPacket.getActiveStatus() == 0) {
            dingMsgService.sendExamine(redPacketUpdateParam.getUserId());
        }
        return Boolean.TRUE;
    }

    @Override
    public Byte queryOrder(Long rpId, Long userId, String paymentNum, Integer platform) {
        logger.info("订单支付查询：订单编号[{}],红包id[{}]", paymentNum, rpId);

        RedPacket redPacket = selectOne(rpId);
        if (null == redPacket) {
            throw new BusinessException("红包不存在");
        }

        // 红包状态已修改 返回支付成功
        if (redPacket.getActiveStatus() != -1) {
            return 3;
        }

        OrderStatusDTO orderStatusDTO = payService.queryOrder(paymentNum);

        if (null == orderStatusDTO || null == orderStatusDTO.getStatus()) {
            throw new BusinessException("支付订单:" + paymentNum + "查询异常");
        }

        //订单支付失败
        if (orderStatusDTO.getStatus().intValue() != 3) {
            return orderStatusDTO.getStatus();
        }

        updateActiveStatus(paymentNum, platform, redPacket);

        return orderStatusDTO.getStatus();
    }

    /**
     * 更新红包支付状态
     * @param paymentNum 支付订单编号
     * @param platform   平台
     * @param redPacket  红包
     */
    private void updateActiveStatus(String paymentNum, Integer platform, RedPacket redPacket) {
        // 该红包是否为免审核红包
        boolean isUnapprovalRedPacket = isUnapprovalRedPacket(redPacket);

        if (isUnapprovalRedPacket) {
            redPacket.setActiveStatus(2);
            redPacket.setOnlineStatus(1);
            logger.info("生成已审核红包：订单编号[{}],红包id[{}]", paymentNum, redPacket.getId());
        } else {
            redPacket.setActiveStatus(0);
            redPacket.setOnlineStatus(0);
            logger.info("生成待审核红包：订单编号[{}],红包id[{}]", paymentNum, redPacket.getId());
        }
        redPacket.setPaymentNum(paymentNum);
        // 保存数据库
        Integer result = redPacketMapper.updatePayStatus(redPacket);

        if (result == 0) {
            throw new BusinessException("更新红包[" + redPacket.getId() + "]支付状态失败");
        }

        bizLogService.sendRedPacket(redPacket.getUserId(), redPacket.getAmount(), platform);

        // 红包为待审核，发送钉钉提醒审核
        if (redPacket.getActiveStatus() == 0) {
            dingMsgService.sendExamine(redPacket.getUserId());
        }
    }

    @Override
    public Boolean isActiveArea(Long rpId, String city, String district) {

        // 通过city,district 查code
        String areaCode = locationFacadeService.getLocationByName(district);
        if (StringUtils.isBlank(areaCode)) {
            areaCode = locationFacadeService.getLocationByName(city);
        }
        if (StringUtils.isBlank(areaCode)) {
            return Boolean.FALSE;
        }
        RedPacket redPacket = selectOne(rpId);
        if (null == redPacket) {
            return Boolean.FALSE;
        }
        if (redPacket.getAreaCode().equals(areaCode)) {
            return Boolean.TRUE;
        }
        return Boolean.FALSE;
    }

    @Override
    public Boolean isPaySuccess(Long rpId) {
        RedPacket redPacket = selectOne(rpId);
        if (null == redPacket) {
            throw new BusinessException("红包不存在");
        }
        // 未支付状态
        if (redPacket.getActiveStatus() == -1) {
            return Boolean.FALSE;
        }
        return Boolean.TRUE;
    }

    @Override
    public Long insertRedPacket(RedPacketAddParam redPacketAddParam) {
        RedPacket redPacket = convertRedPacket(redPacketAddParam);
        Integer result = redPacketMapper.insert(redPacket);
        return 1 == result ? redPacket.getId() : 0L;
    }

    /**
     * 用户修改红包数据转成数据库对象
     * @param redPacketUpdateParam
     * @return com.bxm.localnews.activity.vo.RedPacket
     * @throws
     * @date: 2019/4/19
     */
    private void convertRedPacket(RedPacket redPacket, RedPacketUpdateParam redPacketUpdateParam) {
        if (null != redPacketUpdateParam.getTwisterId() &&
                0L != redPacketUpdateParam.getTwisterId().longValue() &&
                CollectionUtils.isEmpty(redPacketUpdateParam.getPromotionImgs()) &&
                StringUtils.isBlank(redPacketUpdateParam.getPromotionContent())) {
            // 使用了口令模板，且宣传图文为空，免审核
            redPacket.setTwisterId(redPacketUpdateParam.getTwisterId());
            redPacket.setActiveStatus(2);
            redPacket.setOnlineStatus(1);
        } else {
            redPacket.setTwisterId(0L);
            redPacket.setActiveStatus(0);
            redPacket.setOnlineStatus(0);
        }
        redPacket.setContent(redPacketUpdateParam.getContent());
        redPacket.setPromotionContent(redPacketUpdateParam.getPromotionContent());
        logger.info("用户修改红包数据,promotionImgs:[{}]", redPacketUpdateParam.getPromotionImgs());
        redPacket.setPromotionImg(JSON.toJSONString(redPacketUpdateParam.getPromotionImgs(), SerializerFeature.WriteNullListAsEmpty));
    }

    /**
     * 红包创建参数转换数据库对象
     * @param redPacketAddParam
     * @return com.bxm.localnews.activity.vo.RedPacket
     * @throws
     * @date: 2019/4/19
     */
    private RedPacket convertRedPacket(RedPacketAddParam redPacketAddParam) {
        RedPacket redPacket = new RedPacket();
        redPacket.setId(sequenceCreater.nextLongId());
        redPacket.setUserId(redPacketAddParam.getUserId());
        redPacket.setUserImg(redPacketAddParam.getUserImg());
        redPacket.setUserName(redPacketAddParam.getUserName());
        redPacket.setAmount(new BigDecimal(redPacketAddParam.getAmount()));
        redPacket.setSurplusAmount(redPacket.getAmount());
        redPacket.setExpectNum(redPacketAddParam.getExperctNum());
        redPacket.setActualNum(0);
        redPacket.setPacketType(redPacketAddParam.getType());

        redPacket.setTwisterId(null == redPacketAddParam.getTwisterId() ? 0L : redPacketAddParam.getTwisterId());
        redPacket.setActiveStatus(-1);
        redPacket.setOnlineStatus(0);

        if (null != redPacketAddParam.getPromotionImgs()) {
            List<String> compressList = Lists.newArrayList();
            redPacketAddParam.getPromotionImgs().forEach(img -> {
                if (img.contains("?")) {
                    compressList.add(img);
                } else {
                    compressList.add(img + "?x-oss-process=style/mcompress");
                }
            });
            redPacket.setPromotionImg(JSON.toJSONString(compressList, SerializerFeature.WriteNullListAsEmpty));
        }

        redPacket.setContent(redPacketAddParam.getContent());
        redPacket.setPromotionContent(redPacketAddParam.getPromotionContent());
        redPacket.setAreaCode(redPacketAddParam.getAreaCode());
        redPacket.setAreaName(redPacketAddParam.getAreaName());
        return redPacket;
    }

    @Override
    public RedPacketShareDTO getShareInfo(Long rpId, Long userId, Integer type, String nickname) {
        logger.info("[获取分享信息] rpId:[{}],userId:[{}],type:[{}],nickname:[{}]", rpId, userId, type, nickname);
        if (null == type) {
            LOG.warn("[getShareInfo] type is null");
            return null;
        }
        // 分享与模板
        List<RedPacketShare> shares = shareMapper.listShare(type);
        if (null == shares || shares.size() == 0) {
            LOG.warn("[getShareInfo] share info is empty");
            return null;
        }

        Long num = 0L;
        if (1 == type) {
            //1：未参与
            num = redisStringAdapter.increment(RedisConfig.RED_PACKET_SHARE_UNINVOLVED);
        } else if (2 == type) {
            //2：已参与
            num = redisStringAdapter.increment(RedisConfig.RED_PACKET_SHARE_INVOLVED);
        }
        RedPacketShare redPacketShare = shares.get(num.intValue() % shares.size());
        if (null == redPacketShare || StringUtils.isBlank(redPacketShare.getContent())) {
            LOG.warn("[getShareInfo] share info is null,type:{}", type);
            return null;
        }

        // 模板内容替换
        String content = redPacketShare.getContent();
        String obtainAmountStr = null;

        // 红包领取记录
        RedPacketObtainHistory obtainHistory = obtainHistoryMapper.queryByUserAndRp(userId, rpId);
        content = content.replace("{nickname}", nickname);
        if (obtainHistory != null) {
            obtainAmountStr = AmountUtil.halfUpScale2StringValue(obtainHistory.getObtainAmount());
            content = content.replace("{score}", obtainHistory.getScore() + "");
            content = content.replace("{money}", obtainAmountStr);
        }

        // 返回结果
        RedPacketShareDTO result = new RedPacketShareDTO();
        result.setId(redPacketShare.getId());
        result.setShare(content);
        result.setImgUrl(redPacketShare.getImgUrl());
        result.setAmount(obtainAmountStr);

        return result;
    }

    @Override
    public List<RedPacketItemDTO> listRedPacketItem(Long userId) {
        return redPacketItemMapper.listItem();
    }

    @Override
    public List<RedPacketContentDTO> listRedPacketContent(Long userId) {
        return redPacketContentMapper.listContent();
    }

    @Override
    public RedPacketUpdateDTO getRedPacketById(Long rpId, Long userId) {
        RedPacket redPacket = redPacketMapper.selectRedPacketByIdAndUserId(rpId, userId);
        if (null == redPacket) {
            LOG.warn("can not get red packet by rpId:[{}], userId:[{}]", rpId, userId);
            return null;
        }

        return converToRedPacketUpdateDTO(redPacket);
    }

    private RedPacketUpdateDTO converToRedPacketUpdateDTO(RedPacket redPacket) {
        RedPacketUpdateDTO redPacketUpdateDTO = new RedPacketUpdateDTO();
        redPacketUpdateDTO.setRpId(redPacket.getId());
        redPacketUpdateDTO.setComment(redPacket.getComment());
        redPacketUpdateDTO.setContent(redPacket.getContent());
        redPacketUpdateDTO.setPromotionContent(redPacket.getPromotionContent());
        redPacketUpdateDTO.setTwisterId(redPacket.getTwisterId());
        redPacketUpdateDTO.setPromotionImgs(JSONObject.parseObject(redPacket.getPromotionImg(), List.class));
        return redPacketUpdateDTO;
    }

    @Override
    public PageWarper<PacketHistoryDTO> getObtainHistory(ObtainHistoryParam obtainHistoryParam) {
        // 分页查询
        List<RedPacketObtainHistory> histories = obtainHistoryMapper.queryByPageSize(obtainHistoryParam);
        PageWarper pageWarper = new PageWarper<>(histories);
        // 组装List对象
        List<PacketHistoryDTO> packetHistoryDTOS = histories.stream().map(this::covertPacketHistoryDTO).collect(Collectors.toList());
        // 组装Pagewarper
        pageWarper.setList(packetHistoryDTOS);
        return pageWarper;
    }

    @Override
    public List<PacketHistoryDTO> getRanking(Long rpId) {
        // 只获取前20
        List<RedPacketObtainHistory> histories = obtainHistoryMapper.queryScoreList(rpId);
        List<PacketHistoryDTO> packetHistoryDTOS = histories.stream().map(this::covertPacketHistoryDTO).collect(Collectors.toList());
        return packetHistoryDTOS;
    }

    @Override
    public CommandResultDTO markScore(MarkScoreDTO markScoreDTO, Integer platform) {

        RedPacket redPacket = redPacketMapper.queryOne(markScoreDTO.getRpId());
        if (null == redPacket) {
            throw new BusinessException("can not get red packet by id ,id:[" + markScoreDTO.getRpId() + "]");
        }
        // 红包状态检查
        RedPacketObtainStatesEnum rpObtainState = redPacketStateService.rpStatesCheck(redPacket, markScoreDTO.getUserId());
        if (rpObtainState != null) {
            return new CommandResultDTO(rpObtainState.getType());
        }

        // 打分grade
        ScoreDto scoreDto = commandGradeService.grade(redPacket.getContent(), markScoreDTO.getCommandUrl(), markScoreDTO.getCommandLength());

        User user = userIntegrationService.selectByPrimaryKey(markScoreDTO.getUserId());

        if (null == user) {
            throw new BusinessException("can not get user by id ,id:[" + markScoreDTO.getUserId() + "]");
        }

        logger.info("口令评分成功：用户:[{}],得分:[{}],匹配度:[{}],称赞语比例:[{}],语音时长：[{}]",
                user.getNickname(),
                scoreDto.getScore(),
                scoreDto.getMatchScale(),
                scoreDto.getUserScale(),
                scoreDto.getLength());

        if (scoreDto.getScore() > 40) {
            // 领取红包(重试3次)
            return redPacketObtainService.obtainRedPacket(redPacket.getId(), scoreDto, user, platform, 3);
        } else {
            // 记录失败
            obtainFailedHistoryService.addOne(user.getId(), redPacket.getId(), scoreDto.getCommandUrl(), scoreDto.getScore());
            return new CommandResultDTO(RedPacketObtainStatesEnum.LESS40.getType());// 评分少于40
        }
    }

    @Override
    public CommandResultDTO markScore(MarkScorePublicDTO markScorePublicDTO, Integer platform) {
        RedPacket redPacket = redPacketMapper.queryOne(markScorePublicDTO.getRpId());
        if (null == redPacket) {
            throw new BusinessException("can not get red packet by id ,id:[" + markScorePublicDTO.getRpId() + "]");
        }

        // 红包状态检查
        RedPacketObtainStatesEnum rpObtainState = redPacketStateService.rpStatesCheck(redPacket, markScorePublicDTO.getUserId());
        if (rpObtainState != null) {
            return new CommandResultDTO(rpObtainState.getType());
        }

        // 打分grade
        ScoreDto scoreDto = commandGradeService.grade(redPacket.getContent(), markScorePublicDTO.getMedieId());

        // 获取用户信息
        User user = userIntegrationService.selectByPrimaryKey(markScorePublicDTO.getUserId());

        if (null == user) {
            throw new BusinessException("can not get user by id ,id:[" + markScorePublicDTO.getUserId() + "]");
        }

        logger.info("站外口令评分成功：用户:[{}],得分:[{}],匹配度:[{}],称赞语比例:[{}],语音时长：[{}]",
                user.getNickname(),
                scoreDto.getScore(),
                scoreDto.getMatchScale(),
                scoreDto.getUserScale(),
                scoreDto.getLength());

        // 打分状态检查
        if (scoreDto.getScore() > 40) {
            // 成功：领取红包(重试3次)
            return redPacketObtainService.publicObtainRedPacket(redPacket.getId(),
                    redPacket.getPacketType(),
                    markScorePublicDTO.getCity(),
                    markScorePublicDTO.getDistrict(),
                    scoreDto, user, platform, 3);
        } else {
            // 失败：记录失败
            obtainFailedHistoryService.addOne(user.getId(), redPacket.getId(), scoreDto.getCommandUrl(), scoreDto.getScore());
            return new CommandResultDTO(RedPacketObtainStatesEnum.LESS40.getType());// 评分少于40
        }
    }

    private PacketHistoryDTO covertPacketHistoryDTO(RedPacketObtainHistory obtainHistory) {
        PacketHistoryDTO history = new PacketHistoryDTO();
        BeanUtils.copyProperties(obtainHistory, history);
        history.setObtainTime(obtainHistory.getCreateTime().getTime());
        history.setObtainAmount(AmountUtil.halfUpScale2StringValue(obtainHistory.getObtainAmount()));
        return history;
    }

    /**
     * 判断是否是免审核红包。免审核红包:使用了口令模板且没有图片和文字内容。
     * @param redPacket 红包信息
     * @return 免审核红包返回true, 否则返回false
     */
    private boolean isUnapprovalRedPacket(RedPacket redPacket) {

        List promotionImgList = new ArrayList<>();

        if (StringUtils.isNotBlank(redPacket.getPromotionImg())) {
            promotionImgList = JSON.parseObject(redPacket.getPromotionImg(), List.class);
        }

        return null != redPacket.getTwisterId() && 0L != redPacket.getTwisterId()
                && CollectionUtils.isEmpty(promotionImgList) && StringUtils.isBlank(redPacket.getPromotionContent());
    }
}
