package com.bxm.localnews.activity.vote.impl;

import com.bxm.localnews.activity.domain.VoteChoiceCountMapper;
import com.bxm.localnews.activity.domain.VoteChoiceRecordMapper;
import com.bxm.localnews.activity.domain.VoteMapper;
import com.bxm.localnews.activity.domain.VoteOptionsMapper;
import com.bxm.localnews.activity.dto.VoteDetailDTO;
import com.bxm.localnews.activity.dto.VoteOptionDTO;
import com.bxm.localnews.activity.param.VoteParam;
import com.bxm.localnews.activity.param.VotePinParam;
import com.bxm.localnews.activity.vo.VoteBean;
import com.bxm.localnews.activity.vo.VoteChoiceCountBean;
import com.bxm.localnews.activity.vo.VoteOptionsBean;
import com.bxm.localnews.activity.vote.VoteService;
import com.bxm.localnews.activity.vote.strategy.IVoteStrategy;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.Message;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.ArrayUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static com.bxm.localnews.common.constant.RedisConfig.*;

@Service
public class VoteServiceImpl extends BaseService implements VoteService {

    private final VoteMapper voteMapper;

    private final VoteOptionsMapper voteOptionsMapper;

    private final RedisStringAdapter redisStringAdapter;

    private final RedisHashMapAdapter redisHashMapAdapter;

    private final VoteChoiceRecordMapper voteChoiceRecordMapper;

    private final VoteChoiceCountMapper voteChoiceCountMapper;

    private Map<String, IVoteStrategy> voteStrategyMap;

    /**
     * 正式数据的缓存时间
     */
    private static final long EXPIRE_SECOND = 60 * 60 * 24;

    /**
     * 临时数据的过期时间
     */
    private static final long TEMP_EXPIRE_SECOND = 60 * 60;

    @Autowired
    public VoteServiceImpl(VoteMapper voteMapper,
                           VoteOptionsMapper voteOptionsMapper,
                           RedisStringAdapter redisStringAdapter,
                           RedisHashMapAdapter redisHashMapAdapter,
                           VoteChoiceRecordMapper voteChoiceRecordMapper,
                           VoteChoiceCountMapper voteChoiceCountMapper) {
        this.voteMapper = voteMapper;
        this.voteOptionsMapper = voteOptionsMapper;
        this.redisStringAdapter = redisStringAdapter;
        this.redisHashMapAdapter = redisHashMapAdapter;
        this.voteChoiceRecordMapper = voteChoiceRecordMapper;
        this.voteChoiceCountMapper = voteChoiceCountMapper;
    }

    /**
     * 加载投票策略类
     * @param strategy 投票插件选择的策略类型
     * @return 具体的策略实现类
     */
    private IVoteStrategy getVoteStrategy(String strategy) {
        if (null == voteStrategyMap) {
            Collection<IVoteStrategy> voteStrategies = SpringContextHolder.getBeans(IVoteStrategy.class);
            voteStrategyMap = Maps.newHashMap();
            voteStrategies.forEach(item -> voteStrategyMap.put(item.name(), item));
        }

        return voteStrategyMap.get(strategy);
    }

    @Override
    public VoteDetailDTO addTime(VotePinParam param) {
        //获取投票信息
        VoteDetailDTO detail = loadCacheDetail(param.getVoteId());
        //根据对应的策略类添加次数
        if (!detail.getExpired()) {
            Message message = getVoteStrategy(detail.getVoteStrategy()).addTime(detail, param);
            if (!message.isSuccess()) {
                logger.info(message.getLastMessage());
            }
        }

        return syncAndGet(param);
    }

    @Override
    public VoteDetailDTO execVote(VoteParam param) {
        //获取投票插件信息
        VoteDetailDTO detail = loadCacheDetail(param.getVoteId());
        //根据对应的策略类扣除投票次数
        if (!detail.getExpired()) {
            Message message = getVoteStrategy(detail.getVoteStrategy()).vote(detail, param);
            if (!message.isSuccess()) {
                logger.info(message.getLastMessage());
            }
        } else {
            logger.error("投票活动已过期，仍受到投票请求：[{}]", param);
        }

        VotePinParam pinParam = new VotePinParam();
        pinParam.setVoteId(param.getVoteId());
        pinParam.setRelationId(param.getRelationId());
        pinParam.setUserId(param.getUserId());

        return syncAndGet(pinParam);
    }

    private VoteDetailDTO loadCacheDetail(Long voteId) {
        KeyGenerator key = VOTE_INFO_KEY.copy().appendKey(voteId);
        VoteDetailDTO detail = redisStringAdapter.get(key, VoteDetailDTO.class);

        // 从数据库加载缓存
        if (detail == null) {
            VoteBean vote = voteMapper.selectByPrimaryKey(voteId);
            detail = convert(vote);

            List<VoteOptionsBean> sourceOptions = voteOptionsMapper.getByVoteId(voteId);
            if (null != sourceOptions) {
                detail.setOptions(sourceOptions.stream().map(this::convert).collect(Collectors.toList()));
            }

            //存入缓存，保存一天
            redisStringAdapter.set(key, detail, EXPIRE_SECOND);
        } else if (!detail.getExpired() && DateUtils.before(detail.getEndTime(), new Date())) {
            //如果投票缓存未过期但实际已过期
            detail.setExpired(true);
            redisStringAdapter.set(key, detail, EXPIRE_SECOND);
        }

        return detail;
    }

    private VoteDetailDTO convert(VoteBean vote) {
        boolean expired = DateUtils.before(vote.getEndTime(), new Date());
        String title = vote.getTitle();
        if ("CHECKBOX".equals(vote.getOptionType())) {
            title += "（多选）";
        }

        return VoteDetailDTO.builder()
                .voteId(vote.getId())
                .optionType(vote.getOptionType())
                .layoutType(vote.getLayoutType())
                .name(title)
                .voteStrategy(vote.getVoteStrategy())
                .endTime(vote.getEndTime())
                .max(vote.getMaxPoll())
                .expired(expired)
                .build();
    }

    private VoteOptionDTO convert(VoteOptionsBean option) {
        return VoteOptionDTO.builder()
                .text(option.getText())
                .imgUrl(option.getImgUrl())
                .optionId(option.getId())
                .build();
    }

    @Override
    public VoteDetailDTO syncAndGet(VotePinParam param) {
        //获取详情
        VoteDetailDTO detail = loadCacheDetail(param.getVoteId());
        //填充选项信息和用户最后的选择信息
        fill(detail, param);
        //根据投票策略获取投票插件需要显示的文案等信息
        getVoteStrategy(detail.getVoteStrategy()).deal(detail, param);
        return detail;
    }

    /**
     * 获取用户在投票插件中的最后一次选择结果
     * 优先从缓存获取，因为缓存设置了过期时间，补充策略为数据库获取
     * @param param 投票查询条件
     * @return
     */
    private String[] loadLastChoice(VotePinParam param) {
        String userId = param.getUserId().toString();
        KeyGenerator key = LAST_VOTE_KEY.copy().appendKey(param.getVoteId()).appendKey(userId);

        String options = redisStringAdapter.get(key, String.class);

        if (null == options) {
            List<Long> choiceOptionIds = voteChoiceRecordMapper.lastChoice(param);
            String[] optionArray = new String[choiceOptionIds.size()];
            int index = 0;
            for (Long optionId : choiceOptionIds) {
                optionArray[index++] = optionId.toString();
            }

            //用户不存在选择记录
            if (choiceOptionIds.size() == 0) {
                redisStringAdapter.set(key, "", TEMP_EXPIRE_SECOND);
            } else {
                redisStringAdapter.set(key, StringUtils.join(optionArray, ","), EXPIRE_SECOND);
            }

            return optionArray;
        } else if ("".equals(options)) {
            return new String[0];
        }
        return StringUtils.split(options, ",");
    }

    /**
     * 加载投票的当前选项情况
     * @param param 查询参数
     * @return 投票插件的每一个选项的票数
     */
    private Map<String, Long> loadChoiceTotal(VotePinParam param, VoteDetailDTO detail) {
        KeyGenerator key = VOTE_OPTIONS_KEY.copy().appendKey(param.getVoteId());
        Map<String, Long> voteOptionMap = redisHashMapAdapter.entries(key, Long.class);

        if (voteOptionMap == null || voteOptionMap.size() == 0) {
            voteOptionMap = Maps.newHashMap();
            List<VoteChoiceCountBean> choiceCountList = voteChoiceCountMapper.selectByParam(param);

            //缓存未命中并且数据库中不存在数据，进行初始化
            if (choiceCountList.size() == 0) {
                for (VoteOptionDTO option : detail.getOptions()) {
                    voteOptionMap.put(option.getOptionId().toString(), 0L);
                }
            } else {
                for (VoteChoiceCountBean choiceCount : choiceCountList) {
                    voteOptionMap.put(choiceCount.getOptionId().toString(), Long.valueOf(choiceCount.getTotal()));
                }
            }

            redisHashMapAdapter.putAll(key, voteOptionMap);
            redisHashMapAdapter.expire(key, EXPIRE_SECOND);
        }

        return voteOptionMap;
    }

    /**
     * 填充选项的票数，计算占比
     * 填充用户的选择信息
     * @param detail 投票插件详情
     * @param param  查询参数
     */
    private void fill(VoteDetailDTO detail, VotePinParam param) {
        Map<String, Long> voteOptionMap = loadChoiceTotal(param, detail);
        String[] optionArray = loadLastChoice(param);

        long sum = voteOptionMap.values().stream().mapToLong((l) -> l).sum();

        //计算显示的百分比
        for (VoteOptionDTO option : detail.getOptions()) {
            String optionIdStr = option.getOptionId().toString();
            Long cacheData = voteOptionMap.get(optionIdStr);
            long total = null == cacheData ? 0L : cacheData;

            option.setTotal(total);
            option.setChecked(ArrayUtils.contains(optionArray, optionIdStr));

            if (total == 0 || sum == 0) {
                option.setPercent(0L);
            } else {
                double percent = (double) total / sum * 100;
                if (percent < 1) {
                    option.setPercent(1L);
                } else {
                    option.setPercent(Math.round(percent));
                }
            }
        }
    }
}

























