package com.bxm.localnews.admin.controller.base;


import com.bxm.localnews.admin.dto.AdvertMaterialDTO;
import com.bxm.localnews.admin.dto.AdvertPositionDTO;
import com.bxm.localnews.admin.param.AdvertMaterialParam;
import com.bxm.localnews.admin.param.AdvertPositionParam;
import com.bxm.localnews.admin.service.base.AdvertisementService;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.Json;
import com.bxm.newidea.component.vo.PageWarper;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

@Api(tags = "1-36 [管理]广告素材投放管理", description = "广告素材投放管理")
@RestController
@RequestMapping("api/admin/advertisement")
public class AdvertController {

    /**
     * 日志对象
     */
    private static final Logger LOG = LoggerFactory.getLogger(AdvertController.class);


    /**
     * 广告服务
     */
    private AdvertisementService advertisementService;


    /**
     * 构造器注入
     */
    @Autowired
    public AdvertController(AdvertisementService advertisementService) {
        this.advertisementService = advertisementService;
    }


    /**
     *
     * 新闻素材接口
     *
     */
    @ApiOperation(value = "1-36-1 广告素材停启用", notes = "广告素材停启用")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "id", required = true),
            @ApiImplicitParam(name = "status", value = "状态", required = true)
    })
    @PostMapping("/material/status")
    public Json updateAdvertStatusById(Long id, Integer status) {

        if (1 != advertisementService.updateMaterialStatus(id, status)) {
            LOG.error("更新状态失败");
            return ResultUtil.genFailedResult("更新状态失败");
        }

        return ResultUtil.genSuccessResult("更改广告素材状态成功");
    }


    @ApiOperation(value = "1-36-2 新增广告素材", notes = "新增广告素材")
    @PostMapping("/material/add")
    public Json addAdvertDetail(@RequestBody AdvertMaterialParam advertMaterialParam) {
        if (0 < advertisementService.addAdvertisementMaterial(advertMaterialParam)) {

            return ResultUtil.genSuccessResult("新增成功");
        }

        LOG.error("新增广告素材失败");
        return ResultUtil.genFailedResult("新增失败");
    }

    @ApiOperation(value = "1-36-3 编辑广告素材", notes = "编辑广告素材")
    @PostMapping("/material/edit")
    public Json editAdvertDetail(@RequestBody AdvertMaterialParam advertMaterialParam) {


        if (1 != advertisementService.modifyAdvertisementMaterial(advertMaterialParam)) {
            LOG.error("更改广告素材失败");
            return ResultUtil.genFailedResult("更改广告素材失败");
        }

        return ResultUtil.genSuccessResult("更改广告素材成功");

    }


    @ApiOperation(value = "1-36-4 获取广告素材详情", notes = "根据id获取广告详情")
    @GetMapping("/material/detail")
    @ApiImplicitParam(name = "id", value = "id", required = true)
    public Json<AdvertMaterialDTO> getAdvertById(Long id) {

        AdvertMaterialDTO advertMaterialDTO = advertisementService.getAdvertisementMaterialInfo(id);

        if (null == advertMaterialDTO) {
            return ResultUtil.genFailedResult("获取广告素材详情失败");
        }

        return ResultUtil.genSuccessResult(advertMaterialDTO);
    }


    @ApiOperation(value = "1-36-5 删除广告素材", notes = "删除广告素材")
    @PostMapping("/material/delete")
    @ApiImplicitParam(name = "id", value = "id", required = true)
    public Json deleteAdvert(Long id) {

        if (advertisementService.delAdvertisementMaterial(id) > 0) {
            return ResultUtil.genSuccessMsg("删除成功");
        }
        return ResultUtil.genFailedResult("删除失败");
    }

    @ApiOperation(value = "1-36-6 获取广告素材列表", notes = "根据输入参数获取对应的广告素材列表")
    @PostMapping("/material/list")
    public Json<PageWarper<AdvertMaterialDTO>> getAdvertList(@RequestBody AdvertMaterialParam advertMaterialParam) {

        PageWarper<AdvertMaterialDTO> result = advertisementService.queryMaterialList(advertMaterialParam);
        return ResultUtil.genSuccessResult(result);
    }




    /**
     * 以下为广告投放相关接口
     */
    @ApiOperation(value = "1-36-7 广告投放停启用", notes = "广告投放停启用")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "id", required = true),
            @ApiImplicitParam(name = "status", value = "状态", required = true)
    })
    @PostMapping("/position/status")
    public Json updateAdvertPositionStatusById(Long id, Integer status) {
        if (1 != advertisementService.updatePositionStatus(id, status)) {
            LOG.error("更新状态失败");
            return ResultUtil.genFailedResult("更新状态失败");
        }

        return ResultUtil.genSuccessResult("更改广告投放状态成功");
    }


    @ApiOperation(value = "1-36-8 新增广告投放", notes = "新增广告投放")
    @PostMapping("/position/add")
    public Json addAdvertPosition(@RequestBody AdvertPositionParam advertPositionParam) {
        try {
            if (! advertisementService.addAdvertisementPosition(advertPositionParam)) {
                return ResultUtil.genFailedResult("新增广告投放失败");
            }
            return ResultUtil.genSuccessMsg("新增广告投放成功");
        } catch (Exception e) {
            LOG.error("新增广告投放异常", e);
            return ResultUtil.genFailedResult("新增广告投放异常");
        }
    }

    @ApiOperation(value = "1-36-9 编辑广告投放信息", notes = "编辑广告投放信息")
    @PostMapping("/position/edit")
    public Json editAdvertPosition(@RequestBody AdvertPositionParam advertPositionParam) {

        if (!advertisementService.editAdvertisementPosition(advertPositionParam)) {
            return ResultUtil.genFailedResult("编辑广告投放信息失败");
        }

        return ResultUtil.genSuccessMsg("编辑广告投放信息成功");
    }


    @ApiOperation(value = "1-36-10 获取广告投放详情", notes = "根据id获取广告投放详情")
    @GetMapping("/position/detail")
    @ApiImplicitParam(name = "id", value = "id", required = true)
    public Json<AdvertPositionDTO> getAdvertPositionById(Long id) {

        try {
            AdvertPositionDTO advertPositionDTO = advertisementService.getAdvertPositionDetail(id);
            if (null == advertPositionDTO) {
                return ResultUtil.genFailedResult("获取广告投放详情为空");
            }
            return ResultUtil.genSuccessResult(advertPositionDTO);
        } catch (Exception e) {
            LOG.error("获取广告投放详情异常", e);
            return ResultUtil.genFailedResult("获取广告投放详情异常");
        }
    }

    @ApiOperation(value = "1-36-11 删除广告投放", notes = "删除广告投放")
    @PostMapping("/position/delete")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "id", required = true),
            @ApiImplicitParam(name = "positionType", value = "广告投放类型", required = true)
    })
    public Json deleteAdvertPosition(Long id, Integer positionType) {

        if (!advertisementService.delAdvertPosition(id, positionType)) {
            return ResultUtil.genFailedResult("删除失败");
        }

        return ResultUtil.genSuccessMsg("删除成功");

    }

    @ApiOperation(value = "1-36-12 获取广告投放列表", notes = "根据输入参数获取对应的广告数据")
    @PostMapping("/position/list")
    public Json<PageWarper<AdvertPositionDTO>> getAdvertPositionList(@RequestBody AdvertPositionParam advertPositionParam) {
        return ResultUtil.genSuccessResult(advertisementService.queryPositionList(advertPositionParam));
    }

    @ApiOperation(value = "1-36-13 获取素材与投放的关联关系", notes = "根据素材id获取与投放的关联关系")
    @PostMapping("/relation")
    @ApiImplicitParam(name = "materialId", value = "素材id", required = true)
    public Json getAdvertPositionList(Long materialId) {
        return ResultUtil.genSuccessResult(advertisementService.relationListByMaterialId(materialId));
    }

    @ApiOperation(value = "1-36-14 获取启用状态的所有广告素材", notes = "获取启用状态的所有广告素材")
    @PostMapping("/material/all")
    public Json getAdvertPositionList() {
        return ResultUtil.genSuccessResult(advertisementService.getAllMaterial());
    }



}
