package com.bxm.localnews.admin.controller.activity;

import com.bxm.localnews.admin.controller.CommonController;
import com.bxm.localnews.admin.param.MerchantParam;
import com.bxm.localnews.admin.param.MerchantQueryParam;
import com.bxm.localnews.admin.service.activity.MerchantService;
import com.bxm.localnews.admin.vo.MerchantBean;
import com.bxm.localnews.common.vo.Json;
import com.bxm.newidea.component.vo.PageWarper;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;

@Api(tags = "1-15 [管理]商户管理", description = "优惠活动相关的商户管理")
@RestController
@RequestMapping("api/admin/merchant")
public class MerchantController extends CommonController {

    private MerchantService merchantService;

    @Autowired
    public MerchantController(MerchantService merchantService) {
        this.merchantService = merchantService;
    }

    @ApiOperation(value = "1-15-1 分页获取商户列表")
    @GetMapping("merchants")
    public Json<PageWarper<MerchantBean>> getMerchants(MerchantQueryParam param) {
        return Json.build(merchantService.getMerchants(param));
    }

    @ApiOperation(value = "1-15-2 新增商户信息")
    @PostMapping("save")
    public Json save(@Valid MerchantParam param) {
        return Json.build(merchantService.save(param));
    }

    @ApiOperation(value = "1-15-3 编辑商户信息", notes = "如果编辑时存在相应的图片则覆盖")
    @PostMapping("modify")
    public Json modify(MerchantParam param) {
        return Json.build(merchantService.modify(param));
    }

    @ApiOperation(value = "1-15-4 根据商户ID获取商户详情")
    @ApiImplicitParam(name = "id", value = "商户ID", required = true)
    @GetMapping("{id}")
    public Json<MerchantBean> getDetail(@PathVariable long id) {
        return Json.build(merchantService.getMerchant(id));
    }

    @ApiOperation(value = "1-15-5 变更商户状态", notes = "审核商户信息")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "商户ID"),
            @ApiImplicitParam(name = "status", value = "审核状态，0：待审核，1：审核通过，2：审核退回"),
            @ApiImplicitParam(name = "rejectReason", value = "审核退回原因")
    })
    @PostMapping("{id}/{status}")
    public Json changeStatus(@PathVariable long id, @PathVariable int status, String rejectReason) {
        return Json.build(merchantService.changeStatus(id, status, rejectReason));
    }

    @ApiOperation(value = "1-15-6 根据商户ID移除商户信息")
    @ApiImplicitParam(name = "id", value = "商户ID", required = true)
    @PostMapping("remove/{id}")
    public Json remove(@PathVariable long id) {
        return Json.build(merchantService.remove(id));
    }

    @ApiOperation(value = "1-15-7 根据商户名称或商户ID检索商户信息", notes = "返回前10条符合条件的商户信息，返回信息中仅包含商户名称和ID")
    @ApiImplicitParam(name = "keyword", value = "商户ID或商户名称")
    @GetMapping("match/{keyword}")
    public Json<List<MerchantBean>> search(@PathVariable String keyword) {
        return Json.build(merchantService.getTopMatch(keyword));
    }

}
