package com.bxm.localnews.admin.service.activity.impl;

import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.bxm.localnews.admin.constant.PushMessageStatusEnum;
import com.bxm.localnews.admin.domain.AdminPushMessageMapper;
import com.bxm.localnews.admin.domain.AreaCodeMapper;
import com.bxm.localnews.admin.dto.AdminPushMessageDTO;
import com.bxm.localnews.admin.dto.AndroidStatistical;
import com.bxm.localnews.admin.dto.IosStatistical;
import com.bxm.localnews.admin.dto.PushMessageStatistical;
import com.bxm.localnews.admin.param.PushMessageParam;
import com.bxm.localnews.admin.param.PushMsgBuildParam;
import com.bxm.localnews.admin.service.PushMsgIntegService;
import com.bxm.localnews.admin.service.activity.AdminPushMessageService;
import com.bxm.localnews.admin.service.activity.strategy.PushMessageHandle;
import com.bxm.localnews.admin.service.security.AdminUserService;
import com.bxm.localnews.admin.vo.AdminPushMessage;
import com.bxm.localnews.admin.vo.PushMessageRelationUser;
import com.bxm.localnews.mq.common.constant.PlatformTypeEnum;
import com.bxm.localnews.mq.common.constant.SendTypeEunm;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import com.bxm.newidea.component.vo.PageWarper;
import com.google.common.collect.Lists;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * @author zhaoyadong 2019/1/16 16:47
 * @desc
 */
@Service
public class AdminPushMessageServiceImpl implements AdminPushMessageService {

    private AdminPushMessageMapper adminPushMessageMapper;

    private PushMsgIntegService pushMsgIntegService;

    private SequenceCreater sequenceCreater;

    private AdminUserService adminUserService;

    private AreaCodeMapper areaCodeMapper;

    private PushMessageHandle pushMessageHandle;

    @Autowired
    public AdminPushMessageServiceImpl(AdminPushMessageMapper adminPushMessageMapper,
                                       PushMsgIntegService pushMsgIntegService,
                                       SequenceCreater sequenceCreater,
                                       AdminUserService adminUserService,
                                       AreaCodeMapper areaCodeMapper,
                                       PushMessageHandle pushMessageHandle) {
        this.adminPushMessageMapper = adminPushMessageMapper;
        this.pushMsgIntegService = pushMsgIntegService;
        this.sequenceCreater = sequenceCreater;
        this.adminUserService = adminUserService;
        this.areaCodeMapper = areaCodeMapper;
        this.pushMessageHandle = pushMessageHandle;
    }

    @Override
    public PageWarper<AdminPushMessageDTO> queryPushMessageList(PushMessageParam pushMessageParam) {
        PageWarper<AdminPushMessageDTO> pageWarper = new PageWarper<>(this.adminPushMessageMapper.queryPushMessageByPageSize(pushMessageParam));
        pageWarper.setList(pageWarper.getList().stream().map(this::covertPushMessage).collect(Collectors.toList()));
        return pageWarper;
    }

    @Override
    public AdminPushMessage getDetail(Long id) {
        AdminPushMessage pushMessage = this.adminPushMessageMapper.selectByPrimaryKey(id);
        return covertPushMessage(pushMessage);
    }

    @Override
    public Message saveOrUpdatePushMessage(AdminPushMessage adminPushMessage) {
        Message message = Message.build(1);
        message = checkVaildMessage(adminPushMessage, message);
        if (!message.isSuccess()) {
            return message;
        }

        Date now = new Date();

//        if (StringUtils.isNotEmpty(adminPushMessage.getTargetUserid())) {
//            String[] userIds = adminPushMessage.getTargetUserid().split(",");
//            adminPushMessage.setUserTotal(userIds.length);
//        }

        if (adminPushMessage.getId() == null) {
            adminPushMessage.setCreateTime(now);
            adminPushMessage.setModifyTime(now);
            //消息有效期默认24小时
            if (adminPushMessage.getPeriodTime() == null) {
                adminPushMessage.setPeriodTime(24);
            }
            adminPushMessage.setStatus(PushMessageStatusEnum.STAY_CONFIRM.getType());
            addPushMessage(adminPushMessage);
        } else {
            adminPushMessage.setModifyTime(now);
            updatePushMessage(adminPushMessage);
        }

        return message;
    }

    /**
     * 推送消息
     *
     * @param adminPushMessage
     */
    private void pushMessage(AdminPushMessage adminPushMessage) {
        //立即发送
        if (SendTypeEunm.SEND_NOW.getType().equals(String.valueOf(adminPushMessage.getTiming()))) {
            adminPushMessageMapper.updatePushMessageStatus(adminPushMessage.getId(), PushMessageStatusEnum.BEING_SEND.getType());
            pushMessageHandle.pushMessage(adminPushMessage);
        }

        //定时发送
        if (SendTypeEunm.SEND_TIMING.getType().equals(String.valueOf(adminPushMessage.getTiming()))) {
            setScheduleJob(adminPushMessage);
        }
    }

    /**
     * 检测参数是否合法
     *
     * @param adminPushMessage
     */
    private Message checkVaildMessage(AdminPushMessage adminPushMessage, Message message) {
        Date now = new Date();
        if (SendTypeEunm.SEND_TIMING.getType().equals(adminPushMessage.getTiming()) &&
                adminPushMessage.getPushTime().getTime() < now.getTime()) {
            return message.setSuccess(false).setMessage("推送时间不能小于当前时间");
        }
        return message;
    }

    @Override
    public Message updateStatusPushMessage(Long id, Byte status) {
        Message message = Message.build(1);

        AdminPushMessage adminPushMessage = adminPushMessageMapper.selectByPrimaryKey(id);
        if (adminPushMessage == null) {
            return message.setSuccess(false).setMessage("该消息不存在");
        }
        if (PushMessageStatusEnum.HAS_BEEN_SEND.getType().equals(adminPushMessage.getStatus())) {
            return message.setSuccess(false).setMessage("该消息已推送");
        }

        removeScheduleJob(adminPushMessage);
        this.adminPushMessageMapper.updatePushMessageStatus(id, status);

        return message;
    }

    @Override
    public Message testPushMessage(Long id, Long userId) {
        AdminPushMessage adminPushMessage = adminPushMessageMapper.selectByPrimaryKey(id);
        pushMessageHandle.pushMessage(adminPushMessage, userId);
        return Message.build();
    }

    @Override
    public Message testPushMessage(Long id, String userIds) {
        String[] userIdData = StringUtils.split(userIds, ",");
        List<Long> userIdList = Stream.of(userIdData).map(Long::new).collect(Collectors.toList());

        AdminPushMessage adminPushMessage = adminPushMessageMapper.selectByPrimaryKey(id);
        userIdList.forEach(userId -> pushMessageHandle.pushMessage(adminPushMessage, userId));
        return Message.build();
    }

    @Override
    public Message immediatelyPushMessage(Long id) {
        Message message = Message.build(1);
        AdminPushMessage adminPushMessage = adminPushMessageMapper.selectByPrimaryKey(id);
        if (adminPushMessage == null) {
            return message.setSuccess(false).setMessage("该消息不存在");
        }
        if (PushMessageStatusEnum.HAS_BEEN_SEND.getType().equals(adminPushMessage.getStatus())) {
            return message.setSuccess(false).setMessage("该消息已推送");
        }

        Date now = new Date();
        if (SendTypeEunm.SEND_NOW.getType().equals(String.valueOf(adminPushMessage.getTiming()))) {
            adminPushMessage.setPushTime(now);
            adminPushMessage.setStatus(PushMessageStatusEnum.BEING_SEND.getType());
        }

        if (SendTypeEunm.SEND_TIMING.getType().equals(String.valueOf(adminPushMessage.getTiming()))) {
            adminPushMessage.setStatus(PushMessageStatusEnum.STAY_SEND.getType());
        }

        adminPushMessageMapper.updateByPrimaryKeySelective(adminPushMessage);

        pushMessage(adminPushMessage);

        return Message.build();
    }

    @Override
    public List<PushMessageStatistical> getStatistical(Long id) {
        List<PushMessageStatistical> pushMessageStatisticals = convertStatistical(id);

        AdminPushMessage adminPushMessage = this.adminPushMessageMapper.selectByPrimaryKey(id);
        if (!PushMessageStatusEnum.HAS_BEEN_SEND.getType().equals(adminPushMessage.getStatus())) {
            return pushMessageStatisticals;
        }

        List<PushMessageRelationUser> list = this.adminPushMessageMapper.getMessageRelation(id);

        totalPlatform(list, pushMessageStatisticals);
        total(pushMessageStatisticals);

        return pushMessageStatisticals;
    }

    /**
     * 组装推送平台信息
     *
     * @param messageId
     * @return
     */
    private List<PushMessageStatistical> convertStatistical(Long messageId) {
        List<PushMessageStatistical> pushMessageStatisticals = Lists.newArrayList();
        //组装平台数据
        PlatformTypeEnum[] platforms = PlatformTypeEnum.values();
        for (PlatformTypeEnum platform : platforms) {
            PushMessageStatistical pushMessageStatistical = new PushMessageStatistical();
            pushMessageStatistical.setId(messageId);
            pushMessageStatistical.setPushPlatform(platform.name());
            pushMessageStatistical.setOpenRate(0d);
            pushMessageStatistical.setUserTotal(0L);
            pushMessageStatistical.setPushSuccess(0L);
            pushMessageStatistical.setOpenTotal(0L);
            pushMessageStatisticals.add(pushMessageStatistical);
        }
        return pushMessageStatisticals;
    }

    /**
     * 分开统计平台数据
     *
     * @param list
     * @param pushMessageStatisticals
     */
    private void totalPlatform(List<PushMessageRelationUser> list, List<PushMessageStatistical> pushMessageStatisticals) {
        if (CollectionUtils.isNotEmpty(list)) {

            //个推安卓
            AndroidStatistical androidStatistical = new AndroidStatistical("GETUI_ANDROID");
            //个推ios
            IosStatistical iosStatistical = new IosStatistical("GETUI_IOS");

            //友盟安卓
            AndroidStatistical umengAndroidStatistical = new AndroidStatistical("UMENG_ANDROID");
            //友盟ios
            IosStatistical umengIosStatistical = new IosStatistical("UMENG_IOS");

            for (PushMessageStatistical pushMessageStatistical : pushMessageStatisticals) {
                for (PushMessageRelationUser pushMessageRelationUser : list) {
                    if (pushMessageRelationUser.getPlatform().equals(pushMessageStatistical.getPushPlatform())) {
                        if (pushMessageRelationUser.getChecked() == 1) {
                            pushMessageStatistical.setOpenTotal(pushMessageStatistical.getOpenTotal() + 1);
                        }

                        if (PlatformTypeEnum.GETUI.name().equals(pushMessageRelationUser.getPlatform())) {
                            if (pushMessageRelationUser.getPushResult().contains("ok")) {
                                pushMessageStatistical.setPushSuccess(pushMessageStatistical.getPushSuccess() + 1);
                            }
                            if (pushMessageRelationUser.getRegisterClient() == 1) {
                                androidStatistical.setUserTotal(androidStatistical.getUserTotal() + 1);
                                if (pushMessageRelationUser.getChecked() == 1) {
                                    androidStatistical.setOpenTotal(androidStatistical.getOpenTotal() + 1);
                                }
                                if (pushMessageRelationUser.getPushResult().contains("ok")) {
                                    androidStatistical.setPushSuccess(androidStatistical.getPushSuccess() + 1);
                                }
                            }
                            if (pushMessageRelationUser.getRegisterClient() == 2) {
                                iosStatistical.setUserTotal(iosStatistical.getUserTotal() + 1);
                                if (pushMessageRelationUser.getChecked() == 1) {
                                    iosStatistical.setOpenTotal(iosStatistical.getOpenTotal() + 1);
                                }
                                if (pushMessageRelationUser.getPushResult().contains("ok")) {
                                    iosStatistical.setPushSuccess(iosStatistical.getPushSuccess() + 1);
                                }
                            }
                        }

                        if (PlatformTypeEnum.HUAWEI.name().equals(pushMessageRelationUser.getPlatform())) {
                            if (pushMessageRelationUser.getPushResult().contains("Success")) {
                                pushMessageStatistical.setPushSuccess(pushMessageStatistical.getPushSuccess() + 1);
                            }
                        }

                        if (PlatformTypeEnum.UMENG.name().equals(pushMessageRelationUser.getPlatform())) {
                            if (pushMessageRelationUser.getPushResult().contains("SUCCESS")) {
                                pushMessageStatistical.setPushSuccess(pushMessageStatistical.getPushSuccess() + 1);
                            }

                            if (pushMessageRelationUser.getRegisterClient() == 1) {
                                umengAndroidStatistical.setUserTotal(umengAndroidStatistical.getUserTotal() + 1);
                                if (pushMessageRelationUser.getChecked() == 1) {
                                    umengAndroidStatistical.setOpenTotal(umengAndroidStatistical.getOpenTotal() + 1);
                                }
                                if (pushMessageRelationUser.getPushResult().contains("SUCCESS")) {
                                    umengAndroidStatistical.setPushSuccess(umengAndroidStatistical.getPushSuccess() + 1);
                                }
                            }

                            if (pushMessageRelationUser.getRegisterClient() == 2) {
                                umengIosStatistical.setUserTotal(umengIosStatistical.getUserTotal() + 1);
                                if (pushMessageRelationUser.getChecked() == 1) {
                                    umengIosStatistical.setOpenTotal(umengIosStatistical.getOpenTotal() + 1);
                                }
                                if (pushMessageRelationUser.getPushResult().contains("SUCCESS")) {
                                    umengIosStatistical.setPushSuccess(umengIosStatistical.getPushSuccess() + 1);
                                }
                            }
                        }

                        pushMessageStatistical.setUserTotal(pushMessageStatistical.getUserTotal() + 1);
                    }
                }
            }

            //添加个推和友盟的对应客户端统计数据
            pushMessageStatisticals.add(androidStatistical);
            pushMessageStatisticals.add(iosStatistical);
            pushMessageStatisticals.add(umengAndroidStatistical);
            pushMessageStatisticals.add(umengIosStatistical);
        }

        pushMessageStatisticals.removeIf(pushMessageStatistical ->
                PlatformTypeEnum.GETUI.name().equals(pushMessageStatistical.getPushPlatform()) ||
                        PlatformTypeEnum.UMENG.name().equals(pushMessageStatistical.getPushPlatform()));

        pushMessageStatisticals.forEach(pushMessageStatistical -> {
                    if (pushMessageStatistical.getUserTotal() == 0) {
                        pushMessageStatistical.setOpenRate(0d);
                    } else {
                        pushMessageStatistical.setOpenRate((double) pushMessageStatistical.getOpenTotal() / pushMessageStatistical.getUserTotal());
                    }
                }
        );
    }

    /**
     * 合计
     *
     * @param pushMessageStatisticals
     */
    private void total(List<PushMessageStatistical> pushMessageStatisticals) {
        PushMessageStatistical pushMessageStatistical = new PushMessageStatistical();
        pushMessageStatistical.setId(pushMessageStatisticals.get(0).getId());
        pushMessageStatistical.setPushPlatform("ALL");
        Long userTotal = 0L, openTotal = 0L;
        for (PushMessageStatistical pms : pushMessageStatisticals) {
            userTotal = userTotal + pms.getUserTotal();
            openTotal = openTotal + pms.getOpenTotal();
        }
        pushMessageStatistical.setUserTotal(userTotal);
        pushMessageStatistical.setOpenTotal(openTotal);
        if (userTotal == 0) {
            pushMessageStatistical.setOpenRate(0d);
        } else {
            pushMessageStatistical.setOpenRate((double) openTotal / userTotal);
        }
        pushMessageStatisticals.add(pushMessageStatistical);
    }

    /**
     * 更新推送消息
     *
     * @param adminPushMessage
     * @return
     */
    private int updatePushMessage(AdminPushMessage adminPushMessage) {
        AdminPushMessage oldAdminPushMessage = this.adminPushMessageMapper.selectByPrimaryKey(adminPushMessage.getId());
        if (oldAdminPushMessage == null) {
            return 0;
        }

        //定时发送修改为立即发送
        if (SendTypeEunm.SEND_NOW.getType().equals(String.valueOf(adminPushMessage.getTiming()))) {
            removeScheduleJob(adminPushMessage);
        }

//        if (SendTypeEunm.SEND_TIMING.getType().equals(String.valueOf(adminPushMessage.getTiming()))) {
//            if (oldAdminPushMessage.getPushTime() != null &&
//                    adminPushMessage.getPushTime().getTime() != oldAdminPushMessage.getPushTime().getTime()) {
//            }
//            setScheduleJob(adminPushMessage);
//        }
        return this.adminPushMessageMapper.updateByPrimaryKeySelective(adminPushMessage);
    }

    /**
     * 添加推送消息
     *
     * @param adminPushMessage
     * @return
     */
    private int addPushMessage(AdminPushMessage adminPushMessage) {
        Long id = sequenceCreater.nextLongId();
        adminPushMessage.setId(id);
        return this.adminPushMessageMapper.insertSelective(adminPushMessage);
    }

    /**
     * 设置推送相关的定时调度任务，负责在推送消息开启变更状态，发送给用户
     * 设置任务前，先移除旧的定时任务
     *
     * @param entity 消息实体
     */
    private void setScheduleJob(AdminPushMessage entity) {
        removeScheduleJob(entity);

        PushMsgBuildParam param = new PushMsgBuildParam();
        param.setMessageId(entity.getId());
        param.setPushMessage(pushMessageHandle.generateMessage(entity));
        param.setStartTime(entity.getPushTime());

        pushMsgIntegService.timingPush(param);
    }

    /**
     * 移除定时任务
     *
     * @param entity 推送信息详情
     */
    private void removeScheduleJob(AdminPushMessage entity) {
        pushMsgIntegService.removeTimingPush(entity.getId());
    }

    private AdminPushMessageDTO covertPushMessage(AdminPushMessage adminPushMessage) {
        AdminPushMessageDTO pushMessage = new AdminPushMessageDTO();
        BeanUtils.copyProperties(adminPushMessage, pushMessage);
        pushMessage.setCreateUser(this.adminUserService.selectById(pushMessage.getCreateUserId()).getName());
        if (StringUtils.isNotEmpty(pushMessage.getAreaScope())) {
            pushMessage.setAreaCodeList(this.areaCodeMapper.findListByCode(pushMessage.getAreaScope().split(",")));
        }
        return pushMessage;
    }
}
