package com.bxm.localnews.admin.service.activity.impl;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

import com.bxm.localnews.admin.constant.InviteRecordStateEnum;
import com.bxm.localnews.admin.constant.InviteRecordStatusEnum;
import com.bxm.localnews.admin.constant.WithdrawEnum;
import com.bxm.localnews.admin.domain.InviteRecordMapper;
import com.bxm.localnews.admin.domain.UserAccountMapper;
import com.bxm.localnews.admin.domain.UserMapper;
import com.bxm.localnews.admin.domain.WithdrawMapper;
import com.bxm.localnews.admin.dto.BountyDTO;
import com.bxm.localnews.admin.dto.WithdrawFlowDTO;
import com.bxm.localnews.admin.dto.WithdrawUserInfo;
import com.bxm.localnews.admin.integration.WithdrawIntegrationService;
import com.bxm.localnews.admin.param.InviteRecordParam;
import com.bxm.localnews.admin.param.WithdrawParam;
import com.bxm.localnews.admin.service.activity.WithdrawService;
import com.bxm.localnews.admin.util.IPUtil;
import com.bxm.localnews.admin.vo.*;
import com.bxm.localnews.common.constant.InviteTypeEnum;
import com.bxm.localnews.common.vo.IP;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.PageWarper;
import com.google.common.collect.Lists;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

/**
 * @author zhaoyadong 2019/3/12 21:34
 * @desc
 */
@Service
@Slf4j
public class WithdrawServiceImpl implements WithdrawService {

    @Autowired
    private WithdrawMapper withdrawMapper;

    @Autowired
    private UserMapper userMapper;

    @Autowired
    private UserAccountMapper userAccountMapper;

    @Autowired
    private InviteRecordMapper inviteRecordMapper;

    @Autowired
    private IPUtil ipUtil;

    @Autowired
    private WithdrawIntegrationService withdrawIntegrationService;

    @Override
    public PageWarper<WithdrawFlowDTO> getWithdrawList(WithdrawParam withdrawParam) {
        Date now = new Date();
        withdrawParam.setAbortTime(DateUtils.formatAtWill(now, DateUtils.DATE_FORMAT));
        PageWarper<WithdrawFlowDTO> pageWarper = new PageWarper<>(this.withdrawMapper.queryWithdrawList(withdrawParam));
        pageWarper.getList().forEach(withdrawFlowDTO -> {
            User user = userMapper.selectByUserId(withdrawFlowDTO.getUserId());
            if (user != null) {
                IP clientIp = ipUtil.find(withdrawFlowDTO.getClientIp());
                withdrawFlowDTO.setClientIp(withdrawFlowDTO.getClientIp() + "<br/>" + clientIp.getProvince() + " "
                        + clientIp.getCity());
                withdrawFlowDTO.setEquipment(user.getEquipment());
                withdrawFlowDTO.setNickname(user.getNickname());
                withdrawFlowDTO.setStateDesc(WithdrawEnum.getWithdrawEnumByState(withdrawFlowDTO.getState()).getDesc());
                UserAccount userAccount = userAccountMapper.getUserAccount(withdrawFlowDTO.getUserId());
                withdrawFlowDTO.setAccountInfo(userAccount.getWithdrawalCash().setScale(2, BigDecimal.ROUND_HALF_UP)
                        + "/" + userAccount.getTotalCash().setScale(2, BigDecimal.ROUND_HALF_UP));
            }
        });
        return pageWarper;
    }

    @Override
    public WithdrawUserInfo getWithdrawUserInfo(Long id) {
        WithdrawFlow withdrawFlow = getWithdraw(id);
        if (withdrawFlow == null) {
            log.error("当前订单的提现用户相关信息不存在");
            return null;
        }

        return generateWithdrawUserInfo(withdrawFlow.getUserId(), withdrawFlow.getPayAccount());
    }

    private WithdrawUserInfo generateWithdrawUserInfo(Long userId, String openId) {
        WithdrawUserInfo withdrawUserInfo = new WithdrawUserInfo();
        User user = userMapper.selectByUserId(userId);
        if (user == null) {
            log.error("用户[{}]不存在", userId);
            return null;
        }
        withdrawUserInfo.setUserId(userId);
        withdrawUserInfo.setNickname(user.getNickname());
        withdrawUserInfo.setPhone(user.getPhone());
        withdrawUserInfo.setOpenId(openId);
        String registerName = "";
        if (StringUtils.isNotEmpty(user.getRegisterChannel())) {
            registerName = InviteTypeEnum.valueOf(user.getRegisterChannel()).getDesc();
        }

        withdrawUserInfo.setRegisterChannel(user.getRegisteredAddress() + "" + registerName);
        if (StringUtils.isEmpty(user.getRegisteredAddress())) {
            withdrawUserInfo.setRegisterChannel("");
        }
        withdrawUserInfo.setChannel(user.getChannelName());
        withdrawUserInfo.setCreateTime(user.getCreateTime());
        withdrawUserInfo.setEquipment(user.getEquipment());
        IP regIp = ipUtil.find(user.getRegIp());
        withdrawUserInfo.setRegIp(user.getRegIp() + " " + regIp.getProvince() + " " + regIp.getCity());
        withdrawUserInfo.setAreaName(userMapper.selectLocationByUserId(userId));
        UserLoginHistory userLoginHistory = userMapper.selectHistoryByUserId(userId);
        if (userLoginHistory != null) {
            IP historyIp = ipUtil.find(userLoginHistory.getLastLoginIp());
            withdrawUserInfo.setLastLoginIp(userLoginHistory.getLastLoginIp() + " " + historyIp.getProvince()
                    + " " + historyIp.getCity());
            withdrawUserInfo.setLastLginTime(userLoginHistory.getLastLoginTime());
            withdrawUserInfo.setLastLoginEquipment(userLoginHistory.getEquipment());
        }
        UserAccount userAccount = userAccountMapper.getUserAccount(userId);
        if (userAccount != null) {
            withdrawUserInfo.setDrawablelCash(userAccount.getDrawablelCash());
            withdrawUserInfo.setTotalCash(userAccount.getTotalCash());
            withdrawUserInfo.setWithdrawalCash(userAccount.getWithdrawalCash());
        }

        List<InviteRecord> inviteRecords = getUserAllInviteRecordByCash(userId);
        BountyDTO bountyDTO = generateBounty(inviteRecords);
        withdrawUserInfo.setAvailableCash(bountyDTO.getAvailableCash());
        withdrawUserInfo.setDiscardCash(bountyDTO.getDiscardCash());
        withdrawUserInfo.setSuccessInviteNum(inviteRecords.size());
        return withdrawUserInfo;
    }

    @Override
    public void userPassWithdrawal(Long id) {
        WithdrawFlow withdrawFlow = getWithdraw(id);
        if (withdrawFlow == null) {
            log.error("id为[{}]的提现订单不存在", id);
            return;
        }

        if (WithdrawEnum.SUCCESS_PAYMENT.getState().equals(withdrawFlow.getState()) ||
                WithdrawEnum.FAIL_PAYMENT.getState().equals(withdrawFlow.getState())
                || WithdrawEnum.DURING_PAYMENT.getState().equals(withdrawFlow.getState())) {
            log.error("id为[{}]的提现订单已进行处理，状态为[{}]，后台不予处理", id, withdrawFlow.getState());
            return;
        }

        log.info("id为[{}]的提现订单状态为[{}]，当前操作直接提现到微信", id, withdrawFlow.getState());

        //更新提现信息为支付中
        withdrawFlow.setState(WithdrawEnum.DURING_PAYMENT.getState());
        withdrawMapper.updateWithdrawFlow(withdrawFlow);

        //运营后台暂时只支持app提现审核
        withdrawFlow.setWithdrawType((byte) 1);
        withdrawIntegrationService.userWithdraw(withdrawFlow);
    }

    @Override
    public void userCheckWithdrawal(Long id) {
        WithdrawFlow withdrawFlow = getWithdraw(id);
        if (withdrawFlow == null) {
            return;
        }

        withdrawFlow.setState(WithdrawEnum.DELAY_AUDIT.getState());
        withdrawMapper.updateWithdrawFlow(withdrawFlow);
    }

    private WithdrawFlow getWithdraw(Long id) {
        return withdrawMapper.getWithdraw(id);
    }

    /**
     * 获取用户所有现金奖励类型的数据
     *
     * @return
     */
    private List<InviteRecord> getUserAllInviteRecordByCash(Long userId) {
        InviteRecordParam inviteRecordParam = new InviteRecordParam();
        inviteRecordParam.setUserId(userId);
        inviteRecordParam.setAwardType("CASH");
        inviteRecordParam.setAward(BigDecimal.ZERO);
        inviteRecordParam.setStatus((byte) 1);
        return inviteRecordMapper.queryUserInviteRecord(inviteRecordParam);
    }

    /**
     * 生成赏金信息
     *
     * @param inviteRecords
     * @return
     */
    private BountyDTO generateBounty(List<InviteRecord> inviteRecords) {
        BountyDTO bountyDTO = new BountyDTO();
        //筛选有效和无效的赏金信息
        if (CollectionUtils.isEmpty(inviteRecords)) {
            return bountyDTO;
        }

        //有效赏金
        List<InviteRecord> validList = filterValid(inviteRecords);

        if (!CollectionUtils.isEmpty(validList)) {
            List<InviteRecord> availableCashList = Lists.newArrayList();
            BigDecimal availableCash = BigDecimal.ZERO;
            for (InviteRecord inviteRecord : validList) {
                if (InviteRecordStateEnum.ACCEPT_INVITE.getName().equals(inviteRecord.getInviteState())) {
                    inviteRecord.setInviteState(InviteRecordStateEnum.ACCEPT_INVITE.getDesc());
                    availableCashList.add(inviteRecord);
                }
                availableCash = availableCash.add(inviteRecord.getAward());
            }
            for (InviteRecord inviteRecord : validList) {
                if (InviteRecordStateEnum.LOGIN_APP.getName().equals(inviteRecord.getInviteState())) {
                    inviteRecord.setInviteState(InviteRecordStateEnum.LOGIN_APP.getDesc());
                    availableCashList.add(inviteRecord);
                }
            }
            bountyDTO.setAvailableCash(availableCash);
        }

        //失效赏金
        List<InviteRecord> invalidList = filterInValid(inviteRecords);
        if (!CollectionUtils.isEmpty(invalidList)) {
            List<InviteRecord> discardCashList = Lists.newArrayList();
            BigDecimal discardCash = BigDecimal.ZERO;
            for (InviteRecord inviteRecord : invalidList) {
                if (InviteRecordStateEnum.OVERDUE_INVALID.getName().equals(inviteRecord.getInviteState())) {
                    inviteRecord.setInviteState(InviteRecordStateEnum.OVERDUE_INVALID.getDesc());
                }
                if (InviteRecordStateEnum.NOT_NEW_USER.getName().equals(inviteRecord.getInviteState())) {
                    inviteRecord.setInviteState(InviteRecordStateEnum.NOT_NEW_USER.getDesc());
                }
                discardCashList.add(inviteRecord);
                discardCash = discardCash.add(inviteRecord.getAward());
            }
            bountyDTO.setDiscardCash(discardCash);
        }
        return bountyDTO;
    }

    /**
     * 筛选在路上的赏金
     *
     * @param inviteRecords
     * @return
     */
    private List<InviteRecord> filterValid(List<InviteRecord> inviteRecords) {
        //有效赏金
        List<InviteRecord> validList = inviteRecords.stream().filter(inviteRecord ->
                InviteRecordStatusEnum.VALID.getStatus().equals(inviteRecord.getStatus()) &&
                        !InviteRecordStateEnum.SUCCESS_WITHDRAW.getName().equals(inviteRecord.getInviteState()))
                .collect(Collectors.toList());
        return validList;
    }

    /**
     * 筛选已废弃的赏金
     *
     * @param inviteRecords
     * @return
     */
    private List<InviteRecord> filterInValid(List<InviteRecord> inviteRecords) {
        //失效赏金
        List<InviteRecord> invalidList = inviteRecords.stream().filter(inviteRecord ->
                InviteRecordStatusEnum.INVALID.getStatus().equals(inviteRecord.getStatus()))
                .collect(Collectors.toList());
        return invalidList;
    }

    /**
     * 筛选已获得的赏金
     *
     * @param inviteRecords
     * @return
     */
    private List<InviteRecord> filterObtain(List<InviteRecord> inviteRecords) {
        List<InviteRecord> obtainList = inviteRecords.stream().filter(inviteRecord ->
                InviteRecordStatusEnum.VALID.getStatus().equals(inviteRecord.getStatus()) &&
                        InviteRecordStateEnum.SUCCESS_WITHDRAW.getName().equals(inviteRecord.getInviteState()))
                .collect(Collectors.toList());
        return obtainList;
    }
}
