package com.bxm.localnews.admin.service.activity.strategy;

import com.bxm.localnews.admin.domain.ParticipantMapper;
import com.bxm.localnews.admin.vo.ParticipantBean;
import com.bxm.localnews.admin.vo.PrivilegeBean;
import com.bxm.newidea.component.tools.RandomUtils;
import com.bxm.newidea.component.vo.Message;
import com.google.common.collect.Lists;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Component;

import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 首名必中策略
 * 判断活动信息中对第一名邀请人数的判断,如果达到要求则直接派奖
 * 如果未达到要求则随机进行派奖
 */
@Component("toponeStrategy")
@RefreshScope
public class TopOneDrawStrategyServiceImpl extends AbstractDrawStrategyService {

    private final ParticipantMapper participantMapper;

    @Autowired
    public TopOneDrawStrategyServiceImpl(ParticipantMapper participantMapper) {
        this.participantMapper = participantMapper;
    }

    @Override
    Message draw(PrivilegeBean privilege) {
        //抽奖人数
        int prizeNum = privilege.getPrizeNum();

        Set<Long> winners = new HashSet<>(prizeNum);
        List<ParticipantBean> relationWinners = Lists.newArrayList();

        //获取第一名邀请人数
        ParticipantBean topParticipant = participantMapper.getTopUser(privilege.getId());
        //获取参与人数列表
        List<ParticipantBean> participants = participantMapper.getParticipantList(privilege.getId());
        int count = participants.size();

        //如果首名用户邀请人数达到设置的邀请，则直接加入中奖名单
        if (topParticipant != null && topParticipant.getInviteCount() >= privilege.getLessInviteNum()) {
            winners.add(topParticipant.getUserId());
            participants.removeIf(entity -> Objects.equals(entity.getUserId(), topParticipant.getUserId()));
            prizeNum--;
            logger.debug("首名中奖，活动是[{}],中奖人是[{}]", privilege.getId(), topParticipant.getUserId());
        } else {
            logger.info("活动[{}]要求首名邀请人数达到[{}],实际邀请人数[{}]",
                    privilege.getId(), privilege.getLessInviteNum(), count);
        }

        //如果参与人数小于奖品数量则全部中奖
        if (count <= prizeNum) {
            logger.info("活动[{}]实际参与人数[{}],派奖数量[{}],所有参与人均中奖", privilege.getId(), count, prizeNum);
            for (ParticipantBean participant : participants) {
                winners.add(participant.getUserId());
            }
            participants = Lists.newArrayList();
        } else {
            while (prizeNum != 0) {
                //中奖人数有余数，则从无邀请人的参与人中抽取一个。如果没有数据则少发一个奖品
                if (prizeNum == 1) {
                    long singalUserId = getSignalWinner(participants);
                    if (singalUserId > 0) {
                        winners.add(singalUserId);
                        participants.removeIf(entity -> Objects.equals(singalUserId, entity.getUserId()));
                    } else {
                        logger.info("活动[{}]参与人中无剩余的单个参与人员，放弃抽取最后一位中奖用户", privilege.getId());
                    }
                    break;
                }
                //随机抽取一个中奖用户
                count = participants.size();
                int randomNum = RandomUtils.nextInt(0, count - 1);

                ParticipantBean randomWinner = participants.remove(randomNum);
                if (winners.add(randomWinner.getUserId())) {
                    prizeNum--;
                }
                //邀请该用户的用户也中奖
                if (randomWinner.getInviteUserId() != null && randomWinner.getInviteUserId() != 0) {
                    if (winners.add(randomWinner.getInviteUserId())) {
                        relationWinners.add(randomWinner);
                        //从待抽取名单中移除连带中奖的人员
                        participants.removeIf(entity -> Objects.equals(entity.getUserId(), randomWinner.getInviteUserId()));
                        prizeNum--;
                    }
                }
            }
        }

        saveWinner(privilege, winners, relationWinners, participants);
        privilege.setActualNum(winners.size());
        return Message.build(true);
    }

    /**
     * 从无邀请信息的用户中抽取一位获奖
     * @param participants 之前抽奖后剩余的用户
     * @return 中奖人员ID, 如果返回0，表示没有抽中单个用户，放弃抽取最后一位
     */
    private long getSignalWinner(List<ParticipantBean> participants) {
        List<ParticipantBean> signalUsers = participants.stream()
                .filter(entity -> entity.getInviteUserId() == null || entity.getInviteUserId() == 0)
                .collect(Collectors.toList());

        if (signalUsers.size() > 0) {
            int randomNum = RandomUtils.nextInt(0, signalUsers.size() - 1);
            return signalUsers.get(randomNum).getUserId();
        }

        return 0;
    }

}
