package com.bxm.localnews.admin.service.base.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.admin.constant.AdvertTypeEnum;
import com.bxm.localnews.admin.constant.RedisConfig;
import com.bxm.localnews.admin.domain.AdminAdvertAreaMapper;
import com.bxm.localnews.admin.domain.AdvertMaterialMapper;
import com.bxm.localnews.admin.domain.AdvertPositionMapper;
import com.bxm.localnews.admin.domain.AdvertPositionRelationMapper;
import com.bxm.localnews.admin.dto.*;
import com.bxm.localnews.admin.enums.AreaWhiteBlackKeyEnum;
import com.bxm.localnews.admin.param.AdvertMaterialParam;
import com.bxm.localnews.admin.param.AdvertPositionParam;
import com.bxm.localnews.admin.service.base.AdvertisementService;
import com.bxm.localnews.admin.vo.AdvertArea;
import com.bxm.localnews.admin.vo.AdvertMaterial;
import com.bxm.localnews.admin.vo.AdvertMaterialWithRelation;
import com.bxm.localnews.admin.vo.AdvertPosition;
import com.bxm.localnews.admin.vo.AdvertPositionRelation;
import com.bxm.localnews.common.constant.InviteTypeEnum;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.PageWarper;
import com.github.pagehelper.PageInfo;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;

@Service
public class AdvertisementServiceImpl implements AdvertisementService {
    private static final Logger LOG = LoggerFactory.getLogger(AdvertisementServiceImpl.class);

    /**
     * 广告素材数据接口层
     */
    @Resource
    private AdvertMaterialMapper advertMaterialMapper;

    /**
     * 广告投放数据接口层
     */
    @Resource
    private AdvertPositionMapper advertPositionMapper;

    /**
     * 广告素材与投放关联关系接口层
     */
    @Resource
    private AdvertPositionRelationMapper advertPositionRelationMapper;

    /**
     * 广告地区mapper
     */
    @Resource
    private AdminAdvertAreaMapper advertAreaMapper;

    @Resource
    private SequenceCreater sequenceCreater;

    /**
     * redis
     */
    @Resource
    private RedisHashMapAdapter redisHashMapAdapter;

    @Override
    public int updateMaterialStatus(Long id, Integer status) {
        AdvertMaterial advertMaterial = new AdvertMaterial();
        advertMaterial.setId(id);
        advertMaterial.setMaterialStatus(status);

        int update = advertMaterialMapper.updateByPrimaryKeySelective(advertMaterial);

        //删除广告缓存
        delAdvertCacheWithMaterial(id);
        return update;
    }

    @Override
    public int addAdvertisementMaterial(AdvertMaterialParam advertMaterialParam) {
        AdvertMaterial advertMaterial = new AdvertMaterial();
        BeanUtils.copyProperties(advertMaterialParam, advertMaterial);

        int insert = advertMaterialMapper.insertSelective(advertMaterial);

        //删除广告缓存
        delAdvertCacheWithMaterial(advertMaterialParam.getId());
        return insert;
    }

    @Override
    public int modifyAdvertisementMaterial(AdvertMaterialParam advertMaterialParam) {
        AdvertMaterial advertMaterial = new AdvertMaterial();
        BeanUtils.copyProperties(advertMaterialParam, advertMaterial);
        LOG.info("更改广告素材信息,更改的内容为:{}", JSON.toJSONString(advertMaterial));

        int update = advertMaterialMapper.updateByPrimaryKeySelective(advertMaterial);

        //删除广告缓存
        delAdvertCacheWithMaterial(advertMaterialParam.getId());
        return update;
    }


    @Override
    public AdvertMaterialDTO getAdvertisementMaterialInfo(Long id) {
        AdvertMaterial advertMaterial = advertMaterialMapper.selectByPrimaryKey(id);
        if (null == advertMaterial) {
            LOG.error("根据id:{}获取的广告素材详情为空");
            return null;
        }

        //参数装换
        AdvertMaterialDTO advertMaterialDTO = new AdvertMaterialDTO();
        BeanUtils.copyProperties(advertMaterial, advertMaterialDTO);

        return advertMaterialDTO;
    }

    @Override
    public int delAdvertisementMaterial(Long id) {
        int delete = advertMaterialMapper.deleteByPrimaryKey(id);
        //删除广告缓存
        delAdvertCacheWithMaterial(id);
        return delete;
    }

    @Override
    public PageWarper<AdvertMaterialDTO> queryMaterialList(AdvertMaterialParam advertMaterialParam) {
        //转换查询参数,转成vo对象
        AdvertMaterial advertMaterial = new AdvertMaterial();
        BeanUtils.copyProperties(advertMaterialParam, advertMaterial);

        //列表查询
        PageInfo<AdvertMaterial> pageInfo = new PageInfo<>(advertMaterialMapper.queryAdvertMaterialList(advertMaterial));

        //返回值转换
        List<AdvertMaterialDTO> advertMaterialResultList = new ArrayList<>();
        pageInfo.getList().forEach(advertMaterialResult -> {
            AdvertMaterialDTO advertMaterialResultDTO = new AdvertMaterialDTO();
            BeanUtils.copyProperties(advertMaterialResult, advertMaterialResultDTO);
            advertMaterialResultList.add(advertMaterialResultDTO);
        });

        //分页信息set
        PageWarper<AdvertMaterialDTO> resultInfo = new PageWarper<>(advertMaterialResultList);
        resultInfo.setPageNum(pageInfo.getPageNum());
        resultInfo.setPageSize(pageInfo.getPageSize());
        resultInfo.setTotal(pageInfo.getTotal());

        return resultInfo;
    }

    @Override
    public int updatePositionStatus(Long id, Integer status) {
        //参数转换
        AdvertPosition advertPosition = new AdvertPosition();
        advertPosition.setId(id);
        advertPosition.setEnableStatus(status);

        int update = advertPositionMapper.updateByPrimaryKeySelective(advertPosition);

        //删除缓存
        delAdvertCache(advertPositionMapper.selectByPrimaryKey(id).getPositionType());
        return update;
    }

    @Override
    public boolean addAdvertisementPosition(AdvertPositionParam advertPositionParam) {
        //广告投放id
        Long positionId = sequenceCreater.nextLongId();

        //参数转换
        AdvertPosition advertPosition = new AdvertPosition();
        BeanUtils.copyProperties(advertPositionParam, advertPosition);
        advertPosition.setId(positionId);

        //先在position表添加
        advertPositionMapper.insertSelective(advertPosition);

        //在素材和投放关联关系表中添加
        addMaterialWithPositionInfo(positionId, advertPositionParam.getMaterialList());

        //地区管理表中添加
        editAreaInfo(positionId, advertPositionParam.getAreaList());

        //删除缓存
        delAdvertCache(advertPositionParam.getPositionType());

        return true;
    }

    @Override
    public boolean editAdvertisementPosition(AdvertPositionParam advertPositionParam) {
        AdvertPosition advertPosition = new AdvertPosition();
        BeanUtils.copyProperties(advertPositionParam, advertPosition);
        //先在position表更新
        int positionAddResult = advertPositionMapper.updateByPrimaryKeySelective(advertPosition);
        if (0 > positionAddResult) {
            LOG.error("更新广告素材投放失败,更新的参数为:{}", JSON.toJSONString(advertPosition));
            return false;
        }

        //修改素材关联关系
        editMaterialWithPositionInfo(advertPositionParam.getId(), advertPositionParam.getMaterialList());

        //地区管理表中添加
        editAreaInfo(advertPositionParam.getId(), advertPositionParam.getAreaList());

        //删除缓存
        delAdvertCache(advertPositionParam.getPositionType());

        return true;
    }

    @Override
    public AdvertPositionDTO getAdvertPositionDetail(Long positionId) {
        //根据id获取position中的信息
        AdvertPosition advertPosition = advertPositionMapper.selectByPrimaryKey(positionId);
        if (null == advertPosition) {
            LOG.error("根据id:{}获取投放详情为空", positionId);
            return null;
        }
        AdvertPositionDTO resultDTO = new AdvertPositionDTO();
        BeanUtils.copyProperties(advertPosition, resultDTO);

        //获取广告素材list
        List<AdvertMaterialInfo> advertMaterialInfoList = getMaterialByPositionId(positionId);
        resultDTO.setMaterialList(advertMaterialInfoList);

        //获取地区信息
        List<AdvertAreaDTO> areaInfoList = advertAreaMapper.getAdvertAreaList(positionId);
        resultDTO.setAreaList(areaInfoList);

        return resultDTO;
    }

    @Override
    public boolean delAdvertPosition(Long positionId, Integer positionType) {
        int positionDelResult = advertPositionMapper.deleteByPrimaryKey(positionId);
        if (1 != positionDelResult) {
            LOG.error("删除position表失败,positionId为:{}", positionId);
            return false;
        }
        //关联关系表中删除
        advertPositionRelationMapper.deleteByPositionId(positionId);
        //广告地区表中删除
        advertAreaMapper.deleteByAdvertId(positionId);

        //先删除广告缓存
        if (null != positionType) {
            delAdvertCache(positionType);
        }

        return true;
    }

    @Override
    public PageWarper<AdvertPositionDTO> queryPositionList(AdvertPositionParam advertPositionParam) {
        //转换参数成entity
        AdvertPosition advertPosition = new AdvertPosition();
        BeanUtils.copyProperties(advertPositionParam, advertPosition);

        //分页查询
        PageInfo<AdvertPosition> pageInfo = new PageInfo<>(advertPositionMapper.queryAdvertPositionList(advertPosition));

        List<AdvertPositionDTO> resultList = new ArrayList<>();
        pageInfo.getList().forEach(positionQueryResult -> {
            //参数结果转换
            AdvertPositionDTO resultDTO = new AdvertPositionDTO();
            BeanUtils.copyProperties(positionQueryResult, resultDTO);

            //获取地区信息
            List<AdvertAreaDTO> areaList = advertAreaMapper.getAdvertAreaList(positionQueryResult.getId());
            resultDTO.setAreaList(areaList);

            //广告类型名称转换
            resultDTO.setPositionTypeStr(AdvertTypeEnum.getNameByCode(positionQueryResult.getPositionType()));

            resultList.add(resultDTO);

        });
        //分页信息设置
        PageWarper<AdvertPositionDTO> resultInfo = new PageWarper<>(resultList);
        resultInfo.setPageNum(pageInfo.getPageNum());
        resultInfo.setPageSize(pageInfo.getPageSize());
        resultInfo.setTotal(pageInfo.getTotal());

        return resultInfo;
    }

    @Override
    public List<Long> getRelationListByMaterialId(Long materialId) {
        return advertPositionRelationMapper.selectByMaterialId(materialId);
    }

    @Override
    public List<AdvertMaterialDTO> getAllMaterial(String key) {
        List<AdvertMaterial> queryResultList = advertMaterialMapper.queryAllMaterial(key);

        List<AdvertMaterialDTO> resultDTO = new ArrayList<>();
        queryResultList.forEach(advertMaterial -> {
            AdvertMaterialDTO advertMaterialDTO = new AdvertMaterialDTO();
            BeanUtils.copyProperties(advertMaterial, advertMaterialDTO);
            advertMaterialDTO.setMaterialName(advertMaterialDTO.getId() + "_" + advertMaterialDTO.getMaterialName());
            resultDTO.add(advertMaterialDTO);
        });
        return resultDTO;
    }

    @Override
    public List<AdvertTypeDTO> getAllAdvertType() {
        return AdvertTypeEnum.getAllType();
    }

    @Override
    public List<AdvertDeliveryObjectDTO> listAdvertDeliveryObject() {
        List<AdvertDeliveryObjectDTO> advertDeliveryObjectDtoList = new ArrayList<>();
        for (InviteTypeEnum rate : InviteTypeEnum.values()) {
            advertDeliveryObjectDtoList.add(new AdvertDeliveryObjectDTO(rate.getName(), rate.getDesc()));
        }
        return advertDeliveryObjectDtoList;
    }

    /**
     * 新增素材和投放关联关系信息
     *
     * @param advertPositionId 广告投放id
     * @param materialList     素材List
     */
    private void addMaterialWithPositionInfo(Long advertPositionId, List<AdvertMaterialInfo> materialList) {
        advertPositionRelationMapper.deleteByPrimaryKey(advertPositionId);

        if (CollectionUtils.isEmpty(materialList)) {
            LOG.error("素材列表为空,广告投放id为:{}", advertPositionId);
            return;
        }

        //循环遍历素材列表,添加投放和素材的一对多关联关系表
        for (AdvertMaterialInfo advertMaterialInfo : materialList) {
            //逐条添加素材和投放的关联关系表
            AdvertPositionRelation advertPositionRelation = new AdvertPositionRelation();
            //参数组装
            advertPositionRelation.setPositionId(advertPositionId);
            advertPositionRelation.setMaterialId(advertMaterialInfo.getId());
            advertPositionRelation.setMaterialOrder(advertMaterialInfo.getAdvertOrder());

            //添加素材和投放关系关系
            advertPositionRelationMapper.insertSelective(advertPositionRelation);
        }
    }

    /**
     * 编辑素材和投放关联关系信息
     *
     * @param advertPositionId 广告投放id
     * @param materialList     素材List
     */
    private void editMaterialWithPositionInfo(Long advertPositionId, List<AdvertMaterialInfo> materialList) {
        // 删除原数据表中的关联关系
        advertPositionRelationMapper.deleteByPositionId(advertPositionId);

        for (AdvertMaterialInfo advertMaterialInfo : materialList) {
            //逐条添加素材和投放的关联关系表
            AdvertPositionRelation advertPositionRelation = new AdvertPositionRelation();
            //参数组装
            advertPositionRelation.setPositionId(advertPositionId);
            advertPositionRelation.setMaterialOrder(advertMaterialInfo.getAdvertOrder());
            advertPositionRelation.setMaterialId(advertMaterialInfo.getId());

            //添加素材和投放关系关系
            advertPositionRelationMapper.insertSelective(advertPositionRelation);
        }
    }

    /**
     * 编辑地区信息
     *
     * @param positionId        广告投放id
     * @param advertAreaDTOList 广告地区信息List
     */
    private void editAreaInfo(Long positionId, List<AdvertAreaDTO> advertAreaDTOList) {
        // 删除原数据表中的关联关系
        advertAreaMapper.deleteByAdvertId(positionId);

        if (CollectionUtils.isEmpty(advertAreaDTOList)) {
            LOG.error("广告地区相关信息为空,对应的广告投放id为:{}", positionId);
            return;
        }

        //循环遍历地区信息列表 添加对应关系
        List<AdvertArea> advertAreaList = new ArrayList<>();
        for (AdvertAreaDTO advertAreaDTO : advertAreaDTOList) {
            AdvertArea advertAreaEntity = new AdvertArea();
            advertAreaEntity.setId(sequenceCreater.nextLongId());
            advertAreaEntity.setAdvertId(positionId);
            advertAreaEntity.setCode(advertAreaDTO.getAreaCode());
            advertAreaEntity.setAreaName(advertAreaDTO.getAreaName());
            advertAreaEntity.setAreaType(advertAreaDTO.getAreaType());

            //添加至列表中,后续批量添加
            advertAreaList.add(advertAreaEntity);
        }
        advertAreaMapper.insertAdvertAreas(advertAreaList);
    }

    /**
     * 通过广告投放id获取关联的广告素材信息
     *
     * @param positionId 广告投放id
     * @return 广告素材List, 若查询不到则返回空列表
     */
    private List<AdvertMaterialInfo> getMaterialByPositionId(Long positionId) {
        //获取广告素材list
        List<AdvertMaterialWithRelation> materialList = advertPositionRelationMapper.selectByPositionId(positionId);
        List<AdvertMaterialInfo> advertMaterialInfoList = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(materialList)) {
            //参数转换

            for (AdvertMaterialWithRelation advertMaterialWithRelation : materialList) {
                //对象转换
                AdvertMaterialInfo advertMaterialInfo = new AdvertMaterialInfo();
                advertMaterialInfo.setId(advertMaterialWithRelation.getMaterialId());
                advertMaterialInfo.setMaterialName(advertMaterialWithRelation.getMaterialName());
                advertMaterialInfo.setAdvertOrder(advertMaterialInfo.getAdvertOrder());

                //填入素材信息列表中
                advertMaterialInfoList.add(advertMaterialInfo);
            }
        }

        return advertMaterialInfoList;
    }

    /**
     * 根据素材删除广告投放的缓存
     *
     * @param materialId 广告素材id
     */
    private void delAdvertCacheWithMaterial(Long materialId) {
        //根据materialId查询关联的投放素材
        List<Long> positionIdList = advertPositionRelationMapper.selectByMaterialId(materialId);

        //遍历List,查询广告的投放类型,删除对应缓存
        positionIdList.forEach(positionId -> {
            AdvertPosition advertPosition = advertPositionMapper.selectByPrimaryKey(positionId);
            //根据投放类型删除缓存值
            delAdvertCache(advertPosition.getPositionType());
        });
    }

    /**
     * 删除广告缓存
     *
     * @param type 广告投放类型
     */
    private void delAdvertCache(Integer type) {
        LOG.info("根据投放类型删除缓存,投放类型为:{}", type);
        if (null != type) {
            redisHashMapAdapter.remove(RedisConfig.AREA_WHITE_BLACK_LIST.copy().appendKey(AreaWhiteBlackKeyEnum.ADVERT.getKey() + type));
        }
    }

}
