package com.bxm.localnews.admin.service.base.impl;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.admin.constant.RedisConfig;
import com.bxm.localnews.admin.convert.impl.VirtualUserConverter;
import com.bxm.localnews.admin.convert.impl.VirtualUserInfoConverter;
import com.bxm.localnews.admin.domain.UserMapper;
import com.bxm.localnews.admin.domain.VirtualUserMapper;
import com.bxm.localnews.admin.dto.VirtualUserOverviewDTO;
import com.bxm.localnews.admin.integration.NewsIntegrationService;
import com.bxm.localnews.admin.param.UserSyncParam;
import com.bxm.localnews.admin.param.VirtualUserInfoParam;
import com.bxm.localnews.admin.param.VirtualUserParam;
import com.bxm.localnews.admin.param.VirtualUserQueryParam;
import com.bxm.localnews.admin.service.base.VirtualUserService;
import com.bxm.localnews.admin.vo.User;
import com.bxm.localnews.admin.vo.VirtualUser;
import com.bxm.localnews.common.constant.RedisMessageChannel;
import com.bxm.localnews.common.vo.RedisMessageBean;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.PageWarper;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;

@Service
public class VirtualUserServiceImpl extends BaseService implements VirtualUserService {

    private UserMapper userMapper;

    private VirtualUserMapper virtualUserMapper;

    private VirtualUserConverter virtualUserConverter;

    private VirtualUserInfoConverter virtualUserInfoConverter;

    private RedisHashMapAdapter redisHashMapAdapter;

    private NewsIntegrationService newsIntegrationService;

    private RedisStringAdapter redisStringAdapter;

    @Autowired
    public VirtualUserServiceImpl(UserMapper userMapper,
                                  VirtualUserMapper virtualUserMapper,
                                  VirtualUserConverter virtualUserConverter,
                                  VirtualUserInfoConverter virtualUserInfoConverter,
                                  RedisHashMapAdapter redisHashMapAdapter,
                                  NewsIntegrationService newsIntegrationService,
                                  RedisStringAdapter redisStringAdapter) {
        this.userMapper = userMapper;
        this.virtualUserMapper = virtualUserMapper;
        this.virtualUserConverter = virtualUserConverter;
        this.virtualUserInfoConverter = virtualUserInfoConverter;
        this.redisHashMapAdapter = redisHashMapAdapter;
        this.newsIntegrationService = newsIntegrationService;
        this.redisStringAdapter = redisStringAdapter;
    }

    @Override
    public Long save(VirtualUserInfoParam virtualUser) {
        User user = virtualUserConverter.convert(virtualUser);
        VirtualUser virtualUserVo = virtualUserInfoConverter.convert(virtualUser);

        if (null == virtualUser.getId()) {
            //用户表新增
            userMapper.insertSelective(user);
            //虚拟用户表新增
            virtualUserVo.setId(user.getId());
            virtualUserMapper.insertSelective(virtualUserVo);
        } else {
            //更新用户表和虚拟用户表
            userMapper.updateByPrimaryKeySelective(user);
            virtualUserMapper.updateByPrimaryKeySelective(virtualUser);
            //删除缓存
            redisHashMapAdapter.remove(RedisConfig.USER_INFO, String.valueOf(virtualUser.getId()));
            //更新用户冗余信息
            UserSyncParam userSyncParam = new UserSyncParam()
                    .setHeadImg(virtualUser.getHeadImg())
                    .setId(virtualUser.getId())
                    .setNickname(virtualUser.getNickname())
                    .setSex(virtualUser.getSex());
            this.syncUser(userSyncParam);
        }
        return user.getId();
    }

    /**
     * 同步用户冗余信息
     * @param user 用户冗余信息
     */
    private void syncUser(UserSyncParam user) {
        RedisMessageBean<UserSyncParam> userSyncParamRedisMessageBean = new RedisMessageBean<>();
        userSyncParamRedisMessageBean.setEventBody(user);
        userSyncParamRedisMessageBean.setCreateTime(new Date());
        userSyncParamRedisMessageBean.setEventCode(String.valueOf(RedisMessageChannel.USER_UPDATE_EVENT.getCode()));
        userSyncParamRedisMessageBean.setEventId(String.valueOf(nextSequence()));
        userSyncParamRedisMessageBean.setEventName(String.valueOf(RedisMessageChannel.USER_UPDATE_EVENT.getDesc()));
        logger.info("更新用户冗余信息,发送redis消息:[{}]", JSONObject.toJSONString(user));
        redisStringAdapter.convertAndSend(RedisMessageChannel.USER_UPDATE_EVENT.getName(), JSONObject.toJSON(userSyncParamRedisMessageBean));
    }

    @Override
    public PageWarper<VirtualUser> listByPage(VirtualUserParam virtualUserParam) {
        return new PageWarper<>(virtualUserMapper.queryPageSize(virtualUserParam));
    }

    @Override
    public List<VirtualUserOverviewDTO> list(VirtualUserQueryParam virtualUserQueryParam) {
        VirtualUser virtualUser = new VirtualUser();
        BeanUtils.copyProperties(virtualUserQueryParam, virtualUser);
        List<VirtualUser> virtualUserList = virtualUserMapper.selectByModel(virtualUser);
        return virtualUserList.stream().map(e -> VirtualUserOverviewDTO.builder()
                .id(e.getId())
                .headImg(e.getHeadImg())
                .nickname(e.getNickname())
                //方仙女(女  单身，广德)
                .userInfo(getUserInfo(e.getNickname() + "(" + getSex(e.getSex()), e.getRemark()))
                .build()).collect(Collectors.toList());
    }

    @Override
    public VirtualUser getVirtualUserDetail(Long id) {
        VirtualUser virtualUserParam = new VirtualUser();
        virtualUserParam.setId(id);
        List<VirtualUser> virtualUser = virtualUserMapper.selectByModel(virtualUserParam);
        return CollectionUtils.isNotEmpty(virtualUser)?virtualUser.get(0):null;
    }

    @Override
    @Cacheable( value = "listVirtualUser", sync = true)
    public Map<Long,VirtualUser> list() {
        VirtualUser virtualUser = new VirtualUser();
        List<VirtualUser> virtualUserList = virtualUserMapper.selectByModel(virtualUser);
        return virtualUserList.stream().collect(Collectors.toMap(VirtualUser::getId, Function.identity(), (key1, key2) -> key2));
    }

    private String getSex(Byte sex) {
        String sexStr = "男";
        if (null != sex && 1 == sex) {
            sexStr = "男";
        } else if (null != sex && 2 == sex) {
            sexStr = "女";
        }
        return sexStr;
    }

    private String getUserInfo(String userInfo, String remark) {
        if (StringUtils.isEmpty(remark)) {
            return userInfo + ")";
        } else {
            return userInfo + " " + remark + ")";
        }
    }
}
