package com.bxm.localnews.admin.service.forum.impl;

import com.bxm.localnews.admin.constant.RedisConfig;
import com.bxm.localnews.admin.dto.AreaRelationDTO;
import com.bxm.localnews.admin.param.ForumParam;
import com.bxm.localnews.admin.param.ForumUpdateParam;
import com.bxm.localnews.admin.service.base.LocationService;
import com.bxm.localnews.admin.service.forum.ForumService;
import com.bxm.localnews.admin.vo.Forum;
import com.bxm.localnews.news.domain.AreaRelationMapper;
import com.bxm.localnews.news.domain.ForumMapper;
import com.bxm.localnews.news.domain.ForumPostMapper;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.PageWarper;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

@Service
public class ForumServiceImpl implements ForumService {

    @Resource
    private ForumMapper forumMapper;

    @Resource
    private AreaRelationMapper areaRelationMapper;

    @Resource
    private LocationService locationService;

    @Resource
    private SequenceCreater sequenceCreater;

    @Resource
    private RedisHashMapAdapter redisHashMapAdapter;

    @Resource
    private ForumPostMapper forumPostMapper;

    @Override
    public PageWarper<Forum> getList(ForumParam param) {
        PageWarper<Forum> forumList = new PageWarper<>(forumMapper.getList(param));

        if (CollectionUtils.isNotEmpty(forumList.getList())) {
            for (Forum forum : forumList.getList()) {
                completeForum(forum);
            }
        }

        return forumList;
    }

    @Override
    public Forum selectByPrimaryKey(Long id) {
        Forum forum = forumMapper.selectByPrimaryKey(id);
        completeForum(forum);
        return forum;
    }

    @Override
    public int updateStatus(Long id, Byte status) {
        Forum forum = this.forumMapper.selectByPrimaryKey(id);
        if (forum == null) {
            return 0;
        }
        forum.setEnable(status);
        this.forumMapper.updateByPrimaryKeySelective(forum);
        removeCache();

        return 1;
    }

    @Override
    public int saveOrUpdate(Forum forum) {
        int update;
        Long id = forum.getId();

        if (null == id) {
            id = sequenceCreater.nextLongId();
            forum.setId(id);
            forum.setCreateTime(new Date());
            update = this.forumMapper.insert(forum);
        } else {
            update = this.forumMapper.updateByPrimaryKeySelective(forum);
            forumPostMapper.updateForumContentByForumId(forum.getId(),forum.getContent());
        }

        areaRelationMapper.batchDelete(forum.getId(), 1);
        if (CollectionUtils.isNotEmpty(forum.getAreaCodes())) {
            List<AreaRelationDTO> list = new ArrayList<>();
            for (String areaCode : forum.getAreaCodes()) {
                AreaRelationDTO areaRelationDTO = new AreaRelationDTO();
                areaRelationDTO.setId(sequenceCreater.nextLongId());
                areaRelationDTO.setRelationId(forum.getId());
                areaRelationDTO.setAreaCode(areaCode);
                areaRelationDTO.setType((byte) 1);
                list.add(areaRelationDTO);
            }
            areaRelationMapper.batchInsert(list);
        }

        removeCache();

        return update;
    }

    @Override
    public int updateOrder(Long id, Integer order) {
        Forum forum = this.forumMapper.selectByPrimaryKey(id);
        if (forum == null) {
            return 0;
        }
        forum.setOrder(order);
        this.forumMapper.updateByPrimaryKeySelective(forum);
        removeCache();

        return 1;
    }

    @Override
    public List<Forum> getSimpleList(String key) {
        ForumParam param = new ForumParam();
        param.setTitle(key);
        param.setPageSize(0);
        return forumMapper.getList(param);
    }

    @Override
    public void updatePostForum(ForumUpdateParam forumUpdateParam) {
        forumMapper.updatePostForum(forumUpdateParam);
    }

    private void completeForum(Forum forum) {
        if (forum != null) {
            forum.setAreaCodes(areaRelationMapper.getAreaCodes(forum.getId(), 1));
            forum.setAreas(locationService.getLocationsByAreaCodes(forum.getAreaCodes()));
        }
    }

    /**
     * 获取版块列表的key
     *
     * @return
     */
    private KeyGenerator getKey() {
        return RedisConfig.FORUM.copy();
    }

    /**
     * 移除缓存
     */
    private void removeCache() {
        redisHashMapAdapter.remove(this.getKey());
    }
}
