package com.bxm.localnews.admin.service.recommend.impl;

import com.bxm.localnews.admin.config.BizConfigProperties;
import com.bxm.localnews.admin.constant.MixRecommendTypeEnum;
import com.bxm.localnews.admin.constant.PostStatusEnum;
import com.bxm.localnews.admin.domain.AreaCodeMapper;
import com.bxm.localnews.admin.dto.MixRecommandDTO;
import com.bxm.localnews.admin.enums.MixRecommendEnum;
import com.bxm.localnews.admin.integration.NewsIntegrationService;
import com.bxm.localnews.admin.param.MixPoolParam;
import com.bxm.localnews.admin.param.MixRecommandParam;
import com.bxm.localnews.admin.param.MixRecommendTopParam;
import com.bxm.localnews.admin.param.OptimizationNoticeParam;
import com.bxm.localnews.admin.service.QuartzIntegrationService;
import com.bxm.localnews.admin.service.base.LocationService;
import com.bxm.localnews.admin.service.forum.ForumPostService;
import com.bxm.localnews.admin.service.recommend.MixRecommendPoolService;
import com.bxm.localnews.admin.vo.*;
import com.bxm.localnews.news.domain.AdminNewsMapper;
import com.bxm.localnews.news.domain.ForumPostMapper;
import com.bxm.localnews.news.domain.MixedRecommendPoolMapper;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.vo.Message;
import com.bxm.newidea.component.vo.PageWarper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;
import java.util.Objects;

@Service
@Slf4j
public class MixRecommendPoolServiceImpl implements MixRecommendPoolService {

    private MixedRecommendPoolMapper mixedRecommendPoolMapper;
    private AdminNewsMapper adminNewsMapper;
    private AreaCodeMapper areaCodeMapper;
    private ForumPostMapper forumPostMapper;
    private NewsIntegrationService newsIntegrationService;
    private BizConfigProperties bizConfigProperties;
    private LocationService locationService;

    @Autowired
    public MixRecommendPoolServiceImpl(MixedRecommendPoolMapper mixedRecommendPoolMapper,
                                       AdminNewsMapper adminNewsMapper,
                                       AreaCodeMapper areaCodeMapper,
                                       ForumPostMapper forumPostMapper,
                                       NewsIntegrationService newsIntegrationService,
                                       BizConfigProperties bizConfigProperties,
                                       LocationService locationService
                                      ) {
        this.mixedRecommendPoolMapper = mixedRecommendPoolMapper;
        this.adminNewsMapper = adminNewsMapper;
        this.areaCodeMapper = areaCodeMapper;
        this.forumPostMapper = forumPostMapper;
        this.newsIntegrationService = newsIntegrationService;
        this.bizConfigProperties = bizConfigProperties;
        this.locationService = locationService;
    }

    @Override
    public PageWarper<MixRecommandDTO> listPagination(MixRecommandParam mixRecommandParam) {
        PageWarper<MixRecommandDTO> pageWarper = new PageWarper<>(mixedRecommendPoolMapper.queryByPageSize(mixRecommandParam));
        List<MixRecommandDTO> list = pageWarper.getList();
        list.forEach(e -> {
            if (StringUtils.isNotBlank(e.getAreaDetail())) {
                String[] areaCodeArray = e.getAreaDetail().split(",");
                e.setAreaDetail(locationService.getAreaDetail(areaCodeArray));
            }
            if (StringUtils.equals(e.getOrigin(), MixRecommendTypeEnum.MIX_POST.getCode())) {
                e.setLinkUrl(bizConfigProperties.getH5ServerHost() + "/shareInvitationDetail.html?userId="
                        + "&id=" + e.getId());
            } else {
                String h5ServerHost = bizConfigProperties.getH5ServerHost();
                e.setLinkUrl(h5ServerHost + "/newsDetail.html?newsId=" + e.getId() + "&userId=" + "&type=1");
            }
        });
        pageWarper.setList(list);
        return pageWarper;
    }

    @Override
    public int doCancelMixRecommend(MixPoolParam mixPoolParam) {
        int result = mixedRecommendPoolMapper.deleteByPrimaryKey(mixPoolParam.getId());
        //将新闻状态改为未推荐
        if (StringUtils.equals(MixRecommendTypeEnum.MIX_LOCAL_NEWS.getCode(), mixPoolParam.getOrigin())
                || StringUtils.equals(MixRecommendTypeEnum.MIX_NATION_NEWS.getCode(), mixPoolParam.getOrigin())) {
            News news = new News();
            news.setId(mixPoolParam.getId());
            news.setIsRecommend((byte) 0);
            adminNewsMapper.updateByPrimaryKeySelective(news);
        } else {
            ForumPost forumPost = new ForumPost();
            forumPost.setIsRecommend((byte) 0);
            forumPost.setId(mixPoolParam.getId());
            forumPostMapper.updateRecommendById(forumPost);
            //推荐库删除帖子时取消消息推送
            newsIntegrationService.removeOptimizationNotice(mixPoolParam.getId());
        }

        return result;
    }

    @Override
    public int updateMixRecommendWeight(MixPoolParam mixPoolParam, Integer weight) {
        MixedRecommendPool pool = new MixedRecommendPool();
        pool.setWeight(weight);
        pool.setId(mixPoolParam.getId());
        return mixedRecommendPoolMapper.updateByPrimaryKeySelective(pool);
    }

    @Override
    public int updateMixRecommend(MixedRecommendPool mixedRecommendPool) {
        mixedRecommendPool.setModifyTime(new Date());
        return mixedRecommendPoolMapper.updateByPrimaryKey(mixedRecommendPool);
    }

    @Override
    public int addTop(MixRecommendTopParam mixPoolParam) {
        int result = 0;
        if (StringUtils.isNotBlank(mixPoolParam.getNewsId())) {
            String[] newsIds = mixPoolParam.getNewsId().split(",");
            for (String e : newsIds) {
                Long newsId = NumberUtils.toLong(e);
                MixedRecommendPool mixedRecommendPool = new MixedRecommendPool();
                mixedRecommendPool.setId(newsId);
                result += mixedRecommendPoolMapper.updateByPrimaryKeySelective(mixedRecommendPool);
            }
        }
        if (StringUtils.isNotBlank(mixPoolParam.getPostId())) {
            String[] postIds = mixPoolParam.getPostId().split(",");
            for (String e : postIds) {
                Long postId = NumberUtils.toLong(e);
                MixedRecommendPool mixedRecommendPool = new MixedRecommendPool();
                mixedRecommendPool.setId(postId);
                result += mixedRecommendPoolMapper.updateByPrimaryKeySelective(mixedRecommendPool);
            }
        }

        return result;
    }

    @Override
    public int addNewsToRecommendPoolAuto(News news) {
        MixedRecommendPool mixedRecommendPool = mixedRecommendPoolMapper.selectByPrimaryKey(news.getId());
        int result = 0;
        if (Objects.isNull(mixedRecommendPool) && news.getTop() == 1) {
            mixedRecommendPool = new MixedRecommendPool();
            mixedRecommendPool.setOrigin(MixRecommendEnum.LOCALNEWS_ORIGIN.getOrigin());
            Date currentTime = new Date();
            mixedRecommendPool.setAddTime(currentTime);
            mixedRecommendPool.setModifyTime(currentTime);
            mixedRecommendPool.setWeight(-1);
            mixedRecommendPool.setStatus((byte) 1);
            mixedRecommendPool.setTitle(news.getTitle());
            mixedRecommendPool.setAreaDetail(news.getAreaDetail());
            mixedRecommendPool.setIssueTime(news.getIssueTime());
            mixedRecommendPool.setAuto((byte) 0);
            mixedRecommendPool.setId(news.getId());
            mixedRecommendPool.setPublishTime(news.getPublishTime());
            mixedRecommendPool.setOptimizationNotice((byte) 0);
            mixedRecommendPool.setRecommendTime(currentTime);
            mixedRecommendPool.setAuthor(news.getAuthor());
            mixedRecommendPool.setDeliveryType(news.getDeliveryType());
            result = mixedRecommendPoolMapper.insertSelective(mixedRecommendPool);
        }
        return result;
    }

    @Override
    public int addPostToRecommendPoolAuto(ForumPost forumPost) {
        MixedRecommendPool mixedRecommendPool = mixedRecommendPoolMapper.selectByPrimaryKey(forumPost.getId());
        int result = 0;
        if (Objects.isNull(mixedRecommendPool)) {
            mixedRecommendPool = new MixedRecommendPool();
            mixedRecommendPool.setOrigin(MixRecommendEnum.LOCALNEWS_ORIGIN.getOrigin());
            Date currentTime = new Date();
            mixedRecommendPool.setAddTime(currentTime);
            mixedRecommendPool.setModifyTime(currentTime);
            mixedRecommendPool.setWeight(-1);
            mixedRecommendPool.setStatus((byte) 1);
            mixedRecommendPool.setTitle(SpringContextHolder.getBean(ForumPostService.class).getForumPostTitleTotally(forumPost.getTitle(), forumPost.getTextField()));
            mixedRecommendPool.setAreaDetail(forumPost.getAreaCode());
            mixedRecommendPool.setIssueTime(forumPost.getDisplayTime());
            mixedRecommendPool.setAuto((byte) 0);
            mixedRecommendPool.setId(forumPost.getId());
            mixedRecommendPool.setPublishTime(forumPost.getPublishTime());
            mixedRecommendPool.setOptimizationNotice((byte) 0);
            mixedRecommendPool.setRecommendTime(currentTime);
            mixedRecommendPool.setAuthor(forumPost.getUserName());
            mixedRecommendPool.setUserId(forumPost.getUserId());
            mixedRecommendPool.setDeliveryType(forumPost.getDeliveryType());
            result = mixedRecommendPoolMapper.insertSelective(mixedRecommendPool);
        }
        return result;
    }

    @Override
    public int doUpdateRecommendTimeById(Long id, Date recommendTime) {

        MixedRecommendPool record = new MixedRecommendPool();
        record.setId(id);
        record.setRecommendTime(recommendTime);
        int result = mixedRecommendPoolMapper.updateByPrimaryKeySelective(record);

        //如果不是帖子则退出方法
        MixedRecommendPool oldPool = mixedRecommendPoolMapper.selectByPrimaryKey(id);
        if (null == oldPool || !oldPool.getOrigin().equals(MixRecommendTypeEnum.MIX_POST.getCode())) {
            return result;
        }

        //得到旧的推荐时间
        Date oldReCommendTime = oldPool.getRecommendTime();

        //获取帖子
        ForumPost forumPost = forumPostMapper.selectByPrimaryKey(id);

        //判断修改的时间是否为帖子,并且帖子用户创建,并且原始的推荐时间大于当前时间(为推送时) 修改定时推送的定时任务
        boolean flag = Objects.nonNull(forumPost) &&
                Objects.isNull(forumPost.getCreator()) &&
                oldReCommendTime.compareTo(new Date()) > 0;

        if (flag) {
            //调用quartz 生成定时任务
            OptimizationNoticeParam param = new OptimizationNoticeParam();
            param.setId(id);
            param.setStartTime(recommendTime);
            newsIntegrationService.addOptimizationNotice(param);
        }
        return result;
    }

    @Override
    public Message addMixRecommendPool(MixPoolParam mixPoolParam) {
        if (StringUtils.isBlank(mixPoolParam.getOrigin())) {
            return Message.build(false, "类型为空,或不存在!");
        }
        String origin = mixPoolParam.getOrigin();
        MixedRecommendPool mixedRecommendPool =
                MixedRecommendPool.builder()
                        .origin(origin)
                        .id(mixPoolParam.getId())
                        .optimizationNotice(mixPoolParam.getOptimizationNotice())
                        .recommendTime(mixPoolParam.getRecommendTime()).build();

        MixedRecommendPool result = mixedRecommendPoolMapper.selectByPrimaryKey(mixPoolParam.getId());

        if (StringUtils.equals(MixRecommendTypeEnum.MIX_LOCAL_NEWS.getCode(), origin) || StringUtils.equals(MixRecommendTypeEnum.MIX_NATION_NEWS.getCode(), origin)) {
            News news = adminNewsMapper.selectByPrimaryKey(mixPoolParam.getId());

            //新闻置顶无法加入推荐库
            Message newsValidMessage = checkIsNewsValid(news);
            if (!newsValidMessage.isSuccess()) {
                return newsValidMessage;
            }

            //更新新闻是否推荐字段
            News newsUpdate = new News();
            newsUpdate.setId(mixPoolParam.getId());
            newsUpdate.setIsRecommend((byte) 1);
            adminNewsMapper.updateByPrimaryKeySelective(newsUpdate);

            //组装推荐池对象
            mixedRecommendPool.setTitle(news.getTitle());
            mixedRecommendPool.setAreaDetail(news.getAreaDetail());
            mixedRecommendPool.setDeliveryType(news.getDeliveryType());
            mixedRecommendPool.setIssueTime(news.getIssueTime());
            mixedRecommendPool.setPublishTime(news.getPublishTime());
            mixedRecommendPool.setAuthor(news.getAuthor());

            //推荐池里没有则新增，如果有则更新（会有自动入推荐库的本地新闻）
            if (Objects.isNull(result)) {
                saveMixRecommendPool(mixedRecommendPool);
            } else {
                updateMixRecommendPool(mixedRecommendPool);
            }

        } else if (StringUtils.equals(MixRecommendTypeEnum.MIX_POST.getCode(), origin)) {
            ForumPost forumPost = forumPostMapper.selectByPrimaryKey(mixPoolParam.getId());
            //帖子状态异常无法加入推荐库
            if (null == forumPost || forumPost.getStatus() != PostStatusEnum.ENABLE.getCode()) {
                return Message.build(false, "状态异常无法加入推荐库");
            }

            //更新帖子是否推荐字段
            ForumPost forumPostUpdate = new ForumPost();
            forumPostUpdate.setIsRecommend((byte) 1);
            forumPostUpdate.setId(forumPost.getId());
            forumPostMapper.updateRecommendById(forumPostUpdate);

            //组装推荐池对象
            mixedRecommendPool.setTitle(SpringContextHolder.getBean(ForumPostService.class).getForumPostTitleTotally(forumPost.getTitle(), forumPost.getTextField()));
            mixedRecommendPool.setAreaDetail(forumPost.getAreaCode());
            mixedRecommendPool.setDeliveryType(forumPost.getDeliveryType());
            mixedRecommendPool.setIssueTime(forumPost.getDisplayTime());
            mixedRecommendPool.setPublishTime(forumPost.getPublishTime());
            mixedRecommendPool.setAuthor(forumPost.getUserName());
            mixedRecommendPool.setUserId(forumPost.getUserId());

            //推荐池里没有则新增，如果有则更新（会有自动入推荐库的精华帖）
            if (Objects.isNull(result)) {
                saveMixRecommendPool(mixedRecommendPool);
            } else {
                updateMixRecommendPool(mixedRecommendPool);
            }

            //帖子为用户创建时,调用quartz生成定时推送帖子加入推荐库推送任务
            if (Objects.isNull(forumPost.getCreator())) {
                OptimizationNoticeParam param = new OptimizationNoticeParam();
                param.setId(forumPost.getId());
                param.setStartTime(mixPoolParam.getRecommendTime());
                newsIntegrationService.addOptimizationNotice(param);
            }
        }

        return Message.build(true);
    }

    private Message checkIsNewsValid(News news) {
        if (null == news) {
            return Message.build(false, "无此新闻");
        }
        if (news.getTop() == 2) {
            return Message.build(false, "置顶时无法加入推荐库");
        }
        if (!(news.getStatus() == 1 || news.getStatus() == -1)) {
            return Message.build(false, "状态异常无法加入推荐库");
        }
        return Message.build(true);
    }

    /**
     * 添加到推荐池
     *
     * @param mixedRecommendPool
     * @return
     */
    private int saveMixRecommendPool(MixedRecommendPool mixedRecommendPool) {
        Date currentTime = new Date();
        mixedRecommendPool.setAddTime(currentTime);
        mixedRecommendPool.setModifyTime(currentTime);
        mixedRecommendPool.setWeight(0);
        mixedRecommendPool.setStatus((byte) 1);
        //手动添加
        mixedRecommendPool.setAuto((byte) 1);
        return mixedRecommendPoolMapper.insertSelective(mixedRecommendPool);
    }

    /**
     * 将推荐池中中数据设置为手动推荐
     *
     * @param mixedRecommendPool
     * @return
     */
    private int updateMixRecommendPool(MixedRecommendPool mixedRecommendPool) {
        //设置为手动推荐
        Date currentTime = new Date();
        mixedRecommendPool.setAuto((byte) 1);
        mixedRecommendPool.setWeight(0);
        mixedRecommendPool.setModifyTime(currentTime);
        return mixedRecommendPoolMapper.updateByPrimaryKeySelective(mixedRecommendPool);
    }

}
