package com.bxm.localnews.admin.service.security;

import com.bxm.localnews.admin.config.SecurityConfigurationProperties;
import com.bxm.localnews.admin.vo.security.AdminUser;
import com.bxm.newidea.component.tools.DateUtils;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.SignatureAlgorithm;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.stereotype.Component;

import java.io.Serializable;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * jwt token操作工具
 */
@Component
public class JwtTokenService implements Serializable {

    private static final long serialVersionUID = -3301605591108950415L;

    private static final String CLAIM_KEY_USERNAME = "sub";

    private static final String CLAIM_KEY_CREATED = "created";

    private final SecurityConfigurationProperties properties;

    @Autowired
    public JwtTokenService(SecurityConfigurationProperties properties) {
        this.properties = properties;
    }

    /**
     * 从token的payload中获取存储的用户名
     * @param token request head中包含的token
     * @return 解析后的用户名，如果无法解析将返回Null
     */
    public String getUsernameFromToken(String token) {
        String username;
        try {
            final Claims claims = this.getClaimsFromToken(token);
            username = claims.getSubject();
        } catch (Exception e) {
            username = null;
        }
        return username;
    }

    /**
     * 从token的payload中获取存储的创建时间
     * @param token request head中包含的token
     * @return 创建时间，无法解析则返回null
     */
    private Date getCreatedDateFromToken(String token) {
        Date created;
        try {
            final Claims claims = this.getClaimsFromToken(token);
            created = new Date((Long) claims.get(CLAIM_KEY_CREATED));
        } catch (Exception e) {
            created = null;
        }
        return created;
    }

    /**
     * 获取token的过期时间
     * @param token request head中包含的token
     * @return token的过期时间
     */
    protected Date getExpirationDateFromToken(String token) {
        Date expiration;
        try {
            final Claims claims = this.getClaimsFromToken(token);
            expiration = claims.getExpiration();
        } catch (Exception e) {
            expiration = null;
        }
        return expiration;
    }

    /**
     * 获取token中包含的信息
     */
    private Claims getClaimsFromToken(String token) {
        Claims claims;
        try {
            claims = Jwts.parser().setSigningKey(properties.getJwtSecret()).parseClaimsJws(token).getBody();
        } catch (Exception e) {
            claims = null;
        }
        return claims;
    }

    /**
     * 创建token过期时间
     */
    private Date generateExpirationDate() {
        return new Date(System.currentTimeMillis() + properties.getJwtTokenExpiration() * 1000);
    }

    /**
     * 判断token是否过期
     */
    private Boolean isTokenExpired(String token) {
        final Date expiration = this.getExpirationDateFromToken(token);
        return expiration.before(new Date());
    }

    /**
     * token的创建时间是否在用户权限信息变更之后
     */
    private Boolean isCreatedAfterChangeDate(Date created, Date changeDate) {
        return (changeDate != null && created.after(changeDate));
    }

    /**
     * 通过用户信息创建token
     */
    public String generateToken(AdminUser userDetails) {
        Map<String, Object> claims = new HashMap<>();
        claims.put(CLAIM_KEY_USERNAME, userDetails.getUsername());
        claims.put(CLAIM_KEY_CREATED, new Date());
        return this.generateToken(claims);
    }

    private String generateToken(Map<String, Object> claims) {
        return Jwts.builder()
                .setClaims(claims)
                .setExpiration(this.generateExpirationDate())
                .signWith(SignatureAlgorithm.HS512, properties.getJwtSecret())
                .compact();
    }

    /**
     * token是否可被刷新,只要是变更日期之后创建的都认为可以续租
     * @param token     当前token
     * @param lastReset 最后重置时间
     * @return true表示可刷新
     */
    public Boolean canTokenBeRefreshed(String token, Date lastReset) {
        final Date created = this.getCreatedDateFromToken(token);
        // && !isTokenExpired(token);
        return this.isCreatedAfterChangeDate(created, lastReset);
    }

    /**
     * 根据颁发的refreshToken刷新accessToken
     * @param token accessToken
     * @return 刷新后的新accessToken
     */
    public String refreshToken(String token) {
        String refreshedToken;
        try {
            final Claims claims = this.getClaimsFromToken(token);
            claims.put(CLAIM_KEY_CREATED, new Date());
            refreshedToken = this.generateToken(claims);
        } catch (Exception e) {
            refreshedToken = null;
        }
        return refreshedToken;
    }

    /**
     * 即将token续租
     * @param token 原始token
     * @return 续租后的token
     */
    public String renewToken(String token) {
        Date expirationDate = this.getExpirationDateFromToken(token);
        long diffSeconeds = DateUtils.getDiffSeconed(expirationDate, new Date(), false);
        if (diffSeconeds <= properties.getJwtTokenRenew()) {
            return this.refreshToken(token);
        }
        return null;
    }

    /**
     * 验证token是否可用
     * 1.token中的用户与用户账号一致
     * 2.用户权限信息后续未变更
     */
    public Boolean validateToken(String token, UserDetails userDetails) {
        AdminUser user = (AdminUser) userDetails;
        final String username = this.getUsernameFromToken(token);
        final Date created = this.getCreatedDateFromToken(token);

        //不验证token的过期时间，根据用户的最后修改时间来判断过期
        return username.equals(user.getUsername()) && this.isCreatedAfterChangeDate(created, user.getResetTime());
    }

}