package com.bxm.localnews.admin.service.security.impl;

import com.bxm.localnews.admin.constant.RedisConfig;
import com.bxm.localnews.admin.domain.UserBlockMapper;
import com.bxm.localnews.admin.dto.UserBlockDTO;
import com.bxm.localnews.admin.enums.BlockUserTypeEnum;
import com.bxm.localnews.admin.integration.IMIntegrationService;
import com.bxm.localnews.admin.service.security.AdminBlockUserService;
import com.bxm.localnews.admin.vo.UserBlock;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;

@Service
public class AdminBlockUserServiceImpl implements AdminBlockUserService {


    @Autowired
    private UserBlockMapper userBlockMapper;

    @Autowired
    private RedisStringAdapter redisStringAdapter;

    @Autowired
    private IMIntegrationService imIntegrationService;

    @Autowired
    private SequenceCreater sequenceCreater;

    @Override
    public UserBlockDTO getInfo(Long userId) {

        List<UserBlock> userBlockList = userBlockMapper.selectByUserId(userId);
        if (CollectionUtils.isEmpty(userBlockList)) {
            return null;
        }


        return convert(userId, userBlockList);
    }

    @Override
    public Boolean editUserBlock(UserBlockDTO userBlockDTO) {
        Long userId = userBlockDTO.getUserId();
        Date blockTime = userBlockDTO.getBlockTime();
        if (userBlockDTO.getChatLimitFlag()) {
            limit(userBlockDTO, BlockUserTypeEnum.CHAT_LIMIT, getChatLimitRedisKey(userId));
            imIntegrationService.block(userId, blockTime);

        } else {
            cancelLimit(userBlockDTO, BlockUserTypeEnum.CHAT_LIMIT, getChatLimitRedisKey(userId));
            imIntegrationService.unblock(userId);
        }

        if (userBlockDTO.getBlockFlag()) {
            limit(userBlockDTO, BlockUserTypeEnum.BLOCK_LIMIT, getBlockLimitRedisKey(userId));

        } else {
            cancelLimit(userBlockDTO, BlockUserTypeEnum.BLOCK_LIMIT, getBlockLimitRedisKey(userId));
        }

        if (userBlockDTO.getRecommendLimitFlag()) {
            limit(userBlockDTO, BlockUserTypeEnum.RECOMMEND_LIMIT, getRecommendLimitRedisKey(userId));
        } else {
            cancelLimit(userBlockDTO, BlockUserTypeEnum.RECOMMEND_LIMIT, getRecommendLimitRedisKey(userId));
        }


        return null;
    }

    private UserBlockDTO convert(Long userId, List<UserBlock> userBlockList) {
        UserBlockDTO userBlockDTO = UserBlockDTO.builder()
                .userId(userId)
                .blockTime(userBlockList.get(0).getBlockTime())
                .build();
        userBlockList.forEach(userBlock -> {
            if (BlockUserTypeEnum.CHAT_LIMIT.getBlockType().equals(userBlock.getBlockType()) &&
                    StringUtils.isNotBlank(redisStringAdapter.getString(getChatLimitRedisKey(userId)))) {
                userBlockDTO.setChatLimitFlag(true);
            }
            if (BlockUserTypeEnum.BLOCK_LIMIT.getBlockType().equals(userBlock.getBlockType()) &&
                    StringUtils.isNotBlank(redisStringAdapter.getString(getBlockLimitRedisKey(userId)))) {
                userBlockDTO.setBlockFlag(true);
            }
            if (BlockUserTypeEnum.RECOMMEND_LIMIT.getBlockType().equals(userBlock.getBlockType()) &&
                    StringUtils.isNotBlank(redisStringAdapter.getString(getRecommendLimitRedisKey(userId)))) {
                userBlockDTO.setRecommendLimitFlag(true);
            }
        });

        return userBlockDTO;
    }

    private void cancelLimit(UserBlockDTO userBlockDTO, BlockUserTypeEnum blockUserTypeEnum, KeyGenerator redisKey) {

        Long userId = userBlockDTO.getUserId();
        UserBlock userBlock = new UserBlock();
        userBlock.setUserId(userId);
        userBlock.setBlockType(blockUserTypeEnum.getBlockType());
        userBlockMapper.deleteByUserIdAndType(userBlock);

        redisStringAdapter.remove(redisKey);


    }

    private void limit(UserBlockDTO userBlockDTO, BlockUserTypeEnum blockUserTypeEnum, KeyGenerator redisKey) {
        Date blockTime = userBlockDTO.getBlockTime();

        addRecord(userBlockDTO, blockUserTypeEnum.getBlockType());

        redisStringAdapter.set(redisKey, blockTime);
        redisStringAdapter.expire(redisKey, blockTime);

    }

    private void addRecord(UserBlockDTO userBlockDTO, Integer blockType) {

        UserBlock userBlock = UserBlock.builder()
                .id(sequenceCreater.nextLongId())
                .userId(userBlockDTO.getUserId())
                .blockTime(userBlockDTO.getBlockTime())
                .blockType(blockType)
                .build();

        userBlockMapper.deleteByUserIdAndType(userBlock);
        userBlockMapper.insertSelective(userBlock);
    }

    private KeyGenerator getChatLimitRedisKey(Long userId) {
        return RedisConfig.IM_BLOCK_LIST.copy().appendKey(userId);
    }

    private KeyGenerator getBlockLimitRedisKey(Long userId) {
        return RedisConfig.USER_BLOCK_LIMIT.copy().appendKey(userId);
    }

    private KeyGenerator getRecommendLimitRedisKey(Long userId) {
        return RedisConfig.RECOMMEND_BLOCK_LIMIT.copy().appendKey(userId);
    }

}
