package com.bxm.localnews.admin.convert.impl;

import com.bxm.localnews.admin.param.VirtualUserInfoParam;
import com.bxm.localnews.admin.service.base.CommonDictService;
import com.bxm.localnews.admin.service.base.LocationService;
import com.bxm.localnews.admin.vo.Location;
import com.bxm.localnews.admin.vo.User;
import com.bxm.localnews.common.constant.InfoCombineStateEnum;
import com.bxm.newidea.component.tools.BitOperatorUtil;
import com.bxm.newidea.component.tools.StringUtils;
import org.apache.commons.collections.MapUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.Map;

/**
 * 用户表实体转换
 */
@Component
public class VirtualUserConverter extends AbstractConverter<VirtualUserInfoParam, User> {

    @Autowired
    private CommonDictService commonDictService;
    @Autowired
    private LocationService locationService;

    @Override
    public User convert(VirtualUserInfoParam data) {
        User user = new User();
        user.setId(data.getId());
        user.setNickname(data.getNickname());
        user.setHeadImg(data.getHeadImg());
        user.setSex(data.getSex());
        user.setState((byte) 3);
        user.setBirthday(data.getBirthday());
        user.setAddress(data.getAddress());

        user.setJobTitle(data.getJobTitle());
        user.setCompany(data.getCompany());
        user.setJobType(data.getJobType());
        user.setJobCategory(data.getJobCategory());
        user.setHometownCode(data.getCountyCode());
        user.setPersonalProfile(data.getPersonalProfile());
        user.setUserTagIds(data.getUserTagIds());
        user.setRelationshipStatus(data.getMarry());
        user.setInfoCompleteState(initUserInfoCompleteStatus(user));
        fillUser(user);
        return user;
    }

    /**
     * 初始化用户的完善资料度
     * @param user
     * @return
     */
    private Long initUserInfoCompleteStatus(User user) {
        long infoCompleteStatus = 0L;
        //用户在登录时可能携带 性别、昵称、头像等信息，需要将这些作为完善资料度的参考
        if (null != user.getSex() && 0 != user.getSex()) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.SEX_COMPLETE.getIndex());
        }
        if (StringUtils.isNotBlank(user.getNickname())) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.NICKNAME_COMPLETE.getIndex());
        }
        if (StringUtils.isNotBlank(user.getHeadImg())) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.HEAD_IMG_COMPLETE.getIndex());
        }
        if (null != user.getBirthday()) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.BIRTHDAY_COMPLETE.getIndex());
        }
        if (null != user.getRelationshipStatus()) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.RELATION_SHIP_COMPLETE.getIndex());
        }
        if (StringUtils.isNotBlank(user.getHometownCode())) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.HOMETOWN_COMPLETE.getIndex());
        }
        if (null != user.getJobCategory()) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.JOB_COMPLETE.getIndex());
        }
        if (StringUtils.isNotBlank(user.getPersonalProfile())) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.PERSONAL_PROFILE_COMPLETE.getIndex());
        }
        if (StringUtils.isNotBlank(user.getPhone())) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.PHONE_COMPLETE.getIndex());
        }
        if (!CollectionUtils.isEmpty(user.getUserTagIds())) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.USER_TAG_COMPLETE.getIndex());
        }
        return infoCompleteStatus;
    }

    /**
     * 补充用户信息
     *
     * @param user 用户信息
     */
    private void fillUser(User user) {
        // 行业信息,从字典表t_common_dict中取
        if (null != user.getJobType() && null != user.getJobCategory()) {
            user.setJobTypeName(commonDictService.getById(user.getJobType()).getLabel());
            user.setJobCategoryName(commonDictService.getById(user.getJobCategory()).getLabel());
        }

        /*
         * 家乡数据处理，用户表的家乡保存的是区县code，和fullName
         */
        if (null != user.getHometownCode()) {
            Map<String, Location> locationMap = locationService.listLocation();
            if (MapUtils.isNotEmpty(locationMap) && null != locationMap.get(user.getHometownCode())) {
                user.setHometownName(locationMap.get(user.getHometownCode()).getFullName());
            }
        }
    }

}
