package com.bxm.localnews.admin.service.activity.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.admin.domain.RedPacketApprovalHistoryMapper;
import com.bxm.localnews.admin.domain.RedPacketMapper;
import com.bxm.localnews.admin.dto.RedPacketDTO;
import com.bxm.localnews.admin.enums.ApproveRedPacketEnum;
import com.bxm.localnews.admin.enums.RedPacketOnlineStatusEnum;
import com.bxm.localnews.admin.integration.PushMsgIntegService;
import com.bxm.localnews.admin.param.RedPacketParam;
import com.bxm.localnews.admin.service.activity.RedPacketService;
import com.bxm.localnews.admin.vo.RedPacket;
import com.bxm.localnews.admin.vo.RedPacketApprovalHistory;
import com.bxm.localnews.mq.common.constant.PushMessageEnum;
import com.bxm.localnews.mq.common.constant.TemplateTypeEnum;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.mq.common.model.dto.PushPayloadInfo;
import com.bxm.localnews.mq.common.model.dto.PushReceiveScope;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.PageWarper;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.Date;

/**
 * 红包服务实现类
 *
 * @author Dong_hy
 */
@Service
public class RedPacketServiceImpl implements RedPacketService {

    /**
     * 日志对象
     */
    private static final Logger LOG = LoggerFactory.getLogger(RedPacketServiceImpl.class);

    /**
     * 红包数据接口类
     */
    private RedPacketMapper redPacketMapper;

    /**
     * 红包审核记录 数据接口层
     */
    private RedPacketApprovalHistoryMapper redPacketApprovalHistoryMapper;

    /**
     * 序列号创建服务
     */
    private SequenceCreater sequenceCreater;

    /**
     * 消息推送服务
     */
    private PushMsgIntegService pushMsgIntegService;

    @Autowired
    public RedPacketServiceImpl(RedPacketMapper redPacketMapper,
                                RedPacketApprovalHistoryMapper redPacketApprovalHistoryMapper,
                                SequenceCreater sequenceCreater, PushMsgIntegService pushMsgIntegService) {
        this.redPacketMapper = redPacketMapper;
        this.redPacketApprovalHistoryMapper = redPacketApprovalHistoryMapper;
        this.sequenceCreater = sequenceCreater;
        this.pushMsgIntegService = pushMsgIntegService;
    }

    @Override
    public PageWarper<RedPacketDTO> queryList(RedPacketParam redPacketParam) {
        //参数转换
        RedPacket redPacketEntity = new RedPacket();
        BeanUtils.copyProperties(redPacketParam, redPacketEntity);

        PageWarper<RedPacketDTO> queryResultPage = new PageWarper<>(redPacketMapper.queryList(redPacketEntity));
        if (CollectionUtils.isEmpty(queryResultPage.getList())) {
            LOG.info("查询红包数据列表为空,查询参数为:{}", JSON.toJSONString(redPacketEntity));
            return queryResultPage;
        }

        //参数转换-数值计算
        queryResultPage.getList().forEach(redPacketDTO -> {
            //已领金额计算
            BigDecimal receivedAmount = redPacketDTO.getAmount().subtract(redPacketDTO.getSurplusAmount());
            receivedAmount.setScale(2, BigDecimal.ROUND_DOWN);

            redPacketDTO.setReceivedAmount(receivedAmount);
        });


        return new PageWarper<>(redPacketMapper.queryList(redPacketEntity));
    }

    @Override
    public boolean approvalRedPacket(Long id, Integer approvalStatus, String approvalContent) {

        //设置审核的公共参数
        RedPacket redPacketEntity = new RedPacket();
        redPacketEntity.setId(id);
        redPacketEntity.setModifyTime(new Date());
        redPacketEntity.setApprovalTime(new Date());

        //审核通过
        if (ApproveRedPacketEnum.OK.getApprovalCode().equals(approvalStatus)) {
            //保存记录到审核表中
            saveApprovalHistory(id, ApproveRedPacketEnum.OK.getApprovalCode(), null);

            //修改审核状态为审核通过,且登记为上线状态。
            redPacketEntity.setActiveStatus(ApproveRedPacketEnum.OK.getApprovalCode());
            redPacketEntity.setOnlineStatus(RedPacketOnlineStatusEnum.ON_LINE.getStatus());

            //更新审核通过状态成功,则推送消息给客户端
            if (updateSucBySingle(redPacketMapper.approvalRedPacket(redPacketEntity))) {

                //推送审核成功消息
                pushApprovalOKResult(id);
                return true;
            }
            return false;
        }

        //审核不通过
        if (ApproveRedPacketEnum.REJECT.getApprovalCode().equals(approvalStatus)) {
            //===记录拒绝记录到审核表中
            saveApprovalHistory(id, ApproveRedPacketEnum.REJECT.getApprovalCode(), approvalContent);

            //===更新拒绝状态、拒绝内容到红包表中
            redPacketEntity.setActiveStatus(ApproveRedPacketEnum.REJECT.getApprovalCode());
            redPacketEntity.setComment(approvalContent);
            //更新审核拒绝状态成功,则推送消息给客户端
            if (updateSucBySingle(redPacketMapper.approvalRedPacket(redPacketEntity))) {
                //推送审核拒绝消息
                pushApprovalRejectResult(id);
                return true;
            }
            return false;
        }

        //其他状态不允许操作,记录日志返回
        LOG.error("错误状态,不审核,红包id为:{}, 审核状态为:{}", id, approvalStatus);
        return false;

    }

    @Override
    public boolean updateOnlineStatus(Long id, Integer onlineStatus) {
        RedPacket redPacketEntity = new RedPacket();
        redPacketEntity.setId(id);
        redPacketEntity.setOnlineStatus(onlineStatus);
        redPacketEntity.setModifyTime(new Date());

        return updateSucBySingle(redPacketMapper.updateByPrimaryKeySelective(redPacketEntity));
    }

    @Override
    public RedPacketDTO getRedPacketDetail(Long id) {

        //以红包id查询红包详情
        RedPacket redPacket = redPacketMapper.selectByPrimaryKey(id);
        if (null == redPacket) {
            LOG.error("红包id:{}查询详情结果为空", id);
            return null;
        }
        LOG.info("红包id:{}查询详情结果为:{}", id, JSON.toJSONString(redPacket));

        //参数转换
        RedPacketDTO redPacketDTO = new RedPacketDTO();
        BeanUtils.copyProperties(redPacket, redPacketDTO);
        return redPacketDTO;
    }

    @Override
    public boolean editRedPacket(RedPacketParam redPacketParam) {
        //参数转换为数据库实体
        RedPacket redPacketEntity = new RedPacket();
        redPacketEntity.setId(redPacketParam.getId());
        redPacketEntity.setContent(redPacketParam.getContent());
        redPacketEntity.setPromotionContent(redPacketParam.getPromotionContent());
        redPacketEntity.setPromotionImg(redPacketParam.getPromotionImg());
        redPacketEntity.setModifyTime(new Date());

        //更新红包内容
        return updateSucBySingle(redPacketMapper.updateByPrimaryKeySelective(redPacketEntity));

    }

    /**
     * 推送审核通过成功消息
     *
     * @param redPacketId 红包id
     * @return 推送结果
     */
    private boolean pushApprovalOKResult(Long redPacketId) {

        RedPacket redPacket = redPacketMapper.selectByPrimaryKey(redPacketId);


        PushPayloadInfo info = PushPayloadInfo.build(PushMessageEnum.RED_PACKET_PASS);
        String content = "\uD83D\uDC4D恭喜哦，您的口令红包生效了哟， 快来分享给好友吧>";
        Long msgId = sequenceCreater.nextLongId();

        //=======推送消息
        info.setMsgId(msgId);
        info.addExtend("msgId", msgId);
        info.addExtend("rpId", redPacketId);
        info.addExtend("userId", redPacket.getUserId());

        PushMessage pushMessage = PushMessage.build();
        pushMessage.setTitle("");
        pushMessage.setContent(content);
        pushMessage.setType(TemplateTypeEnum.NOTIFCTION);
        pushMessage.setPushReceiveScope(PushReceiveScope.pushSignle(redPacket.getUserId()));
        pushMessage.setPayloadInfo(info);

        this.pushMsgIntegService.pushMsg(pushMessage);
        return true;
    }


    /**
     * 推送审核拒绝 成功消息
     *
     * @param redPacketId 红包id
     * @return 推送结果
     */
    private boolean pushApprovalRejectResult(Long redPacketId) {

        RedPacket redPacket = redPacketMapper.selectByPrimaryKey(redPacketId);


        PushPayloadInfo info = PushPayloadInfo.build(PushMessageEnum.RED_PACKET_NOT_PASS);
        String content = "\uD83D\uDE05很遗憾哦，您发的口令审核未通过，快来修改下吧>";
        Long msgId = sequenceCreater.nextLongId();

        //=======推送消息
        info.setMsgId(msgId);
        info.addExtend("msgId", msgId);
        info.addExtend("userId", redPacket.getUserId());

        PushMessage pushMessage = PushMessage.build();
        pushMessage.setTitle("");
        pushMessage.setContent(content);
        pushMessage.setType(TemplateTypeEnum.NOTIFCTION);
        pushMessage.setPushReceiveScope(PushReceiveScope.pushSignle(redPacket.getUserId()));
        pushMessage.setPayloadInfo(info);

        this.pushMsgIntegService.pushMsg(pushMessage);
        return true;
    }

    /**
     * 录入审核记录表
     *
     * @param redPacketId    红包id
     * @param approvalStatus 审核状态 1:拒绝,2:通过
     * @param comment        审核拒绝内容,在审核状态为拒绝时填入审核拒绝内容
     */
    private void saveApprovalHistory(Long redPacketId, Integer approvalStatus, String comment) {
        RedPacketApprovalHistory redPacketApprovalHistoryEntity = new RedPacketApprovalHistory();
        redPacketApprovalHistoryEntity.setId(sequenceCreater.nextLongId());
        redPacketApprovalHistoryEntity.setRpId(redPacketId);
        redPacketApprovalHistoryEntity.setCreateTime(new Date());
        redPacketApprovalHistoryEntity.setStatus(approvalStatus);
        redPacketApprovalHistoryEntity.setComment(comment);
        LOG.info("红包审核录入审核记录表,红包id为:{}, 审核状态为:{}", redPacketId, approvalStatus);

        redPacketApprovalHistoryMapper.insertSelective(redPacketApprovalHistoryEntity);

    }

    /**
     * 校验更新单条数据是否成功
     *
     * @param updateResult 更新结果
     * @return 校验结果, 更新单条数据结果为1则返回true 否则返回false
     */
    private boolean updateSucBySingle(int updateResult) {
        return 1 == updateResult;
    }
}
