package com.bxm.localnews.admin.service.activity.impl;

import com.bxm.component.mybatis.utils.MybatisBatchBuilder;
import com.bxm.localnews.admin.domain.VoteChoiceCountMapper;
import com.bxm.localnews.admin.domain.VoteMapper;
import com.bxm.localnews.admin.domain.VoteOptionsMapper;
import com.bxm.localnews.admin.param.VoteQueryParam;
import com.bxm.localnews.admin.service.activity.VoteService;
import com.bxm.localnews.admin.utils.SecurityUtils;
import com.bxm.localnews.admin.vo.VoteBean;
import com.bxm.localnews.admin.vo.VoteChoiceCountBean;
import com.bxm.localnews.admin.vo.VoteOptionsBean;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.Message;
import com.bxm.newidea.component.vo.PageWarper;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import org.mybatis.spring.SqlSessionTemplate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;

import static com.bxm.localnews.admin.constant.RedisConfig.VOTE_INFO_KEY;
import static com.bxm.localnews.admin.constant.RedisConfig.VOTE_OPTIONS_KEY;

@Service
public class VoteServiceImpl extends BaseService implements VoteService {

    private final VoteMapper voteMapper;

    private final VoteOptionsMapper voteOptionsMapper;

    private final SqlSessionTemplate sqlSessionTemplate;

    private final RedisStringAdapter redisStringAdapter;

    private final VoteChoiceCountMapper voteChoiceCountMapper;

    @Autowired
    public VoteServiceImpl(VoteMapper voteMapper,
                           VoteOptionsMapper voteOptionsMapper,
                           @Qualifier(value = "primarySessionTemplate") SqlSessionTemplate sqlSessionTemplate,
                           RedisStringAdapter redisStringAdapter,
                           VoteChoiceCountMapper voteChoiceCountMapper) {
        this.voteMapper = voteMapper;
        this.voteOptionsMapper = voteOptionsMapper;
        this.sqlSessionTemplate = sqlSessionTemplate;
        this.redisStringAdapter = redisStringAdapter;
        this.voteChoiceCountMapper = voteChoiceCountMapper;
    }

    @Override
    public VoteBean get(Long voteId) {
        Preconditions.checkArgument(null != voteId);

        VoteBean entity = voteMapper.selectByPrimaryKey(voteId);
        entity.setOptions(voteOptionsMapper.queryByVoteId(voteId));
        return entity;
    }

    @Override
    public PageWarper<VoteBean> getVoteByPage(VoteQueryParam param) {
        return new PageWarper<>(voteMapper.queryByPage(param));
    }

    @Override
    public List<VoteBean> getEnableVotes() {
        return voteMapper.selectEnableList();
    }

    @Override
    public Message remove(Long voteId) {
        Preconditions.checkArgument(null != voteId);

        VoteBean entity = new VoteBean();
        entity.setId(voteId);
        entity.setDeleteFlag((byte) 1);
        entity.setDeleteTime(new Date());
        entity.setDeleteUserId(SecurityUtils.currentUserId());

        return Message.build(voteMapper.remove(entity));
    }

    @Override
    public Message save(VoteBean entity) {

        Message message;
        fillLayoutType(entity);

        if (entity.getId() == null || entity.getId() == 0) {
            entity.setId(nextId());
            entity.setCreateTime(new Date());
            entity.setDeleteFlag((byte) 0);
            entity.setCreator(SecurityUtils.currentUserId());

            message = Message.build(voteMapper.insert(entity));
        } else {
            message = Message.build(voteMapper.updateByPrimaryKey(entity));
        }
        saveOptions(entity.getOptions(), entity.getId());
        clearCache(entity.getId());

        return message;
    }

    private void clearCache(Long voteId) {
        KeyGenerator key = VOTE_OPTIONS_KEY.copy().appendKey(voteId);
        redisStringAdapter.remove(key);
        key = VOTE_INFO_KEY.copy().appendKey(voteId);
        redisStringAdapter.remove(key);
    }

    private void saveOptions(List<VoteOptionsBean> requestOptions, Long voteId) {
        List<VoteOptionsBean> existsOptions = voteOptionsMapper.queryByVoteId(voteId);
        //设置选项顺序对选项进行处理

        List<VoteOptionsBean> addOptions = Lists.newArrayList();
        List<VoteOptionsBean> modifyOptions = Lists.newArrayList();
        List<VoteOptionsBean> removeOptions = Lists.newArrayList();

        List<VoteChoiceCountBean> newChoiceOptions = Lists.newArrayList();
        List<VoteChoiceCountBean> existsChoiceList = voteChoiceCountMapper.selectByParam(voteId);

        //提交的选项中包含数据库中不存在的，标记为新增
        int index = 0;
        for (VoteOptionsBean option : requestOptions) {
            option.setOrder(++index);
            option.setDeleteFlag((byte) 0);
            option.setVoteId(voteId);

            if (null == option.getId() || option.getId() == 0) {
                option.setId(nextId());
                addOptions.add(option);

                newChoiceOptions.add(VoteChoiceCountBean.builder()
                        .id(nextId())
                        .optionId(option.getId())
                        .voteId(voteId)
                        .total(0)
                        .build());
            } else if (existsOptions.stream().anyMatch(item -> option.getId().equals(item.getId()))) {
                modifyOptions.add(option);

                VoteChoiceCountBean existOptionCount = existsChoiceList.stream().filter((item) ->
                        option.getId().equals(item.getOptionId()))
                        .findFirst().orElse(VoteChoiceCountBean.builder().total(0).build());

                newChoiceOptions.add(VoteChoiceCountBean.builder()
                        .id(nextId())
                        .optionId(option.getId())
                        .voteId(voteId)
                        .total(existOptionCount.getTotal())
                        .build());
            }
        }

        //数据库中存在的选项在请求参数中不存在，表示被删除
        for (VoteOptionsBean existsOption : existsOptions) {
            if (requestOptions.stream().noneMatch(item -> existsOption.getId().equals(item.getId()))) {
                existsOption.setDeleteFlag((byte) 1);
                existsOption.setDeleteTime(new Date());
                existsOption.setDeleteUserId(SecurityUtils.currentUserId());

                removeOptions.add(existsOption);
            }
        }

        if (addOptions.size() > 0) {
            MybatisBatchBuilder.create(VoteOptionsMapper.class, addOptions).sessionTemplate(sqlSessionTemplate)
                    .run(VoteOptionsMapper::insert);
        }

        if (modifyOptions.size() > 0) {
            MybatisBatchBuilder.create(VoteOptionsMapper.class, modifyOptions).sessionTemplate(sqlSessionTemplate)
                    .run(VoteOptionsMapper::updateByPrimaryKey);
        }

        if (removeOptions.size() > 0) {
            MybatisBatchBuilder.create(VoteOptionsMapper.class, removeOptions).sessionTemplate(sqlSessionTemplate)
                    .run(VoteOptionsMapper::removeOption);
        }

        voteChoiceCountMapper.deleteByVoteId(voteId);
        if (newChoiceOptions.size() > 0) {
            MybatisBatchBuilder.create(VoteChoiceCountMapper.class, newChoiceOptions).sessionTemplate(sqlSessionTemplate)
                    .run(VoteChoiceCountMapper::insert);
        }
    }

    /**
     * 设置布局类型，如果存在选项不包含图片，即设置为TEXT
     *
     * @param entity 投票插件详情
     */
    private void fillLayoutType(VoteBean entity) {
        String layoutType = "IMG";
        List<VoteOptionsBean> options = entity.getOptions();
        for (VoteOptionsBean option : options) {
            if (StringUtils.isBlank(option.getImgUrl())) {
                layoutType = "TEXT";
                break;
            }
        }

        entity.setLayoutType(layoutType);
    }
}






























