package com.bxm.localnews.admin.service.base.impl;

import com.bxm.localnews.admin.constant.RedisConfig;
import com.bxm.localnews.admin.enums.TooltipStrategyEnum;
import com.bxm.localnews.admin.domain.AdminToolsLocationMapper;
import com.bxm.localnews.admin.domain.AdminToolsMapper;
import com.bxm.localnews.admin.dto.TooltipStrategyDTO;
import com.bxm.localnews.admin.enums.AreaWhiteBlackKeyEnum;
import com.bxm.localnews.admin.param.ToolsInfoParam;
import com.bxm.localnews.admin.param.ToolsParam;
import com.bxm.localnews.admin.service.base.AdminToolsService;
import com.bxm.localnews.admin.vo.Tools;
import com.bxm.localnews.admin.vo.ToolsLocation;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.PageWarper;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;

/**
 * @author zhaoyadong 2018/10/23 13:58
 * @desc
 */
@Service
public class AdminToolsServiceImpl implements AdminToolsService {

    @Resource
    private AdminToolsMapper adminToolsMapper;

    @Resource
    private AdminToolsLocationMapper adminToolsLocationMapper;

    @Resource
    private RedisHashMapAdapter redisHashMapAdapter;

    @Resource
    private SequenceCreater sequenceCreater;

    @Override
    public PageWarper<Tools> getToolsList(ToolsParam toolsParam) {
        PageWarper<Tools> pageWrapper = new PageWarper<>(this.adminToolsMapper.queryToolsList(toolsParam));

        if (CollectionUtils.isNotEmpty(pageWrapper.getList())) {
            pageWrapper.getList().forEach(tool -> {
                if (0 == tool.getGlobalFlag()) {
                    tool.setLocationList(this.adminToolsLocationMapper.getToolsLocationByToolId(tool.getId()));
                }
            });
        }

        return pageWrapper;
    }

    @Override
    public Tools getToolsById(Long id) {
        Tools tools = this.adminToolsMapper.selectByPrimaryKey(id);
        tools.setLocationList(adminToolsLocationMapper.getToolsLocationByToolId(tools.getId()));
        return tools;
    }

    @Override
    public int updateToolsStatusById(List<String> ids, Byte status) {
        int update = this.adminToolsMapper.updateToolsStatusById(ids, status);
        removeCache();
        return update;
    }

    @Override
    public int saveOrUpdate(ToolsInfoParam toolsInfoParam) {
        //为了一个参数，要做一次参数转换，也是醉了
        Tools toolInfo = new Tools();
        BeanUtils.copyProperties(toolsInfoParam, toolInfo);

        int operationResult;
        //id为空则为新增
        if (null == toolInfo.getId()) {
            toolInfo.setId(sequenceCreater.nextLongId());
            operationResult = adminToolsMapper.insertSelective(toolInfo);
        } else {
            //否则为更新
            operationResult = adminToolsMapper.updateByPrimaryKeySelective(toolInfo);
        }

        updateToolsLocation(toolInfo.getId(), toolsInfoParam.getToolAreaList());
        removeCache();

        return operationResult;
    }

    @Override
    public int deleteTools(Long id) {
        int update = this.adminToolsMapper.deleteByPrimaryKey(id);
        removeCache();
        return update;
    }

    @Override
    public List<TooltipStrategyDTO> getAllTooltipStrategy() {
        List<TooltipStrategyDTO> resultList = new ArrayList<>();
        //遍历枚举值
        for (TooltipStrategyEnum tooltipStrategyEnum : TooltipStrategyEnum.values()) {
            TooltipStrategyDTO tooltipStrategyDTO = new TooltipStrategyDTO();
            tooltipStrategyDTO.setName(tooltipStrategyEnum.getName());
            tooltipStrategyDTO.setDescription(tooltipStrategyEnum.getDescription());
            resultList.add(tooltipStrategyDTO);
        }

        return resultList;
    }

    /**
     * 更新对应地区的信息
     * @param toolId 工具id
     * @param toolAreaList 工具投放区域
     */
    private void updateToolsLocation(Long toolId, List<ToolsLocation> toolAreaList) {
        this.adminToolsLocationMapper.deleteByToolId(toolId);
        insertToolsLocation(toolAreaList, toolId);
    }

    /**
     * 添加地区信息
     * @param toolAreaList 工具地区信息列表
     * @param toolId 工具id
     */
    private void insertToolsLocation(List<ToolsLocation> toolAreaList, Long toolId) {
        if (CollectionUtils.isEmpty(toolAreaList)) {
            return;
        }

        //初始化待插入的数据列表
        List<ToolsLocation> toolsLocations = new ArrayList<>(toolAreaList.size());
        toolAreaList.forEach(toolAreaInfo -> {
            ToolsLocation toolsLocation = new ToolsLocation();
            toolsLocation.setId(sequenceCreater.nextLongId());
            toolsLocation.setAreaCode(toolAreaInfo.getAreaCode());
            toolsLocation.setToolId(toolId);
            toolsLocation.setAreaType(toolAreaInfo.getAreaType());

            //塞入待插入数据的列表
            toolsLocations.add(toolsLocation);
        });

        this.adminToolsLocationMapper.insertToolsLocations(toolsLocations);
    }

    /**
     * 移除缓存
     */
    private void removeCache() {
        redisHashMapAdapter.remove(RedisConfig.AREA_WHITE_BLACK_LIST.copy().appendKey(AreaWhiteBlackKeyEnum.TOOLS.getKey()));
    }
}
