package com.bxm.localnews.admin.service.base.impl;

import com.bxm.localnews.admin.constant.RedisConfig;
import com.bxm.localnews.admin.domain.LocationChannelAreaMapper;
import com.bxm.localnews.admin.domain.LocationChannelMapper;
import com.bxm.localnews.admin.dto.ChannelAreaDTO;
import com.bxm.localnews.admin.dto.ChannelInfoDetailDTO;
import com.bxm.localnews.admin.dto.UserChannelDTO;
import com.bxm.localnews.admin.param.UserChannelParam;
import com.bxm.localnews.admin.service.base.UserChannelService;
import com.bxm.localnews.admin.vo.LocationChannelBean;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.vo.PageWarper;
import com.github.pagehelper.PageInfo;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;


/**
 * @author : jieli.xu
 * @date 2020/6/18 17:27
 * 用户栏目管理服务
 */
@Service
public class UserChannelServiceImpl extends BaseService implements UserChannelService {

    @Autowired
    private LocationChannelMapper locationChannelMapper;

    @Autowired
    private LocationChannelAreaMapper locationChannelAreaMapper;

    @Autowired
    private RedisHashMapAdapter redisHashMapAdapter;


    @Override
    public PageWarper<UserChannelDTO> getList(UserChannelParam userChannelParam) {
        PageInfo<LocationChannelBean> pageInfo = new PageInfo<>(locationChannelMapper.getList(userChannelParam));
        List<UserChannelDTO> resultList = new ArrayList<>();
        pageInfo.getList().forEach(locationChannelBean -> {
            resultList.add(convert(locationChannelBean));
        });
        PageWarper<UserChannelDTO> pageWarper = new PageWarper<>(resultList);
        pageWarper.setPageNum(pageInfo.getPageNum());
        pageWarper.setPageSize(pageInfo.getPageSize());
        pageWarper.setTotal(pageInfo.getTotal());
        return pageWarper;
    }


    @Override
    public ChannelInfoDetailDTO getDetail(UserChannelParam userChannelParam) {
        LocationChannelBean locationChannelBean = locationChannelMapper.selectByPrimaryKey(userChannelParam.getChannelId());
        ChannelInfoDetailDTO userChannelDTO = new ChannelInfoDetailDTO();
        BeanUtils.copyProperties(locationChannelBean, userChannelDTO);
        userChannelDTO.setId(locationChannelBean.getId() + "");
        userChannelDTO.setChannelAreaDTOS(locationChannelAreaMapper.getListByChannelId(locationChannelBean.getId()));
        return userChannelDTO;
    }

    @Override
    public Boolean updateStatus(UserChannelParam userChannelParam) {
        if (locationChannelMapper.updateStatus(userChannelParam.getChannelId()) > 0) {
            ChannelInfoDetailDTO channelInfoDetailDTO = this.getDetail(userChannelParam);
            //清空redis缓存
            this.resetRedis(channelInfoDetailDTO);
            return true;
        }
        return false;
    }

    @Override
    public Boolean insert(ChannelInfoDetailDTO channelInfoDetailDTO) {
        LocationChannelBean bean = new LocationChannelBean();
        BeanUtils.copyProperties(channelInfoDetailDTO, bean);
        bean.setId(nextId());
        if (locationChannelMapper.insertSelective(bean) > 0) {
            //更新区域位置关联信息
            if (null != channelInfoDetailDTO.getChannelAreaDTOS() && !channelInfoDetailDTO.getChannelAreaDTOS().isEmpty()) {
                channelInfoDetailDTO.getChannelAreaDTOS().forEach(entity -> {
                    entity.setLocationChannelId(bean.getId());
                    entity.setAreaType(bean.getGlobalStatus());
                });
                locationChannelAreaMapper.batchInsert(channelInfoDetailDTO.getChannelAreaDTOS());
            }
        }
        //清空缓存
        channelInfoDetailDTO.setId(Objects.toString(bean.getId()));
        this.resetRedis(channelInfoDetailDTO);
        return true;
    }

    @Override
    public Boolean update(ChannelInfoDetailDTO channelInfoDetailDTO) {
        if (null == channelInfoDetailDTO.getId()) {
            return false;
        }
        //清空缓存
        this.resetRedis(channelInfoDetailDTO);

        LocationChannelBean bean = new LocationChannelBean();
        BeanUtils.copyProperties(channelInfoDetailDTO, bean);
        bean.setId(Long.parseLong(channelInfoDetailDTO.getId()));
        if (locationChannelMapper.updateByPrimaryKeySelective(bean) > 0) {
            //更新区域位置关联信息
            locationChannelAreaMapper.deleteByLocationId(Long.parseLong(channelInfoDetailDTO.getId()));
            if (!CollectionUtils.isEmpty(channelInfoDetailDTO.getChannelAreaDTOS())) {
                channelInfoDetailDTO.getChannelAreaDTOS().forEach(entity -> {
                    if (null == entity.getLocationChannelId()) {
                        entity.setLocationChannelId(bean.getId());
                        entity.setAreaType(bean.getGlobalStatus());
                    }
                });
                locationChannelAreaMapper.batchInsert(channelInfoDetailDTO.getChannelAreaDTOS());
            }
        }
        return true;
    }

    @Override
    public List<LocationChannelBean> getEnableList() {
        return locationChannelMapper.getEnableChannel();
    }

    private UserChannelDTO convert(LocationChannelBean bean) {
        return UserChannelDTO.builder()
                .channelName(bean.getChannelName())
                .channelUrl(bean.getChannelUrl())
                .focusOn(bean.getFocusOn())
                .status(bean.getStatus())
                .id(bean.getId() + "")
                .isFixed(bean.getIsFixed())
                .globalStatus(bean.getGlobalStatus())
                .defaultChose(bean.getDefaultChose())
                .sort(bean.getSort())
                .name(bean.getGlobalStatus() == 1 ? "全国" : bean.getName())
                .build();
    }

    /**
     * 重置区域编码栏目,如果为空，删除多个
     *
     * @param channelInfoDetailDTO
     */
    private void resetRedis(ChannelInfoDetailDTO channelInfoDetailDTO) {
        //清空前端缓存
        if (channelInfoDetailDTO.getGlobalStatus().equals(1)) {
            redisHashMapAdapter.increment(RedisConfig.LOCATION_VERSION_CACHE, "0", 1);
        } else {
            //老的历史记录删除
            List<ChannelAreaDTO> listAreaBean = locationChannelAreaMapper.getListByChannelId(Long.parseLong(channelInfoDetailDTO.getId()));
            if (CollectionUtils.isNotEmpty(listAreaBean)) {
                if (CollectionUtils.isNotEmpty(channelInfoDetailDTO.getChannelAreaDTOS())) {
                    listAreaBean.removeAll(channelInfoDetailDTO.getChannelAreaDTOS());
                }
                listAreaBean.forEach(entity -> {
                    redisHashMapAdapter.increment(RedisConfig.LOCATION_VERSION_CACHE, entity.getAreaCode(), 1);
                });
            }
            //新的信息删除
            if (!CollectionUtils.isEmpty(channelInfoDetailDTO.getChannelAreaDTOS())) {
                channelInfoDetailDTO.getChannelAreaDTOS().forEach(entity -> {
                    redisHashMapAdapter.increment(RedisConfig.LOCATION_VERSION_CACHE, entity.getAreaCode(), 1);
                });
            }
        }
    }

}
