package com.bxm.localnews.admin.service.base.impl;

import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.admin.constant.RedisConfig;
import com.bxm.localnews.admin.convert.impl.VirtualUserConverter;
import com.bxm.localnews.admin.convert.impl.VirtualUserInfoConverter;
import com.bxm.localnews.admin.domain.*;
import com.bxm.localnews.admin.domain.auth.UserRoleMapper;
import com.bxm.localnews.admin.dto.AdminVirtualUserDTO;
import com.bxm.localnews.admin.dto.ChatVirtualUserDTO;
import com.bxm.localnews.admin.dto.VirtualUserOverviewDTO;
import com.bxm.localnews.admin.integration.UserAuthCodeIntegrationService;
import com.bxm.localnews.admin.param.UserSyncParam;
import com.bxm.localnews.admin.param.VirtualUserInfoParam;
import com.bxm.localnews.admin.param.VirtualUserParam;
import com.bxm.localnews.admin.param.VirtualUserQueryParam;
import com.bxm.localnews.admin.param.security.UserRoleParam;
import com.bxm.localnews.admin.service.base.UserTagService;
import com.bxm.localnews.admin.service.base.VirtualUserService;
import com.bxm.localnews.admin.vo.AdminVirtual;
import com.bxm.localnews.admin.vo.AllVirtureUserVo;
import com.bxm.localnews.admin.vo.User;
import com.bxm.localnews.admin.vo.VirtualUser;
import com.bxm.localnews.admin.vo.auth.UserRole;
import com.bxm.localnews.admin.vo.auth.UserRoleBean;
import com.bxm.localnews.admin.vo.security.AdminUser;
import com.bxm.localnews.auth.enums.RoleCodeEnum;
import com.bxm.localnews.common.constant.RedisMessageChannel;
import com.bxm.localnews.common.vo.RedisMessageBean;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.PageWarper;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

@Service
@Slf4j
public class VirtualUserServiceImpl extends BaseService implements VirtualUserService {

    private UserMapper userMapper;

    private VirtualUserMapper virtualUserMapper;

    private VirtualUserConverter virtualUserConverter;

    private VirtualUserInfoConverter virtualUserInfoConverter;

    private RedisHashMapAdapter redisHashMapAdapter;

    private RedisStringAdapter redisStringAdapter;

    private UserTagService userTagService;

    private AdminVirtualMapper adminVirtualMapper;

    private AdminUserMapper adminUserMapper;

    private IMMapper imMapper;

    private UserRoleMapper userRoleMapper;

    private UserAuthCodeIntegrationService userAuthCodeIntegrationService;

    private SequenceCreater sequenceCreater;


    @Autowired
    public VirtualUserServiceImpl(UserMapper userMapper,
                                  VirtualUserMapper virtualUserMapper,
                                  VirtualUserConverter virtualUserConverter,
                                  VirtualUserInfoConverter virtualUserInfoConverter,
                                  RedisHashMapAdapter redisHashMapAdapter,
                                  RedisStringAdapter redisStringAdapter,
                                  UserTagService userTagService,
                                  AdminVirtualMapper adminVirtualMapper,
                                  AdminUserMapper adminUserMapper,
                                  IMMapper imMapper,
                                  UserRoleMapper userRoleMapper,
                                  UserAuthCodeIntegrationService userAuthCodeIntegrationService,
                                  SequenceCreater sequenceCreater) {
        this.userMapper = userMapper;
        this.virtualUserMapper = virtualUserMapper;
        this.virtualUserConverter = virtualUserConverter;
        this.virtualUserInfoConverter = virtualUserInfoConverter;
        this.redisHashMapAdapter = redisHashMapAdapter;
        this.redisStringAdapter = redisStringAdapter;
        this.userTagService = userTagService;
        this.adminVirtualMapper = adminVirtualMapper;
        this.adminUserMapper = adminUserMapper;
        this.imMapper = imMapper;
        this.userRoleMapper = userRoleMapper;
        this.userAuthCodeIntegrationService = userAuthCodeIntegrationService;
        this.sequenceCreater = sequenceCreater;
    }

    @Override
    public Long save(VirtualUserInfoParam virtualUser, AdminUser loginUser) {
        User user = virtualUserConverter.convert(virtualUser);
        VirtualUser virtualUserVo = virtualUserInfoConverter.convert(virtualUser);

        if (null == virtualUser.getId()) {
            //用户表新增
            userMapper.insertSelective(user);
            //虚拟用户表新增
            virtualUserVo.setId(user.getId());
            virtualUserMapper.insertSelective(virtualUserVo);
            // 兴趣标签新增
            userTagService.save(user.getId(), user.getUserTagIds());
            //初始化马甲号管理信息
            initVirtualAdminInfo(user.getId(), loginUser);

        } else {
            //更新用户表和虚拟用户表
            userMapper.updateByPrimaryKeySelective(user);
            virtualUserMapper.updateByPrimaryKeySelective(virtualUserVo);

            // 更新兴趣标签
            userTagService.update(user.getId(), user.getUserTagIds());

            //删除缓存
            redisStringAdapter.remove(RedisConfig.USER_INFO.copy().appendKey(virtualUser.getId()));
            redisHashMapAdapter.remove(RedisConfig.USER_TAG, String.valueOf(virtualUser.getId()));

            //更新用户冗余信息
            UserSyncParam userSyncParam = new UserSyncParam()
                    .setHeadImg(virtualUser.getHeadImg())
                    .setId(virtualUser.getId())
                    .setNickname(virtualUser.getNickname())
                    .setSex(virtualUser.getSex());
            this.syncUser(userSyncParam);
        }
        return user.getId();
    }

    private void initVirtualAdminInfo(Long userId, AdminUser loginUser) {
        //使马甲号默认为马甲号管理员（不然不能调用切换马甲号功能）
        UserRole userRole = new UserRole();
        userRole.setId(sequenceCreater.nextLongId());
        userRole.setUserId(userId);
        userRole.setRoleCode(RoleCodeEnum.VIRTUAL_ADMIN.name());
        userRole.setCreateTime(new Date());
        userRoleMapper.insertSelective(userRole);

        UserRoleParam param = new UserRoleParam();
        param.setUserId(userId);
        param.setRoleCode(RoleCodeEnum.VIRTUAL_ADMIN.name());
        userAuthCodeIntegrationService.addRole(param);

        //需求为新增的马甲号为新增人管控
        AdminVirtual virtual = AdminVirtual.builder()
                .id(nextId())
                .addTime(new Date())
                .modifyTime(new Date())
                .virtualUserId(userId)
                .adminUserAccount(loginUser.getUsername())
                .adminUserId(loginUser.getId())
                .operateUserId(loginUser.getId())
                .operateUserAccount(loginUser.getUsername())
                .build();
        adminVirtualMapper.insertSelective(virtual);
    }

    /**
     * 同步用户冗余信息
     *
     * @param user 用户冗余信息
     */
    private void syncUser(UserSyncParam user) {
        RedisMessageBean<UserSyncParam> userSyncParamRedisMessageBean = new RedisMessageBean<>();
        userSyncParamRedisMessageBean.setEventBody(user);
        userSyncParamRedisMessageBean.setCreateTime(new Date());
        userSyncParamRedisMessageBean.setEventCode(String.valueOf(RedisMessageChannel.USER_UPDATE_EVENT.getCode()));
        userSyncParamRedisMessageBean.setEventId(String.valueOf(nextSequence()));
        userSyncParamRedisMessageBean.setEventName(String.valueOf(RedisMessageChannel.USER_UPDATE_EVENT.getDesc()));
        logger.info("更新用户冗余信息,发送redis消息:[{}]", JSONObject.toJSONString(user));
        redisStringAdapter.convertAndSend(RedisMessageChannel.USER_UPDATE_EVENT.getName(), JSONObject.toJSON(userSyncParamRedisMessageBean));
    }

    @Override
    public PageWarper<AdminVirtualUserDTO> listByPage(VirtualUserParam virtualUserParam) {
        return new PageWarper<>(virtualUserMapper.queryPageSize(virtualUserParam));
    }

    @Override
    public List<VirtualUserOverviewDTO> list(VirtualUserQueryParam virtualUserQueryParam) {
        VirtualUser virtualUser = new VirtualUser();
        BeanUtils.copyProperties(virtualUserQueryParam, virtualUser);
        List<VirtualUser> virtualUserList = virtualUserMapper.selectByModel(virtualUser);
        return virtualUserList.stream().map(e -> VirtualUserOverviewDTO.builder()
                .id(e.getId())
                .headImg(e.getHeadImg())
                .nickname(e.getNickname())
                //方仙女(女  单身，广德)
                .userInfo(getUserInfo(e.getNickname() + "(" + getSex(e.getSex()), e.getRemark()))
                .build()).collect(Collectors.toList());
    }

    @Override
    public VirtualUser getVirtualUserDetail(Long id) {
        VirtualUser virtualUserParam = new VirtualUser();
        virtualUserParam.setId(id);
        List<VirtualUser> virtualUsers = virtualUserMapper.selectByModel(virtualUserParam);

        if (CollectionUtils.isEmpty(virtualUsers)) {
            return null;
        }
        VirtualUser virtualuser = virtualUsers.get(0);
        virtualuser.setUserTagIds(JSONObject.parseArray(virtualuser.getHobby(), Long.class));
        if (StringUtils.isNotBlank(virtualuser.getHometownCode())) {
            String[] hometownCodeSplit = virtualuser.getHometownCode().split("_");
            if (hometownCodeSplit.length == 3) {
                virtualuser.setHometownCode(hometownCodeSplit[2]);
                virtualuser.setCountyCode(hometownCodeSplit[2]);
                virtualuser.setCityCode(hometownCodeSplit[1]);
                virtualuser.setProvinceCode(hometownCodeSplit[0]);
            }
        }
        return virtualuser;
    }

    @Override
    @Cacheable(value = "listVirtualUser", sync = true)
    public Map<Long, VirtualUser> list() {
        VirtualUser virtualUser = new VirtualUser();
        List<VirtualUser> virtualUserList = virtualUserMapper.selectByModel(virtualUser);
        return virtualUserList.stream().collect(Collectors.toMap(VirtualUser::getId, Function.identity(), (key1, key2) -> key2));
    }

    @Override
    public Boolean changeVestControl(Long virtualUserId, Long adminUserId, AdminUser loginUser, Byte appShow) {

        VirtualUser virtualUser = virtualUserMapper.getVirtualUserInfo(virtualUserId);
        if (virtualUser == null) {
            log.error("管控马甲号接口，马甲号信息不存在，virtualUserId:{},adminUserId:{},loginUser:{},appShow:{}",
                    virtualUserId, adminUserId, loginUser, appShow);
            return false;
        } else {
            VirtualUser virtualUserVO = new VirtualUser();
            virtualUserVO.setId(virtualUser.getId());
            virtualUserVO.setAppShow(appShow);
            virtualUserMapper.updateByPrimaryKeySelective(virtualUserVO);
        }

        //查询运营马甲号关联表
        AdminVirtual admin = adminVirtualMapper.selectByVirtualUserId(virtualUserId);
        //运营用户信息
        AdminUser adminUser = adminUserMapper.selectById(adminUserId);
        if (Objects.isNull(admin)) {
            AdminVirtual virtual = AdminVirtual.builder()
                    .id(nextId())
                    .addTime(new Date())
                    .modifyTime(new Date())
                    .virtualUserId(virtualUserId)
                    .adminUserAccount(adminUser.getUsername())
                    .adminUserId(adminUserId)
                    .operateUserId(loginUser.getId())
                    .operateUserAccount(loginUser.getUsername())
                    .build();
            adminVirtualMapper.insertSelective(virtual);
        } else {
            admin.setModifyTime(new Date());
            admin.setAdminUserId(adminUserId);
            admin.setVirtualUserId(virtualUserId);
            admin.setAdminUserAccount(adminUser.getUsername());
            adminVirtualMapper.updateByPrimaryKeySelective(admin);
        }
        return Boolean.TRUE;
    }

    @Override
    public List<ChatVirtualUserDTO> getVirtualList(AdminUser loginUser) {
        List<AdminVirtualUserDTO> list = virtualUserMapper.queryVirtualByAdminUserId(loginUser.getId());
        if (CollectionUtils.isEmpty(list)) {
            return Collections.emptyList();
        }

        // 查询这些用户的最后一次聊天时间
        Map<Long, Date> recentTimeMap = getRecentTime(list.stream().map(VirtualUser::getId).collect(Collectors.toList()));

        return list.stream()
                .map(e -> {
                    ChatVirtualUserDTO cv = new ChatVirtualUserDTO();
                    BeanUtils.copyProperties(e, cv);
                    //获得当前的马甲号 有多少未回复的数量
                    cv.setUnReplyNum(getUnReplyNum(e));
                    cv.setRecentTime(recentTimeMap.get(e.getId()));
                    return cv;
                })
                //客户端只显示前100条记录
                .limit(100)
                .sorted()
                .collect(Collectors.toList());
    }

    private Map<Long, Date> getRecentTime(List<Long> userIds) {
        List<AllVirtureUserVo> allVirtureUserVos = imMapper.selectLastTimeByUserIds(userIds);
        Map<Long, Date> collect = allVirtureUserVos.stream().collect(Collectors.toMap(AllVirtureUserVo::getUserId, AllVirtureUserVo::getMaxTime));
        return collect;
    }

    @Override
    public List<User> getUserByKeyWord(String keyWord) {
        return userMapper.getUserByKeyWord(keyWord);
    }

    private Integer getUnReplyNum(AdminVirtualUserDTO adminVirtualUserDTO) {
        //当前虚拟用户私聊过的真实用户
        KeyGenerator unReplyNumKey = RedisConfig.VIRTURE_USER_UNREPLY.copy().appendKey(adminVirtualUserDTO.getId());
        Map<String, Integer> entries = redisHashMapAdapter.entries(unReplyNumKey, Integer.class);
        Collection<Integer> values = entries.values();
        int sum = values.stream().mapToInt(Integer::intValue).sum();
        return sum;
    }

    private String getSex(Byte sex) {
        String sexStr = "男";
        if (null != sex && 1 == sex) {
            sexStr = "男";
        } else if (null != sex && 2 == sex) {
            sexStr = "女";
        }
        return sexStr;
    }

    private String getUserInfo(String userInfo, String remark) {
        if (StringUtils.isEmpty(remark)) {
            return userInfo + ")";
        } else {
            return userInfo + " " + remark + ")";
        }
    }
}
