package com.bxm.localnews.admin.service.im.impl;

import com.bxm.localnews.admin.common.ImChatRoomProperties;
import com.bxm.localnews.admin.constant.DeleteFlag;
import com.bxm.localnews.admin.domain.ImChatroomExtendMapper;
import com.bxm.localnews.admin.domain.ImTimingRedPacketPlanExtendMapper;
import com.bxm.localnews.admin.entry.ImTimingRedPacketPlan;
import com.bxm.localnews.admin.integration.IMIntegrationService;
import com.bxm.localnews.admin.param.*;
import com.bxm.localnews.admin.service.im.ImChatRoomRedPacketService;
import com.bxm.localnews.admin.vo.ImChatroom;
import com.bxm.localnews.admin.vo.im.ImTimingRedPacketPlanVO;
import com.bxm.localnews.common.vo.Json;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import com.bxm.newidea.component.vo.PageWarper;
import com.github.pagehelper.Page;
import com.github.pagehelper.PageHelper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author Gonzo
 * @date 2019-11-01 11:22
 */
@Service
@Slf4j
public class ImChatRoomRedPacketServiceImpl implements ImChatRoomRedPacketService {

    @Autowired
    private ImTimingRedPacketPlanExtendMapper imTimingRedPacketPlanExtendMapper;

    @Autowired
    private ImChatroomExtendMapper imChatroomExtendMapper;

    @Autowired
    private SequenceCreater sequenceCreater;

    @Autowired
    private ImChatRoomProperties imChatRoomProperties;

    @Autowired
    private IMIntegrationService imIntegrationService;

    @Autowired
    private ScheduledThreadPoolExecutor scheduledThreadPoolExecutor;

    private static final BigDecimal MIX = BigDecimal.valueOf(0.01);

    private ImTimingRedPacketPlanVO build(ImTimingRedPacketPlan plan) {

        return ImTimingRedPacketPlanVO.builder()
                .chatRoomId(plan.getRedPacketTargetId())
                .createTime(plan.getCreateTime())
                .id(plan.getId())
                .modifyTime(plan.getModifyTime())
                .redPacketName(plan.getRedPacketName())
                .redPacketTotalAmount(plan.getRedPacketTotalAmount())
                .redPacketTotalNum(plan.getRedPacketTotalNum())
                .sentTimeStr(DateUtils.PATTERN_HOUR_MINUTE_FORMAT.get().format(plan.getSentTime()))
                .status(plan.getStatus())
                .build();

    }

    @Override
    public Json createTimingRedPacketPlan(CreateTimingRedPacketPlanParam param) {

        // 校验当前聊天室是否有相差5分钟之内的红包
        List<ImTimingRedPacketPlan> imTimingRedPacketPlans = imTimingRedPacketPlanExtendMapper.listPlansByChatRoomId(param.getChatRoomId());

        if (hasNearest(imTimingRedPacketPlans, param.getSentTime())) {
            return Json.badReqeuset("请勿添加相差时间5m之内的红包");
        }

        if (!checkAmt(param.getRedPacketTotalAmount(), param.getRedPacketTotalNum())) {
            return Json.badReqeuset("平均每个红包不可少于0.01元");
        }


        Date now = new Date();
        ImTimingRedPacketPlan plan = new ImTimingRedPacketPlan();

        // 目标id（群聊红包：聊天室id，个人红包：用户id）
        plan.setRedPacketTargetId(param.getChatRoomId());
        // 红包金额 单位元
        plan.setRedPacketTotalAmount(param.getRedPacketTotalAmount());
        //  红包总可领取数
        plan.setRedPacketTotalNum(param.getRedPacketTotalNum());
        // 每日定时发送的时间
        plan.setSentTime(param.getSentTime());
        // 发送红包用户id 一期是红包小助手
        plan.setRedPacketSourceUserId(imChatRoomProperties.getChatRoomTimingRedPacketAssistantUserId());
        // 红包寄语
        plan.setRedPacketRemark(imChatRoomProperties.getChatRoomTimingRedPacketRemark());

        // 下面这些一期还不支持，后期可以拓展的，先暂时设置为默认值

        // 红包名称
        plan.setRedPacketName("定时红包");
        // 红包来源：1：管理后台配置 2：App用户发送
        plan.setRedPacketSource((byte) 1);
        // 红包类型：1：群聊红包 2：个人红包
        plan.setRedPacketType((byte) 1);
        // 已发送次数
        plan.setSendTimes(0);
        // 如果是指定日期，则有效
        plan.setSentDate(null);
        // 红包金额类型 1：现金红包
        plan.setRedPacketAmountType((byte) 1);
        // 领取用户类型：1：邀请好友之后可领取
        plan.setRedPacketAuthType((byte) 1);
        // 红包派发生成策略（群聊红包可用）：1：每个用户可获得0.1~总金额 / 总个数 * 2区间内的金额
        plan.setRedPacketDistributeArithmeticType((byte) 1);
        // 定时类型：1：每日定时发送 2：指定日期与时间发送
        plan.setTimingType((byte) 1);
        // 发送总次数 -1为无限次
        plan.setTotalTimes(-1);

        // 状态：1：正常
        plan.setStatus((byte) 1);
        plan.setCreateTime(now);
        plan.setDeleteFlag(DeleteFlag.NOT_DELETED);
        plan.setDeleteTime(null);
        plan.setId(sequenceCreater.nextLongId());
        plan.setModifyTime(now);


        imTimingRedPacketPlanExtendMapper.insertSelective(plan);

        // 定时任务
        addTask(plan.getId());

        return Json.ok();
    }

    private boolean hasNearest(List<ImTimingRedPacketPlan> imTimingRedPacketPlans, Date sentTime) {

        ZoneId zone = ZoneId.systemDefault();

        LocalDateTime localDateTime1 = LocalDateTime.of(LocalDate.now(),
                LocalDateTime.ofInstant(sentTime.toInstant(), zone).toLocalTime());

        return imTimingRedPacketPlans.stream()
                .filter(p -> {
                    // 是否有效
                    if (Objects.equals(p.getStatus().intValue(), 1)) {

                        // 暂时只做每日定时发送比较
                        if (Objects.equals(p.getTimingType().intValue(), 1)) {

                            LocalDateTime localDateTime2 = LocalDateTime.of(LocalDate.now(),
                                    LocalDateTime.ofInstant(p.getSentTime().toInstant(), zone).toLocalTime());

                            Duration duration = localDateTime2.isBefore(localDateTime1) ? Duration.between(localDateTime2,
                                    localDateTime1)
                                    : Duration.between(localDateTime1, localDateTime2);

                            // 小于5m
                            return duration.toMinutes() < 5;
                        }
                    }
                    return false;
                }).count() > 0;

    }

    /**
     * 金额是否超过最低0.1
     * @return
     */
    private boolean checkAmt(BigDecimal redPacketTotalAmount, Integer redPacketTotalNum) {

        if (redPacketTotalAmount.divide(BigDecimal.valueOf(redPacketTotalNum),
                2, RoundingMode.HALF_DOWN)
                .compareTo(MIX) < 0) {

            return false;
        }

        return true;
    }

    private Message checkChatRoomStatus(String chatRoomId) {

        ImChatroom chatroom = imChatroomExtendMapper.selectByChatRoomId(chatRoomId);

        if (Objects.isNull(chatroom)) {
            return Message.build(false).setMessage("聊天室不存在");
        }

        // 聊天室未开启，也可以保存

        /*if (Objects.equals(chatroom.getEnableChatRoom().intValue(), 0)) {
            return Message.build(false).setMessage("聊天室未开启");
        }

        if (Objects.equals(chatroom.getEnableTimingRedPacket().intValue(), 0)) {
            return Message.build().setMessage("聊天室未开启定时红包功能");
        }*/

        return Message.build();

    }

    @Override
    public Json updateTimingRedPacketPlan(UpdateTimingRedPacketPlanParam param) {

        ImTimingRedPacketPlan plan = imTimingRedPacketPlanExtendMapper.selectByPrimaryKey(param.getId());

        if (Objects.isNull(plan)) {
            return Json.badReqeuset("红包不存在");
        }

        if (Objects.nonNull(param.getSentTime())) {

            // 校验当前聊天室是否有相差5分钟之内的红包
            List<ImTimingRedPacketPlan> imTimingRedPacketPlans = imTimingRedPacketPlanExtendMapper.listPlansByChatRoomId(plan.getRedPacketTargetId());

            imTimingRedPacketPlans = imTimingRedPacketPlans.stream()
                    .filter(p -> {
                        // 过滤当前要编辑的计划
                        return !Objects.equals(p.getId(), param.getId());
                    }).collect(Collectors.toList());

            if (hasNearest(imTimingRedPacketPlans, param.getSentTime())) {
                return Json.badReqeuset("请勿添加相差时间5m之内的红包");
            }
        }

        if (!checkAmt(param.getRedPacketTotalAmount(), param.getRedPacketTotalNum())) {
            return Json.badReqeuset("平均每个红包不可少于0.01元");
        }

        Message message = checkChatRoomStatus(plan.getRedPacketTargetId());

        if (!message.isSuccess()) {
            return Json.badReqeuset(message.getLastMessage());
        }

        ImTimingRedPacketPlan update = new ImTimingRedPacketPlan();

        BeanUtils.copyProperties(param, update);

        Date now = new Date();
        update.setModifyTime(now);

        imTimingRedPacketPlanExtendMapper.updateByPrimaryKeySelective(update);

        // 定时任务
        addTask(param.getId());

        return Json.ok();
    }


    private void addTask(Long timingRedPacketPlanId) {

        CreateRedPacketTaskFacadeParam param = new CreateRedPacketTaskFacadeParam();
        param.setTimingRedPacketPlanId(timingRedPacketPlanId);

        // 演出两秒 因为如果是创建的话会导致消费方拿不到数据
        scheduledThreadPoolExecutor.schedule(() -> {

            if (log.isDebugEnabled()) {
                log.debug("创建id: {} 的红包定时任务", timingRedPacketPlanId);
            }

            // 创建红包
            boolean res = imIntegrationService.createRedPacketTask(param);
            if (!res) {
                log.warn("创建id: {}的红包定时任务失败！", timingRedPacketPlanId);
            }
        }, 2, TimeUnit.SECONDS);
    }

    private void removeTask(Long timingRedPacketPlanId) {

        RemoveRedPacketTaskFacadeParam param = new RemoveRedPacketTaskFacadeParam();
        param.setTimingRedPacketPlanId(timingRedPacketPlanId);

        // 移除红包
        boolean res = imIntegrationService.removeRedPacketTask(param);
        if (!res) {
            log.warn("移除id: {}的红包定时任务失败！", timingRedPacketPlanId);
        }
    }

    @Override
    public PageWarper<ImTimingRedPacketPlanVO> listPlansByChatRoomId(ListPlansByChatRoomIdParam param) {

        Page<ImTimingRedPacketPlan> page = PageHelper
                .startPage(param)
                .doSelectPage(() -> imTimingRedPacketPlanExtendMapper.listPlansByChatRoomId(param.getChatRoomId()));

        PageWarper pageWarper = new PageWarper(page);

        pageWarper.setList(page.getResult().stream()
                .map(this::build)
                .collect(Collectors.toList()));

        return pageWarper;
    }


    @Override
    public Json deleteTimingRedPacketPlan(IdParam param) {
        ImTimingRedPacketPlan update = new ImTimingRedPacketPlan();
        Date now = new Date();


        update.setId(param.getId());
        update.setDeleteTime(now);
        update.setDeleteFlag(DeleteFlag.DELETED);
        update.setModifyTime(now);

        imTimingRedPacketPlanExtendMapper.updateByPrimaryKeySelective(update);

        removeTask(param.getId());

        return Json.ok();
    }
}
