package com.bxm.localnews.admin.service.security.impl;

import com.bxm.localnews.admin.domain.*;
import com.bxm.localnews.admin.dto.AdminUserDTO;
import com.bxm.localnews.admin.dto.OrderInfoDTO;
import com.bxm.localnews.admin.dto.Tuple;
import com.bxm.localnews.admin.dto.UserInviteCollectDTO;
import com.bxm.localnews.admin.enums.OrderTypeEnum;
import com.bxm.localnews.admin.param.AdminUserParam;
import com.bxm.localnews.admin.param.UserActivationCardParam;
import com.bxm.localnews.admin.service.security.AdminUserService;
import com.bxm.localnews.admin.vo.security.AdminUser;
import com.bxm.localnews.admin.vo.security.AdminUserRole;
import com.bxm.localnews.admin.vo.security.UserParentInfoBean;
import com.bxm.localnews.common.constant.RespCode;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.Json;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.redis.impl.DefaultKeyGenerator;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.Message;
import com.bxm.newidea.component.vo.PageWarper;
import com.github.pagehelper.PageHelper;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import static com.bxm.newidea.component.tools.MD5Util.hgmd5;

@Service
@Slf4j
public class AdminUserServiceImpl implements AdminUserService {

    private AdminUserMapper adminUserMapper;

    private AdminRoleMapper adminRoleMapper;

    private RedisHashMapAdapter redisHashMapAdapter;

    private AdminUserRoleMapper adminUserRoleMapper;

    private UserVipMapper userVipMapper;

    private KeyGenerator tokenKey = DefaultKeyGenerator.build("admin", "cache", "user");

    private final UserInviteHistoryMapper userInviteHistoryMapper;

    @Autowired
    public AdminUserServiceImpl(RedisStringAdapter redisStringAdapter,
                                AdminRoleMapper adminRoleMapper,
                                RedisHashMapAdapter redisHashMapAdapter,
                                AdminUserRoleMapper adminUserRoleMapper,
                                UserVipMapper userVipMapper,
                                AdminUserMapper adminUserMapper,
                                UserInviteHistoryMapper userInviteHistoryMapper) {
        this.adminUserMapper = adminUserMapper;
        this.adminRoleMapper = adminRoleMapper;
        this.redisHashMapAdapter = redisHashMapAdapter;
        this.adminUserRoleMapper = adminUserRoleMapper;
        this.userVipMapper = userVipMapper;
        this.userInviteHistoryMapper = userInviteHistoryMapper;
    }


    @Override
    public AdminUser getUser(String userName) {
        return this.adminUserMapper.getByUserName(userName);
    }

    @Override
    public void removeToken(Long userId) {
        log.debug("移除用户token，userId:{}", userId);
        if (null != userId) {
            redisHashMapAdapter.remove(tokenKey, String.valueOf(userId));
        }
    }

    @Override
    public void addTokenCache(Long userId, String token) {
        log.debug("创建用户TOKEN，userId:{},token:{}", userId, token);
        if (null != userId && null != token) {
            redisHashMapAdapter.put(tokenKey, String.valueOf(userId), token);
        }
    }

    @Override
    public Json modifyPassword(AdminUserDTO param) {
        String oldPassword = hgmd5(param.getOldPassword());
        String newPassword = hgmd5(param.getNewPassword());

        AdminUser oldSysUser = this.adminUserMapper.getByUserName(param.getUsername());

        if (null == oldSysUser) {
            return ResultUtil.genFailedResult("用户不存在");
        }

        if (!oldSysUser.getPassword().equals(oldPassword)) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "原密码输入不正确");
        }
        if (oldSysUser.getPassword().equals(newPassword)) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "新密码不能和旧密码相同");
        }

        // 密码加密
        oldSysUser.setPassword(newPassword);
        this.editUserPassWord(oldSysUser);

        this.removeToken(oldSysUser.getId());
        return ResultUtil.genSuccessMsg("密码修改成功");

    }

    @Override
    public Json modifyPasswordByAdmin(AdminUserDTO param) {
        String newPassword = hgmd5(param.getNewPassword());
        AdminUser oldSysUser = this.adminUserMapper.getByUserName(param.getUsername());

        if (null == oldSysUser) {
            return ResultUtil.genFailedResult("用户不存在");
        }

        oldSysUser.setPassword(newPassword);
        this.editUserPassWord(oldSysUser);

        this.removeToken(oldSysUser.getId());
        return ResultUtil.genSuccessMsg("密码修改成功");
    }

    @Override
    public Boolean editUserPassWord(AdminUser sysUser) {
        this.removeToken(sysUser.getId());
        return this.adminUserMapper.editUserPassWord(sysUser);
    }

    @Override
    public PageWarper<AdminUserDTO> queryAdminUsers(AdminUserParam adminUserParam) {
        Preconditions.checkArgument(adminUserParam != null);

        PageHelper.startPage(adminUserParam.getPageNum(), adminUserParam.getPageSize());
        List<AdminUserDTO> list = this.adminUserMapper.queryAdminUser(adminUserParam);
        PageWarper<AdminUserDTO> pageWarper = new PageWarper<>(list);

        return pageWarper;
    }

    @Override
    public AdminUser selectByUsername(String username) {
        return this.adminUserMapper.selectByUsername(username);
    }

    @Override
    public int upsert(AdminUserDTO user) {
        String password = hgmd5(user.getPassword());
        user.setPassword(password);

        this.removeToken(user.getId());
        return this.adminUserMapper.upsert(user);
    }

    @Override
    public int updateByPrimaryKeySelective(AdminUser record) {
        String password = hgmd5(record.getPassword());
        record.setPassword(password);

        this.removeToken(record.getId());
        return this.adminUserMapper.updateByPrimaryKeySelective(record);
    }

    @Override
    public AdminUser selectById(Long id) {
        return this.adminUserMapper.selectById(id);
    }


    @Override
    public Message updateUserRole(Long userId, Integer roleId) {
        Long exitRoleId = adminRoleMapper.selectRoleIdByUserId(userId);
        AdminUserRole adminUserRole = new AdminUserRole();
        adminUserRole.setUserId(userId);
        adminUserRole.setRoleId(roleId);
        if (Objects.isNull(exitRoleId)) {
            adminUserRoleMapper.insertSelective(adminUserRole);
        } else {
            adminUserRoleMapper.updateRoleByUserId(adminUserRole);
        }
        return Message.build(true);
    }

    @Override
    public List<AdminUserDTO> getAvailableAdminUser() {
        AdminUserParam adminUserParam = new AdminUserParam();
        adminUserParam.setStatus(0);
        return adminUserMapper.queryAdminUser(adminUserParam);
    }

    @Override
    public PageWarper<OrderInfoDTO> getOpenCardPageList(UserActivationCardParam userActivationCardParam) {
        Preconditions.checkArgument(userActivationCardParam != null);
        userActivationCardParam.setPayType(OrderTypeEnum.WELFARE_VIP.getCode());
        PageHelper.startPage(userActivationCardParam.getPageNum(), userActivationCardParam.getPageSize());
        List<OrderInfoDTO> list = userVipMapper.getPageList(userActivationCardParam);

        // 根据用户ID获取上级信息
        List<Long> idList = list.stream().map(OrderInfoDTO::getUserId).collect(Collectors.toList());

        List<UserParentInfoBean> parentInfoList = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(idList)) {
            parentInfoList = userVipMapper.getParentInfoById(idList);
        }

        for (OrderInfoDTO item : list) {
            if (StringUtils.equals(item.getActivationType(), "0")) {
                item.setActivationType("付费开通");
            } else if (StringUtils.equals(item.getActivationType(), "1")) {
                item.setActivationType("激活码");
            } else if (StringUtils.equals(item.getActivationType(), "2")) {
                item.setActivationType("运营开通");
            } else {
                item.setActivationType("未知");
            }

            parentInfoList.forEach(info -> {
                if (Objects.equals(info.getUserId(), item.getUserId())) {
                    item.setParentNickName(info.getParentNickName());
                    item.setParentPhone(info.getParentPhone());
                }
            });

            // 邀请人数处理
            UserInviteCollectDTO collect = getCollect(item.getUserId());
            if (Objects.nonNull(collect)) {
                item.setTotal(collect.getTotal());
                item.setTodayTotal(collect.getTodayTotal());
                item.setYesterdayTotal(collect.getYesterdayTotal());
                item.setOneLevel(collect.getOneLevel());
                item.setTwoLevel(collect.getTwoLevel());
            }
        }

        return new PageWarper<>(list);
    }

    /**
     * 获取用户邀请徒弟的统计信息，包括今日、昨日、总共的数量
     *
     * @param userId 当前用户
     * @return 统计结果
     */
    private UserInviteCollectDTO getCollect(Long userId) {
        List<Tuple> collectList = userInviteHistoryMapper.getCollect(userId);


        String todayFormat = DateUtils.formatDate(new Date());

        Calendar yesterdayCalendar = Calendar.getInstance();
        yesterdayCalendar.setTime(new Date());
        yesterdayCalendar.add(Calendar.DAY_OF_YEAR, -1);
        String yesterdayFormat = DateUtils.formatDate(yesterdayCalendar.getTime());

        int total = 0, today = 0, yesterday = 0;

        for (Tuple tuple : collectList) {
            total += tuple.getTotal();

            if (StringUtils.equals(todayFormat, tuple.getLabel())) {
                today = tuple.getTotal();
            }
            if (StringUtils.equals(yesterdayFormat, tuple.getLabel())) {
                yesterday = tuple.getTotal();
            }
        }
        //一级徒弟人数
        Integer levelOne = userInviteHistoryMapper.getCollectLevel(userId, 1);
        UserInviteCollectDTO result = new UserInviteCollectDTO();
        result.setTodayTotal(today);
        result.setYesterdayTotal(yesterday);
        result.setTotal(total);
        result.setOneLevel(levelOne);
        result.setTwoLevel(total - levelOne);
        return result;
    }
}
