package com.bxm.localnews.base.service.impl;

import com.bxm.localnews.base.constant.PublishStateEnum;
import com.bxm.localnews.base.constant.VersionForceEnum;
import com.bxm.localnews.base.domain.AppVersionMapper;
import com.bxm.localnews.base.service.AppVersionService;
import com.bxm.localnews.common.config.BizConfigProperties;
import com.bxm.localnews.common.constant.AppConst;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.common.dto.AppVersionDTO;
import com.bxm.localnews.common.util.IPUtil;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.common.vo.IP;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.StringUtils;
import com.fasterxml.jackson.core.type.TypeReference;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * Created by Administrator on 2018/2/22 0022.
 */
@Service
@Transactional(rollbackFor = Exception.class)
@RefreshScope
public class AppVersionServiceImpl extends BaseService implements AppVersionService {

    private AppVersionMapper appVersionMapper;

    private BizConfigProperties bizConfigProperties;

    private RedisStringAdapter redisStringAdapter;

    private RedisHashMapAdapter redisHashMapAdapter;

    private IPUtil ipUtil;

    @Autowired
    public AppVersionServiceImpl(AppVersionMapper appVersionMapper, BizConfigProperties bizConfigProperties,
                                 RedisStringAdapter redisStringAdapter, RedisHashMapAdapter redisHashMapAdapter,
                                 IPUtil ipUtil) {
        this.appVersionMapper = appVersionMapper;
        this.bizConfigProperties = bizConfigProperties;
        this.redisStringAdapter = redisStringAdapter;
        this.redisHashMapAdapter = redisHashMapAdapter;
        this.ipUtil = ipUtil;
    }

    /**
     * 判断是否是美国的ip
     * @param ip
     * @return
     */
    private Boolean isAbroadOfCountry(String ip) {
        Boolean isAbroadCountry = redisHashMapAdapter.get(getCountryKey(), ip, Boolean.class);
        if (isAbroadCountry == null) {
            isAbroadCountry = Boolean.FALSE;

            IP i = ipUtil.find(ip);
            if (i == null || i.getCountry() == null) {
                isAbroadCountry = Boolean.TRUE;
            }
            if (i != null && "美国".equals(i.getCountry())) {
                isAbroadCountry = Boolean.TRUE;
                redisHashMapAdapter.put(getCountryKey(), ip, Boolean.TRUE);
            }
        }

        return isAbroadCountry;
    }

    private KeyGenerator getCountryKey() {
        return RedisConfig.BASE_IP.copy().appendKey("list");
    }

    @Override
    public boolean inWhiteList(String phone) {
        return bizConfigProperties.getWhiteList().contains(phone);
    }

    @Override
    public AppVersionDTO getAppVersion(BasicParam basicParam) {
        AppVersionDTO appVersionDTO = new AppVersionDTO();
        appVersionDTO.setMobileType(basicParam.getPlatform());
        appVersionDTO.setCode(basicParam.getChnl());

        List<AppVersionDTO> appVersionDTOS = getAllAppVersion();
        //筛选同一渠道的版本列表
        List<AppVersionDTO> filterAppVersionDTOs = appVersionDTOS.stream().filter(app ->
                app.getMobileType().equals(basicParam.getPlatform())
                        && app.getCode().equals(basicParam.getChnl())
        ).collect(Collectors.toList());

        //客户端版本---IOS1.2.0版本及其以前版本传入字段冲突
        String curVer = basicParam.getCurVer();
        if (basicParam.getPlatform() == AppConst.PLATFORM.IOS) {
            curVer = isHighVersion(basicParam.getVersion(), "1.2.0") == 1 ? basicParam.getCurVer() : basicParam.getVersion();
        }

        if (CollectionUtils.isEmpty(filterAppVersionDTOs)) {
            appVersionDTO.setVersion(curVer);
            appVersionDTO.setIsForce(VersionForceEnum.NOT_UPDATE.getState());
            return appVersionDTO;
        }


        //获取对应渠道最新的版本信息
        appVersionDTO = matchAllVersion(filterAppVersionDTOs);

        //客户端版本不低于服务端版本
        if (isHighVersion(curVer, appVersionDTO.getVersion()) != -1) {
            appVersionDTO.setVersion(curVer);
            appVersionDTO.setIsForce(VersionForceEnum.NOT_UPDATE.getState());
            return appVersionDTO;
        }

        //获取可用的最高版本信息
        List<AppVersionDTO> enableAppVersionDTOS = filterAppVersionDTOs.stream().filter(app -> app.getEnable() == 1).collect(Collectors.toList());

        //没有可用的高版本---就设置为当前版本
        if (CollectionUtils.isEmpty(enableAppVersionDTOS)) {
            appVersionDTO.setVersion(curVer);
            appVersionDTO.setIsForce(VersionForceEnum.NOT_UPDATE.getState());
            return appVersionDTO;
        }

        appVersionDTO = matchAllVersion(enableAppVersionDTOS);


        //客户端版本不低于服务端版本
        if (isHighVersion(curVer, appVersionDTO.getVersion()) != -1) {
            appVersionDTO.setVersion(curVer);
            appVersionDTO.setIsForce(VersionForceEnum.NOT_UPDATE.getState());
            return appVersionDTO;
        }

        if (VersionForceEnum.FORCE_UPDATE.getState().equals(appVersionDTO.getIsForce())) {
            if (isHighVersion(curVer, appVersionDTO.getForceVersion()) == 1) {
                appVersionDTO.setIsForce(VersionForceEnum.REMING_UPDATE.getState());
            } else {
                appVersionDTO.setIsForce(VersionForceEnum.FORCE_UPDATE.getState());
            }
        }

        //获取客户端对应的版本信息
        AppVersionDTO curVersion = null;
        for (AppVersionDTO app : appVersionDTOS) {
            if (app.getVersion().equals(curVer) && app.getCode().equals(basicParam.getChnl())) {
                curVersion = app;
            }
        }
        if (curVersion == null) {
            return appVersionDTO;
        }

        if (appVersionDTO.getVersion().equals(curVersion.getVersion())) {
            appVersionDTO.setVersion(curVer);
            appVersionDTO.setIsForce(VersionForceEnum.NOT_UPDATE.getState());
        }

        //设置客户端版本对应的状态
        appVersionDTO.setStatus(curVersion.getStatus());

        return appVersionDTO;
    }

    /**
     * 匹配对应最新的渠道版本（包括审核状态的版本）
     * @param appVersionDTOS
     * @return
     */
    private AppVersionDTO matchAllVersion(List<AppVersionDTO> appVersionDTOS) {
        AppVersionDTO appVersionDTO = appVersionDTOS.get(appVersionDTOS.size() - 1);
        if (appVersionDTOS.size() >= 2) {
            for (int i = appVersionDTOS.size() - 2; i >= 0; i--) {
                if (isHighVersion(appVersionDTOS.get(i).getVersion(), appVersionDTO.getVersion()) == 1) {
                    appVersionDTO = appVersionDTOS.get(i);
                }
            }
        }

        return appVersionDTO;
    }

    /**
     * 查询更新要求的版本
     * @param appVersionDTOS
     * @param appVersionDTO
     * @return
     */
    private AppVersionDTO queryAppVersion(List<AppVersionDTO> appVersionDTOS, AppVersionDTO appVersionDTO) {
        Optional<AppVersionDTO> optional = appVersionDTOS.stream().filter(app ->
                app.getVersion().equals(appVersionDTO.getForceVersion())).findFirst();
        if (optional.isPresent()) {
            return optional.get();
        }
        return null;
    }

    @Override
    public AppVersionDTO getAppVersion(BasicParam basicParam, String ip) {
        AppVersionDTO appVersionDTO = getAppVersion(basicParam);

        if (AppConst.PLATFORM.WEB == basicParam.getPlatform()) {
            appVersionDTO.setStatus(PublishStateEnum.NORMAL.getState());
            return appVersionDTO;
        }

        if (AppConst.PLATFORM.IOS == basicParam.getPlatform()
                && PublishStateEnum.NORMAL.getState().equals(appVersionDTO.getStatus())
                && isAbroadOfCountry(ip)) {
            appVersionDTO.setStatus(PublishStateEnum.PURSE.getState());
        }

        return appVersionDTO;
    }

    /**
     * 获取所有渠道的版本信息
     * @return
     */
    private List<AppVersionDTO> getAllAppVersion() {
        TypeReference<List<AppVersionDTO>> typeReference = new TypeReference<List<AppVersionDTO>>() {
        };
        List<AppVersionDTO> appVersionDTOS = redisStringAdapter.get(this.getAppChannelListKey(), typeReference);

        if (CollectionUtils.isEmpty(appVersionDTOS)) {
            appVersionDTOS = this.appVersionMapper.getAllAppVersion();
            redisStringAdapter.set(this.getAppChannelListKey(), appVersionDTOS);
        }

        return appVersionDTOS;
    }

    /**
     * 获取渠道版本列表的key
     * @return
     */
    private KeyGenerator getAppChannelListKey() {
        return RedisConfig.BASE_APPVERSION_CHANNEL.copy().appendKey("list");
    }

    @Override
    public int isHighVersion(String curVersion, String version) {
        //如果服务端版本高于客户端版本
        if (curVersion == null && version != null) {
            return -1;
        }
        //如果客户端版本高于服务端版本
        if (curVersion != null && version == null) {
            return 1;
        }
        //如果版本一致
        if (StringUtils.equals(curVersion, version)) {
            return 0;
        }
        //进行版本比较
        return StringUtils.compareVersion(curVersion, version);
    }
}
