package com.bxm.localnews.base.service.impl;

import java.util.List;
import java.util.stream.Collectors;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.base.domain.AreaCodeMapper;
import com.bxm.localnews.base.domain.AreaDivisionMapper;
import com.bxm.localnews.base.domain.LocationMapper;
import com.bxm.localnews.base.dto.LocationDetailDTO;
import com.bxm.localnews.base.service.LocationFacadeService;
import com.bxm.localnews.base.service.LocationService;
import com.bxm.localnews.common.config.BizConfigProperties;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.common.dto.AreaDivisionDTO;
import com.bxm.localnews.common.dto.CustomerServiceDTO;
import com.bxm.localnews.common.dto.LocationDTO;
import com.bxm.localnews.common.dto.LocationMeta;
import com.bxm.localnews.common.vo.AreaCode;
import com.bxm.localnews.common.vo.AreaDivision;
import com.bxm.localnews.common.vo.Location;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.StringUtils;
import com.fasterxml.jackson.core.type.TypeReference;

import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Service
public class LocationServiceImpl extends BaseService implements LocationService, LocationFacadeService {

    /**
     * 日志对象
     */
    private static final Logger LOG = LoggerFactory.getLogger(LocationServiceImpl.class);

    private static final int AREA_CODE_LENGTH = 12;

    private static final int AREA_CODE_HALF_LENGTH = 6;

    private final AreaCodeMapper areaCodeMapper;

    @Autowired
    private AreaDivisionMapper areaDivisionMapper;

    @Autowired
    private LocationMapper locationMapper;

    @Autowired
    private RedisStringAdapter redisStringAdapter;

    @Autowired
    private BizConfigProperties bizConfigProperties;

    /**
     * redis Hash服务类
     */
    @Autowired
    private RedisHashMapAdapter redisHashMapAdapter;

    @Autowired
    public LocationServiceImpl(AreaCodeMapper areaCodeMapper) {
        this.areaCodeMapper = areaCodeMapper;
    }


    @Override
    public AreaCode selectByAreaName(String name, String province) {
        List<AreaCode> areaCodeList = areaCodeMapper.selectByAreaName(province);
        if (areaCodeList != null) {
            for (AreaCode areaCode : areaCodeList) {
                String tail = areaCode.getAreaCode().substring(2, 6);
                if ("0000".equalsIgnoreCase(tail)) {
                    String codeSplit = areaCode.getAreaCode().substring(0, 2);
                    List<AreaCode> areaList = areaCodeMapper.selectByAreaCode(name, codeSplit);
                    if (areaList != null) {
                        return areaList.get(0);
                    }
                }
            }

        }
        return null;
    }

    @Override
    public LocationMeta listLocation() {
        KeyGenerator hotCityKey = RedisConfig.HOT_LOCATION;
        TypeReference<List<LocationDTO>> hotCityTypeReference = new TypeReference<List<LocationDTO>>() {
        };
        List<LocationDTO> hotLocationList = redisStringAdapter.get(hotCityKey, hotCityTypeReference);
        if (hotLocationList == null) {
            hotLocationList = listHotLocation();
            redisStringAdapter.set(hotCityKey, hotLocationList);
        }

        KeyGenerator openedCityKey = RedisConfig.OPENED_LOCATION;
        TypeReference<List<LocationDTO>> openedCityTypeReference = new TypeReference<List<LocationDTO>>() {
        };
        List<LocationDTO> openedLocationList = redisStringAdapter.get(openedCityKey, openedCityTypeReference);
        if (openedLocationList == null) {
            openedLocationList = listOpenedLocation();
            redisStringAdapter.set(openedCityKey, openedLocationList);
        }

        LocationMeta locationMeta = new LocationMeta();
        locationMeta.setHotLocation(hotLocationList);
        locationMeta.setOpenedLocation(openedLocationList);

        return locationMeta;
    }

    @Override
    public List<AreaDivisionDTO> listAreaDivisionDTO(String parentCode) {
        AreaDivision areaDivisionParam = new AreaDivision();
        areaDivisionParam.setParentCode(parentCode);
        List<AreaDivision> areaDivisionList = areaDivisionMapper.selectByModel(areaDivisionParam);
        return areaDivisionList.stream().map(this::convertDivision).collect(Collectors.toList());
    }

    @Override
    public com.bxm.localnews.base.dto.LocationFacadeDTO getInternalLocationByGeocode(String code) {
        com.bxm.localnews.base.dto.LocationFacadeDTO locationDTO = null;
        //直接以12位编码查询
        if (StringUtils.isNotEmpty(code)) {
            if (AREA_CODE_LENGTH == code.length()) {
                locationDTO = this.getLocationByCode(code);
                if (null == locationDTO) {
                    code = code.substring(0, 6);
                }
            }

            if (AREA_CODE_HALF_LENGTH == code.length()) {
                //当定位时取得的是6位城市编码
                String fullCode = code + "000000";
                locationDTO = this.getLocationByCode(fullCode);
                //如果是区查询上一级
                if (null == locationDTO) {
                    fullCode = code.substring(0, 4) + "00000000";
                    locationDTO = this.getLocationByCode(fullCode);
                }
            }
        }
        //默认 在配置文件中配置
        if (null == locationDTO) {
            locationDTO = new com.bxm.localnews.base.dto.LocationFacadeDTO();
            locationDTO.setCode(bizConfigProperties.getLocationCode());
            locationDTO.setName(bizConfigProperties.getLocationName());
        }
        return locationDTO;
    }

    private AreaDivisionDTO convertDivision(AreaDivision areaDivision) {
        AreaDivisionDTO areaDivisionDTO = new AreaDivisionDTO();
        areaDivisionDTO.setCode(areaDivision.getCode());
        areaDivisionDTO.setLevel(areaDivision.getLevel());
        areaDivisionDTO.setName(areaDivision.getName());
        return areaDivisionDTO;
    }

    private List<LocationDTO> listHotLocation() {
        Location locationParam = new Location();
        locationParam.setHot((byte) 1);
        locationParam.setShow((byte) 1);
        List<Location> locationList = locationMapper.selectByModel(locationParam);
        List<LocationDTO> locationDTOList = locationList.stream().map(this::convertLocation).collect(Collectors.toList());
        return locationDTOList;
    }

    private List<LocationDTO> listOpenedLocation() {
        Location locationParam = new Location();
        locationParam.setShow((byte) 1);
        List<Location> locationList = locationMapper.selectByModel(locationParam);


        List<LocationDTO> locationDTOList = locationList.stream().map(this::convertLocation).collect(Collectors.toList());
        //填充首字母
        locationDTOList.forEach(x -> x.setPinyin(x.getPinyin().substring(0, 1).toUpperCase()));

        return locationDTOList;
    }

    private LocationDTO convertLocation(Location location) {
        LocationDTO locationDTO = new LocationDTO();
        locationDTO.setCode(location.getCode());
        locationDTO.setHot(location.getHot());
        locationDTO.setLevel(location.getLevel());
        locationDTO.setName(location.getName());
        locationDTO.setPinyin(location.getPinyin().substring(0, 1).toUpperCase());
        return locationDTO;
    }

    @Override
    public com.bxm.localnews.base.dto.LocationFacadeDTO getLocationByCode(String code) {
        com.bxm.localnews.base.dto.LocationFacadeDTO locationDTO = null;
        Location location = locationMapper.selectByCode(code);
        if (null != location) {
            locationDTO = new com.bxm.localnews.base.dto.LocationFacadeDTO();
            BeanUtils.copyProperties(location, locationDTO);
        }

        return locationDTO;
    }

    @Override
    public LocationDetailDTO getLocationDetailByCode(String code) {
        com.bxm.localnews.base.dto.LocationDetailDTO locationDTO = null;
        Location location = locationMapper.selectByCode(code);
        if (null != location) {
            locationDTO = new com.bxm.localnews.base.dto.LocationDetailDTO();
            BeanUtils.copyProperties(location, locationDTO);
        }

        return locationDTO;
    }

    @Override
    public LocationDetailDTO getAcitivateVipLocation() {
        com.bxm.localnews.base.dto.LocationDetailDTO locationDTO = null;
        Location location = locationMapper.selectActivateLocation();
        if (null != location) {
            locationDTO = new com.bxm.localnews.base.dto.LocationDetailDTO();
            BeanUtils.copyProperties(location, locationDTO);
        }
        return locationDTO;
    }

    @Override
    public boolean compareLocationCode(String originalCode, List<String> standardCodeList) {
        logger.debug("参加活动地区编码相比较，用户地区编号：{} ，活动地区列表：{}", originalCode, standardCodeList);
        //直接以12位编码查询
        if (null != originalCode && CollectionUtils.isNotEmpty(standardCodeList)) {
            String fullCode = originalCode;
            if (AREA_CODE_LENGTH != fullCode.length()) {
                //当定位时取得的是6位城市编码
                fullCode = originalCode + "000000";
                com.bxm.localnews.base.dto.LocationFacadeDTO locationDTO = getLocationByCode(fullCode);
                //如果是区是查不到的，查询上一级
                if (null == locationDTO) {
                    fullCode = originalCode.substring(0, 4) + "00000000";
                }
            }
            if (standardCodeList.contains(fullCode)) {
                return true;
            }
        }
        return false;
    }

    @Override
    public com.bxm.localnews.base.dto.LocationFacadeDTO getLocationByGeocode(String code) {
        return this.getInternalLocationByGeocode(code);
    }


    @Override
    public CustomerServiceDTO getCustomerServiceInfo(String code) {

        //从缓存中获取
        CustomerServiceDTO customerInfoCache = redisHashMapAdapter.get(RedisConfig.CUSTOMER_SERVICE_INFO, code, CustomerServiceDTO.class);
        if (null != customerInfoCache) {
            LOG.info("以地区编码:{}从缓存中获取到的客服信息为:{}", code, JSON.toJSONString(customerInfoCache));
            return customerInfoCache;
        }

        CustomerServiceDTO customerServiceDTO = new CustomerServiceDTO();
        //缓存中未获取到则从数据库中获取
        Location location = locationMapper.selectByCode(code);
        //若该地区为开通 则也是给全国默认客服信息
        if (null == location) {
            logger.info("根据区域编码:{}获取的地区信息为空,返回默认的客服信息", code);
            customerServiceDTO.setCustomerServiceNumber(bizConfigProperties.getCustomerServiceNumber());
            return customerServiceDTO;
        }


        //若有该地区信息,但是客服微信号或者客服二维码为空,则设置默认的客服微信号、二维码
        if (StringUtils.isBlank(location.getCustomerServiceNumber())) {
            //从配置中获取默认的客服微信号和客服二维码
            customerServiceDTO.setCustomerServiceNumber(bizConfigProperties.getCustomerServiceNumber());
            customerServiceDTO.setCustomerServiceQrCode(location.getCustomerServiceQrCode());
        }else {
            customerServiceDTO.setCustomerServiceNumber(location.getCustomerServiceNumber());
            customerServiceDTO.setCustomerServiceQrCode(location.getCustomerServiceQrCode());
        }

        //数据库查询完设置缓存值
        redisHashMapAdapter.put(RedisConfig.CUSTOMER_SERVICE_INFO, code, customerServiceDTO);

        return customerServiceDTO;
    }

    @Override
    public List<String> listAllLocationWithCommunity() {
        return locationMapper.selectLocationWithCommunity();
    }


}
