package com.bxm.localnews.base.service.impl;

import com.bxm.localnews.base.domain.AppVersionMapper;
import com.bxm.localnews.base.service.AppVersionService;
import com.bxm.localnews.common.config.BizConfigProperties;
import com.bxm.localnews.common.constant.AppConst;
import com.bxm.localnews.common.constant.PublishStateEnum;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.common.constant.VersionForceEnum;
import com.bxm.localnews.common.dto.AppVersionDTO;
import com.bxm.localnews.common.util.IPUtil;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.common.vo.IP;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.StringUtils;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * Created by Administrator on 2018/2/22 0022.
 */
@Service
@Transactional(rollbackFor = Exception.class)
public class AppVersionServiceImpl extends BaseService implements AppVersionService {

    private AppVersionMapper appVersionMapper;

    private BizConfigProperties bizConfigProperties;

    private RedisStringAdapter redisStringAdapter;

    private RedisHashMapAdapter redisHashMapAdapter;

    private IPUtil ipUtil;

    private LoadingCache<String, List<AppVersionDTO>> cache;

    @Autowired
    public AppVersionServiceImpl(AppVersionMapper appVersionMapper,
                                 BizConfigProperties bizConfigProperties,
                                 RedisStringAdapter redisStringAdapter,
                                 RedisHashMapAdapter redisHashMapAdapter,
                                 IPUtil ipUtil) {
        this.appVersionMapper = appVersionMapper;
        this.bizConfigProperties = bizConfigProperties;
        this.redisStringAdapter = redisStringAdapter;
        this.redisHashMapAdapter = redisHashMapAdapter;
        this.ipUtil = ipUtil;

        initCache();
    }

    /**
     * 判断是否是美国的ip
     *
     * @param ip 请求IP
     * @return true表示为美国IP，主要判断iOS审核状态
     */
    private Boolean isAbroadOfCountry(String ip) {
        if (StringUtils.isNoneEmpty(ip)) {
            Boolean isAbroadCountry = redisHashMapAdapter.get(getCountryKey(), ip, Boolean.class);
            if (isAbroadCountry == null) {
                isAbroadCountry = Boolean.FALSE;

                IP i = ipUtil.find(ip);
                if (i == null || i.getCountry() == null) {
                    isAbroadCountry = Boolean.FALSE;
                } else if ("美国".equals(i.getCountry())) {
                    isAbroadCountry = Boolean.TRUE;
                    redisHashMapAdapter.put(getCountryKey(), ip, Boolean.TRUE);
                }
            }

            return isAbroadCountry;
        }

        return true;
    }

    private KeyGenerator getCountryKey() {
        return RedisConfig.BASE_IP.copy().appendKey("list");
    }

    @Override
    public boolean inWhiteList(String phone) {
        return bizConfigProperties.getWhiteList().contains(phone);
    }

    @Override
    public AppVersionDTO getAppVersion(BasicParam basicParam) {
        //筛选同一渠道的版本列表
        String combineChannel = basicParam.getPlatform() + basicParam.getChnl();
        List<AppVersionDTO> filterVersionList = cache.getUnchecked(combineChannel);

        //客户端版本
        String curVer = basicParam.getCurVer();

        AppVersionDTO appVersionDTO = new AppVersionDTO();
        appVersionDTO.setMobileType(basicParam.getPlatform());
        appVersionDTO.setCode(basicParam.getChnl());
        appVersionDTO.setVersion(curVer);
        appVersionDTO.setIsForce(VersionForceEnum.NOT_UPDATE.getState());


        // 如果该平台渠道不存在对应的版本，则直接返回
        if (CollectionUtils.isEmpty(filterVersionList)) {
            return appVersionDTO;
        }

        //获取对应渠道最新的版本信息（包括审核状态）
        appVersionDTO = getLastVersionClone(filterVersionList);

        // 客户端版本高于或等于当前配置的最新版本，则直接返回
        if (isHighVersion(curVer, appVersionDTO.getVersion()) != -1) {
            appVersionDTO.setIsForce(VersionForceEnum.NOT_UPDATE.getState());
            return appVersionDTO;
        }

        // 获取可用的最高版本信息
        List<AppVersionDTO> enableHigherVersionList = filterVersionList.stream().filter(app -> app.getEnable() == 1)
                .collect(Collectors.toList());

        //没有可用的高版本则不升级
        if (CollectionUtils.isEmpty(enableHigherVersionList)) {
            appVersionDTO.setVersion(curVer);
            appVersionDTO.setIsForce(VersionForceEnum.NOT_UPDATE.getState());
            return appVersionDTO;
        }

        appVersionDTO = getLastVersionClone(enableHigherVersionList);

        // 客户端版本比当前渠道最高版本还新，则啥也不做
        if (isHighVersion(curVer, appVersionDTO.getVersion()) != -1) {
            appVersionDTO.setVersion(curVer);
            appVersionDTO.setIsForce(VersionForceEnum.NOT_UPDATE.getState());
            return appVersionDTO;
        }

        Byte updateType = getUpdateType(curVer, enableHigherVersionList);
        appVersionDTO.setIsForce(updateType);

        return appVersionDTO;
    }

    /**
     * 获取当前版本对应的升级方式
     * 如果当前渠道存在高于当前版本的配置，则判断以下情况
     * 1.在当前版本到最新版本，是否存在强制升级
     * 2. 不存在强制升级，则使用当前最新版本的升级方式
     *
     * @param curVer                  当前客户端版本
     * @param enableHigherVersionList 当前客户端平台、渠道对应的可选升级版本
     * @return 具体的升级方式
     */
    private Byte getUpdateType(String curVer, List<AppVersionDTO> enableHigherVersionList) {
        Byte updateType = enableHigherVersionList.get(0).getIsForce();

        for (AppVersionDTO appVersionDTO : enableHigherVersionList) {
            if (StringUtils.compareVersion(appVersionDTO.getVersion(), curVer) <= 0) {
                break;
            }

            // 从最新版本到当前版本，是否出现过强制升级
            if (Objects.equals(VersionForceEnum.FORCE_UPDATE.getState(), appVersionDTO.getIsForce())) {
                updateType = appVersionDTO.getIsForce();
                break;
            }
        }

        return updateType;
    }

    /**
     * 匹配对应最新的渠道版本（包括审核状态的版本）
     *
     * @param appVersionList 当前渠道的版本信息列表
     * @return 当前渠道的最新版本
     */
    private AppVersionDTO getLastVersionClone(List<AppVersionDTO> appVersionList) {
        AppVersionDTO appVersionDTO = appVersionList.get(0);
        AppVersionDTO cloneVersionDTO = new AppVersionDTO();

        BeanUtils.copyProperties(appVersionDTO, cloneVersionDTO);
        return cloneVersionDTO;
    }

    @Override
    public AppVersionDTO getAppVersion(BasicParam basicParam, String ip) {
        AppVersionDTO appVersionDTO = getAppVersion(basicParam);

        if (AppConst.PLATFORM.WEB == basicParam.getPlatform()) {
            appVersionDTO.setStatus(PublishStateEnum.NORMAL.getState());
            return appVersionDTO;
        }

        // 如果是ios，就算是正常状态，但是IP在美国，一样返回审核模式
        if (AppConst.PLATFORM.IOS == basicParam.getPlatform()
                && PublishStateEnum.NORMAL.getState().equals(appVersionDTO.getStatus())
                && isAbroadOfCountry(ip)) {
            appVersionDTO.setStatus(PublishStateEnum.PURSE.getState());
        }

        return appVersionDTO;
    }

    /**
     * 根据平台和渠道获取对应的版本升级列表
     *
     * @param platformAndChannel 平台和渠道组合编码
     * @return 获取当前所有的版本配置信息
     */
    private List<AppVersionDTO> getAllAppVersion(String platformAndChannel) {
        TypeReference<List<AppVersionDTO>> typeReference = new TypeReference<List<AppVersionDTO>>() {
        };
        List<AppVersionDTO> appVersionList = redisStringAdapter.get(getAppChannelListKey(), typeReference);

        if (CollectionUtils.isEmpty(appVersionList)) {
            appVersionList = appVersionMapper.getAllAppVersion();
            redisStringAdapter.set(getAppChannelListKey(), appVersionList);
        }

        List<AppVersionDTO> result = Lists.newArrayList();

        for (AppVersionDTO appVersionDTO : appVersionList) {
            String combineChannel = appVersionDTO.getMobileType() + appVersionDTO.getCode();

            if (StringUtils.equals(combineChannel, platformAndChannel)) {
                result.add(appVersionDTO);
            }
        }

        // 进行排序
        result = result.stream().sorted((v1, v2) -> -StringUtils.compareVersion(v1.getVersion(), v2.getVersion()))
                .collect(Collectors.toList());

        return result;
    }

    private void initCache() {
        cache = CacheBuilder.newBuilder()
                .maximumSize(100)
                .expireAfterWrite(30, TimeUnit.SECONDS)
                .build(new CacheLoader<String, List<AppVersionDTO>>() {
                    @Override
                    public List<AppVersionDTO> load(String key) throws Exception {
                        return getAllAppVersion(key);
                    }
                });
    }

    /**
     * 获取渠道版本列表的key
     *
     * @return 获取版本配置信息的redis key
     */
    private KeyGenerator getAppChannelListKey() {
        return RedisConfig.BASE_APPVERSION_CHANNEL.copy().appendKey("list");
    }

    @Override
    public int isHighVersion(String curVersion, String version) {
        //如果服务端版本高于客户端版本
        if (curVersion == null && version != null) {
            return -1;
        }
        //如果客户端版本高于服务端版本
        if (curVersion != null && version == null) {
            return 1;
        }
        //如果版本一致
        if (StringUtils.equals(curVersion, version)) {
            return 0;
        }
        //进行版本比较
        return StringUtils.compareVersion(curVersion, version);
    }
}
