package com.bxm.localnews.base.service.impl;

import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.JSON;
import com.bxm.localnews.base.domain.AreaCodeMapper;
import com.bxm.localnews.base.domain.AreaDivisionMapper;
import com.bxm.localnews.base.domain.LocationMapper;
import com.bxm.localnews.base.service.LocationFacadeService;
import com.bxm.localnews.base.service.LocationService;
import com.bxm.localnews.common.config.BizConfigProperties;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.common.dto.*;
import com.bxm.localnews.common.param.ListAreaLocationParam;
import com.bxm.localnews.common.param.LocationDetailCacheParam;
import com.bxm.localnews.common.vo.*;
import com.bxm.localnews.dto.UserInfoDTO;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.LocationUtils;
import com.bxm.newidea.component.tools.StringUtils;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import static com.alibaba.fastjson.JSON.toJSON;
import static com.bxm.localnews.common.constant.RedisConfig.OPENED_LOCATION;
import static com.bxm.localnews.common.constant.RedisConfig.OPERATION_LOCATION;
import static org.apache.commons.lang3.StringUtils.isBlank;

@Service
public class LocationServiceImpl extends BaseService implements LocationService, LocationFacadeService {

    /**
     * 日志对象
     */
    private static final Logger LOG = LoggerFactory.getLogger(LocationServiceImpl.class);

    private static final int AREA_CODE_LENGTH = 12;

    private static final int AREA_CODE_HALF_LENGTH = 6;

    private final AreaCodeMapper areaCodeMapper;

    private final AreaDivisionMapper areaDivisionMapper;

    private final LocationMapper locationMapper;

    private final BizConfigProperties bizConfigProperties;

    private final UserIntegrationService userIntegrationService;

    /**
     * redis Hash服务类
     */
    private final RedisHashMapAdapter redisHashMapAdapter;

    private LoadingCache<KeyGenerator, List<LocationDTO>> cache;

    private LoadingCache<LocationDetailCacheParam, LocationDetailDTO> singleCache;

    @Autowired
    public LocationServiceImpl(AreaCodeMapper areaCodeMapper,
                               AreaDivisionMapper areaDivisionMapper,
                               LocationMapper locationMapper,
                               BizConfigProperties bizConfigProperties,
                               UserIntegrationService userIntegrationService,
                               RedisHashMapAdapter redisHashMapAdapter) {
        this.areaCodeMapper = areaCodeMapper;
        this.areaDivisionMapper = areaDivisionMapper;
        this.locationMapper = locationMapper;
        this.bizConfigProperties = bizConfigProperties;
        this.userIntegrationService = userIntegrationService;
        this.redisHashMapAdapter = redisHashMapAdapter;
    }

    @Override
    public AreaCode selectByAreaName(String name, String province) {
        List<AreaCode> areaCodeList = areaCodeMapper.selectByAreaName(province);
        if (areaCodeList != null) {
            for (AreaCode areaCode : areaCodeList) {
                String tail = areaCode.getAreaCode().substring(2, 6);
                if ("0000".equalsIgnoreCase(tail)) {
                    String codeSplit = areaCode.getAreaCode().substring(0, 2);
                    List<AreaCode> areaList = areaCodeMapper.selectByAreaCode(name, codeSplit);
                    if (areaList != null) {
                        return areaList.get(0);
                    }
                }
            }

        }
        return null;
    }

    @PostConstruct
    public void init() {
        // 已开通城市 or 热门城市简略信息缓存
        cache = CacheBuilder.newBuilder()
                .expireAfterWrite(10, TimeUnit.MINUTES)
                .maximumSize(3000)
                .build(new CacheLoader<KeyGenerator, List<LocationDTO>>() {
                    @Override
                    public List<LocationDTO> load(KeyGenerator key) {
                        if (RedisConfig.HOT_LOCATION.equals(key)) {
                            return listHotLocation();
                        } else if (OPERATION_LOCATION.equals(key)) {
                            return listOperationLocation();
                        } else {
                            return listOpenedLocation();
                        }
                    }
                });

        // 根据code获取城市的缓存
        singleCache = CacheBuilder.newBuilder()
                .expireAfterWrite(10, TimeUnit.MINUTES)
                .maximumSize(3000)
                .build(new CacheLoader<LocationDetailCacheParam, LocationDetailDTO>() {
                    @Override
                    public LocationDetailDTO load(LocationDetailCacheParam param) {
                        if (logger.isDebugEnabled()) {
                            logger.debug("根据code获取城市信息，请求参数: {}", JSON.toJSONString(param));
                        }

                        String code = param.getCode();
                        LocationDetailDTO locationDTO = new LocationDetailDTO();
                        locationDTO.setCode(code);
                        //直接以12位编码查询
                        if (StringUtils.isNotEmpty(code)) {
                            if (AREA_CODE_LENGTH == code.length()) {
                                locationDTO = getLocationDetailByCode(code, param.isLastCodeQuery());
                                if (null == locationDTO) {
                                    code = code.substring(0, 6);
                                }
                            }

                            if (AREA_CODE_HALF_LENGTH == code.length()) {
                                //当定位时取得的是6位城市编码
                                String fullCode = code + "000000";
                                LocationDetailDTO locationDTOTemp = getLocationDetailByCode(fullCode, param.isLastCodeQuery());
                                // 指定了当前code查不到时查询上一级
                                if (Objects.isNull(locationDTOTemp)) {
                                    if (param.isQueryParent()) {
                                        fullCode = code.substring(0, 4) + "00000000";
                                        locationDTO = getLocationDetailByCode(fullCode);
                                    }
                                } else {
                                    // 缓存不允许为空 这里用一个临时变量记住
                                    locationDTO = locationDTOTemp;
                                }
                            }
                        }
                        return locationDTO;
                    }
                });
    }

    @Override
    public LocationMeta listLocation(ListAreaLocationParam param) {
        Long userId = param.getUserId();

        LocationMeta locationMeta = new LocationMeta();
        //已开通城市列表
        locationMeta.setOpenedLocation(cache.getUnchecked(OPENED_LOCATION));

        // 如果未带经纬度
        if (Objects.isNull(param.getLat()) || Objects.isNull(param.getLon())) {
            // 热门城市列表
            locationMeta.setHotLocation(cache.getUnchecked(RedisConfig.HOT_LOCATION));

        } else {
            // 带了经纬度 使用经纬度计算出现有的开通的城市中最近的N个
            locationMeta.setHotLocation(cache.getUnchecked(OPERATION_LOCATION).stream()
                    // 排序
                    .sorted((p1, p2) -> {
                        // 分别获取当前位置距离已开通城市位置的距离
                        double l1 = LocationUtils.getDistance(param.getLat(), param.getLon(), p1.getLat(), p1.getLng());
                        double l2 = LocationUtils.getDistance(param.getLat(), param.getLon(), p2.getLat(), p2.getLng());
                        p1.setDistance(l1);
                        p2.setDistance(l2);
                        // 比较
                        return Double.compare(l1, l2);
                    })
                    // 截取显示的数量
                    .limit(bizConfigProperties.getMaxNearbyCityRecommendNum())
                    .collect(Collectors.toList()));

        }

        //用户家乡定位信息
        if (null != userId) {
            UserInfoDTO userInfoDTO = userIntegrationService.getUserFromRedisDB(userId);
            if (null != userInfoDTO && null != userInfoDTO.getHometownCode()) {
                LocationDTO locationDTO = new LocationDTO();
                locationDTO.setCode(userInfoDTO.getHometownCode());
                //从缓存获取简拼
                LocationDetailDTO location = singleCache.getUnchecked(LocationDetailCacheParam.builder()
                        .code(userInfoDTO.getHometownCode())
                        .build());
                locationDTO.setName(location.getName());

                locationMeta.setHometown(locationDTO);
            } else {
                logger.warn("获取城市信息时，用户不存在：{}", userId);
            }
        }

        return locationMeta;
    }

    @Override
    public List<AreaDivisionDTO> listAreaDivisionDTO(String parentCode) {
        AreaDivision areaDivisionParam = new AreaDivision();
        areaDivisionParam.setParentCode(parentCode);
        List<AreaDivision> areaDivisionList = areaDivisionMapper.selectByModel(areaDivisionParam);
        return areaDivisionList.stream().map(this::convertDivision).collect(Collectors.toList());
    }

    @Override
    public LocationDetailDTO getInternalLocationByGeocode(String code, BasicParam basicParam) {
        // 默认继续查询上一级
        return getInternalLocationByGeocode(code, basicParam, true, true, true);
    }

    @Override
    public LocationDetailDTO getInternalLocationByGeocode(String code, BasicParam basicParam, boolean queryParent,
                                                          boolean queryDefault, boolean lastCodeQuery) {

        LocationDetailDTO locationInfo = singleCache.getUnchecked(LocationDetailCacheParam.builder()
                .code(code)
                .queryParent(queryParent)
                // 是否使用code 查询最新城市编码
                .lastCodeQuery(lastCodeQuery)
                .build());

        // 如果查询不到城市 则根据参数确定是否进行默认定位
        if ((null == locationInfo || isBlank(locationInfo.getName()))) {
            logger.warn("用户定位失败, 是否填充默认定位城市: {}, [{}}],原城市:[{}],参数:[{}]",
                    queryDefault,
                    bizConfigProperties.getLocationCode(),
                    code,
                    toJSON(basicParam));

            if (queryDefault) {
                // 默认 在配置文件中配置
                locationInfo = singleCache.getUnchecked(LocationDetailCacheParam.builder()
                        .code(bizConfigProperties.getLocationCode())
                        .queryParent(true)
                        .lastCodeQuery(false)
                        .build());
            }
        }
        return locationInfo;
    }

    private AreaDivisionDTO convertDivision(AreaDivision areaDivision) {
        AreaDivisionDTO areaDivisionDTO = new AreaDivisionDTO();
        areaDivisionDTO.setCode(areaDivision.getCode());
        areaDivisionDTO.setLevel(areaDivision.getLevel());
        areaDivisionDTO.setName(areaDivision.getName());
        return areaDivisionDTO;
    }

    private List<LocationDTO> listHotLocation() {
        Location locationParam = new Location();
        locationParam.setHot((byte) 1);
        locationParam.setShow((byte) 1);
        List<Location> locationList = locationMapper.selectByModel(locationParam);
        return locationList.stream().map(this::convertLocation).collect(Collectors.toList());
    }

    /**
     * 查询运营进行管理的城市
     *
     * @return 城市列表
     */
    private List<LocationDTO> listOperationLocation() {
        Location locationParam = new Location();
        locationParam.setEnableCommunityContent(1);

        List<Location> locationList = locationMapper.selectByModel(locationParam);
        return locationList.stream().map(this::convertLocation).collect(Collectors.toList());
    }

    /**
     * 查询已开通信息的简略信息
     *
     * @return
     */
    private List<LocationDTO> listOpenedLocation() {
        List<Location> locationList = listOpenedLocationDetail();

        List<LocationDTO> locationDTOList = locationList.stream().map(this::convertLocation).collect(Collectors.toList());
        //填充首字母
        locationDTOList.forEach(x -> x.setPinyin(x.getPinyin().substring(0, 1).toUpperCase()));

        return locationDTOList;
    }

    /**
     * 查询允许选择的详细信息
     *
     * @return
     */
    private List<Location> listOpenedLocationDetail() {
        Location locationParam = new Location();
        locationParam.setShow((byte) 1);
        return locationMapper.selectByModel(locationParam);
    }

    private LocationDTO convertLocation(Location location) {
        LocationDTO locationDTO = new LocationDTO();
        locationDTO.setCode(location.getCode());
        locationDTO.setHot(location.getHot());
        locationDTO.setLevel(location.getLevel());
        locationDTO.setName(location.getName());
        locationDTO.setPinyin(location.getPinyin().substring(0, 1).toUpperCase());
        locationDTO.setLastCode(location.getLastCode());
        locationDTO.setOrder(location.getOrder());
        locationDTO.setLat(location.getLat());
        locationDTO.setLng(location.getLng());

        return locationDTO;
    }

    @Override
    public LocationDetailDTO getLocationDetailByCode(String code) {
        return getLocationDetailByCode(code, false);
    }

    @Override
    public LocationDetailDTO getLocationDetailByCode(String code, boolean lastCodeQuery) {
        LocationDetailDTO locationDTO = null;
        Location location = locationMapper.selectByCode(code, lastCodeQuery);
        if (null != location) {
            locationDTO = new LocationDetailDTO();
            BeanUtils.copyProperties(location, locationDTO);
        }

        return locationDTO;
    }


    @Override
    public LocationDetailDTO getAcitivateVipLocation() {
        LocationDetailDTO locationDTO = null;
        Location location = locationMapper.selectActivateLocation();
        if (null != location) {
            locationDTO = new LocationDetailDTO();
            BeanUtils.copyProperties(location, locationDTO);
        }
        return locationDTO;
    }

    @Override
    public boolean compareLocationCode(String originalCode, List<String> standardCodeList) {
        logger.debug("参加活动地区编码相比较，用户地区编号：{} ，活动地区列表：{}", originalCode, standardCodeList);
        //直接以12位编码查询
        if (null != originalCode && CollectionUtils.isNotEmpty(standardCodeList)) {
            String fullCode = completeAreaCode(originalCode);
            return standardCodeList.contains(fullCode);
        }
        return false;
    }

    @Override
    public String getLocationByName(String name) {
        return locationMapper.selectCodeByName(name);
    }

    @Override
    public void update(Location record) {
        locationMapper.updateByCodeSelective(record);
        cache.cleanUp();
        singleCache.cleanUp();
    }

    @Override
    public CustomerServiceDTO getCustomerServiceInfo(String code, Byte type) {
        //如果不传类型，则说明走的是默认的获取客服信息而不是公众号信息
        if (Objects.isNull(type)) {
            return getCustomerInfo(code);
        } else {
            return getOfficialAccountInfo(code, type);
        }
    }


    private CustomerServiceDTO getCustomerInfo(String code) {
        //从缓存中获取
        CustomerServiceDTO customerInfoCache = redisHashMapAdapter.get(RedisConfig.CUSTOMER_SERVICE_INFO, code, CustomerServiceDTO.class);
        if (null != customerInfoCache) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("以地区编码:{}从缓存中获取到的客服信息为:{}", code, JSON.toJSONString(customerInfoCache));
            }
            return customerInfoCache;
        }
        CustomerServiceDTO customerServiceDTO = new CustomerServiceDTO();
        //缓存中未获取到则从数据库中获取
        Location location = locationMapper.selectCustomerServiceByCode(code);
        //若该地区为未开通或者该地区开通但关闭客户助手 则给全国默认客服信息
        if (null == location) {
            logger.debug("根据区域编码:{}获取的地区信息为空,返回默认的客服信息", code);
            customerServiceDTO.setCustomerServiceNumber(bizConfigProperties.getCustomerServiceNumber());
            String customerServicePhone = getGroupCustomerPhoneServiceInfo(code).getGroupCustomerServicePhone();
            customerServiceDTO.setPhone(customerServicePhone);
            return customerServiceDTO;
        }
        //若有该地区信息,但是客服微信号或者客服二维码为空,则设置默认的客服微信号、二维码
        if (isBlank(location.getCustomerServiceNumber())) {
            //从配置中获取默认的客服微信号和客服二维码
            customerServiceDTO.setCustomerServiceNumber(bizConfigProperties.getCustomerServiceNumber());
            customerServiceDTO.setCustomerServiceQrCode(location.getCustomerServiceQrCode());
        } else {
            customerServiceDTO.setCustomerServiceNumber(location.getCustomerServiceNumber());
            customerServiceDTO.setCustomerServiceQrCode(location.getCustomerServiceQrCode());
        }
        customerServiceDTO.setPhone(getGroupCustomerPhoneServiceInfo(code).getGroupCustomerServicePhone());
        //数据库查询完设置缓存值
        redisHashMapAdapter.put(RedisConfig.CUSTOMER_SERVICE_INFO, code, customerServiceDTO);
        return customerServiceDTO;
    }

    private CustomerServiceDTO getOfficialAccountInfo(String code, Byte type) {
        CustomerServiceDTO customerInfoCache = new CustomerServiceDTO();
        LocationOfficialAccountVO officialAccountVO = redisHashMapAdapter.get(RedisConfig.QRCODE_LIST_INFO, code, LocationOfficialAccountVO.class);
        if (Objects.isNull(officialAccountVO)) {
            Location location = locationMapper.selectByCode(code, false);
            //缓存和数据库都没数据，默认返回客服信息
            if (Objects.isNull(location)) {
                return getCustomerInfo(code);
            }
            String officialAccountQrcode = location.getOfficialAccountQrcode();
            //不属于新的二维码列表信息，默认展示客服信息
            if (!JSONUtil.isJson(officialAccountQrcode)) {
                return getCustomerInfo(code);
            } else {
                LocationOfficialAccountVO officialAccount = new LocationOfficialAccountVO();
                List<QrCodeInfoVO> qrCodeInfoVOS = JSONUtil.toList(location.getOfficialAccountQrcode(), QrCodeInfoVO.class);
                for (QrCodeInfoVO qrCodeInfoVO : qrCodeInfoVOS) {
                    if (Objects.equals(qrCodeInfoVO.getType(), type)) {
                        customerInfoCache.setCustomerServiceQrCode(qrCodeInfoVO.getQrCodeUrl());
                        customerInfoCache.setPhone(qrCodeInfoVO.getPhone());
                        customerInfoCache.setCustomerServiceNumber(qrCodeInfoVO.getWxAccount());
                        //手机号和微信号不存在，走客服的信息
                        if (qrCodeInfoVO.getPhone() == null || qrCodeInfoVO.getWxAccount() == null) {
                            CustomerServiceDTO customerInfo = getCustomerInfo(code);
                            customerInfoCache.setPhone(qrCodeInfoVO.getPhone() == null ? customerInfo.getPhone() : qrCodeInfoVO.getPhone());
                            customerInfoCache.setCustomerServiceNumber(qrCodeInfoVO.getWxAccount() == null ? customerInfo.getCustomerServiceNumber() : qrCodeInfoVO.getWxAccount());
                        }
                    }
                }
                officialAccount.setCode(code);
                officialAccount.setName(location.getName());
                officialAccount.setQrCodeInfoList(qrCodeInfoVOS);
                redisHashMapAdapter.put(RedisConfig.QRCODE_LIST_INFO, code, officialAccount);
                return customerInfoCache;
            }
        } else {
            customerInfoCache.setCode(officialAccountVO.getCode());
            List<QrCodeInfoVO> qrCodeInfoList = officialAccountVO.getQrCodeInfoList();
            boolean qrCodeResult = false;
            for (QrCodeInfoVO qrCodeInfoVO : qrCodeInfoList) {
                if (Objects.equals(qrCodeInfoVO.getType(), type)) {
                    customerInfoCache.setCustomerServiceQrCode(qrCodeInfoVO.getQrCodeUrl());
                    customerInfoCache.setPhone(qrCodeInfoVO.getPhone());
                    customerInfoCache.setCustomerServiceNumber(qrCodeInfoVO.getWxAccount());
                    //手机号和微信号不存在，走客服的信息
                    if (qrCodeInfoVO.getPhone() == null || qrCodeInfoVO.getWxAccount() == null) {
                        CustomerServiceDTO customerInfo = getCustomerInfo(code);
                        customerInfoCache.setPhone(qrCodeInfoVO.getPhone() == null ? customerInfo.getPhone() : qrCodeInfoVO.getPhone());
                        customerInfoCache.setCustomerServiceNumber(qrCodeInfoVO.getWxAccount() == null ? customerInfo.getCustomerServiceNumber() : qrCodeInfoVO.getWxAccount());
                    }
                    qrCodeResult = true;
                }
            }
            //类型不匹配，默认走客服信息
            if (!qrCodeResult) {
                return getCustomerInfo(code);
            }
            return customerInfoCache;
        }
    }

    @Override
    public List<String> listAllLocationWithCommunity() {
        return locationMapper.selectLocationWithCommunity();
    }

    @Override
    public GroupCustomerPhoneDTO getGroupCustomerPhoneServiceInfo(String code) {
        GroupCustomerPhoneDTO groupCustomerPhoneCache = redisHashMapAdapter.get(RedisConfig.GROUP_CUSTOMER_PHONE, code, GroupCustomerPhoneDTO.class);
        // 如果缓存为空则
        if (groupCustomerPhoneCache != null) {
            logger.info("地区：{}，从缓存中取出团购客服电话信息：{}", code, JSON.toJSONString(groupCustomerPhoneCache));
            return groupCustomerPhoneCache;
        }
        GroupCustomerPhoneDTO groupCustomerPhoneDTO = new GroupCustomerPhoneDTO();
        groupCustomerPhoneDTO.setGroupCustomerName(bizConfigProperties.getGroupCustomerName());
        groupCustomerPhoneDTO.setCode(code);
        // 从数据库中取数据
        Location location = locationMapper.selectGroupCustomPhoneByCode(code);
        if (location == null || StringUtils.isBlank(location.getGroupCustomerServicePhone())) {
            //如果数据库中结果为空，则使用全国统一的团购客服电话
            groupCustomerPhoneDTO.setGroupCustomerServicePhone(bizConfigProperties.getGroupCustomerPhone());
            return groupCustomerPhoneDTO;
        }
        groupCustomerPhoneDTO.setGroupCustomerServicePhone(location.getGroupCustomerServicePhone());

        // 缓存数据查询结果
        redisHashMapAdapter.put(RedisConfig.GROUP_CUSTOMER_PHONE, code, groupCustomerPhoneDTO);

        return groupCustomerPhoneDTO;
    }

    @Override
    public OfficialAccountDTO getOfficialAccountInfo(String code) {
        //返回默认值
        if (StringUtils.isBlank(code)) {
            logger.warn("地区编码为空");
            OfficialAccountDTO resultDTO = new OfficialAccountDTO();
            resultDTO.setCode(code);
            resultDTO.setQrCodeUrl(bizConfigProperties.getOfficialAccountQrcode());
            return resultDTO;
        }

        OfficialAccountDTO officialAccountCache = redisHashMapAdapter.get(RedisConfig.OFFICIAL_ACCOUNT_QRCODE, code, OfficialAccountDTO.class);
        //如果缓存不为空为空
        if (officialAccountCache != null) {
            return officialAccountCache;
        }
        OfficialAccountDTO resultDTO = new OfficialAccountDTO();
        resultDTO.setCode(code);

        Location location = locationMapper.selectByCode(code, false);
        if (location == null || StringUtils.isBlank(location.getOfficialAccountQrcode())) {
            resultDTO.setQrCodeUrl(bizConfigProperties.getOfficialAccountQrcode());
            return resultDTO;
        }
        //兼容老接口
        if (JSONUtil.isJson(location.getOfficialAccountQrcode())) {
            List<QrCodeInfoVO> qrCodeInfoVOS = JSONUtil.toList(location.getOfficialAccountQrcode(), QrCodeInfoVO.class);
            if (!qrCodeInfoVOS.isEmpty()) {
                resultDTO.setQrCodeUrl(qrCodeInfoVOS.get(0).getQrCodeUrl());
            }
        } else {
            resultDTO.setQrCodeUrl(location.getOfficialAccountQrcode());
        }
        // 缓存数据查询结果
        redisHashMapAdapter.put(RedisConfig.OFFICIAL_ACCOUNT_QRCODE, code, resultDTO);
        return resultDTO;
    }

    @Override
    public LocationProvinceAndCityDTO getProvinceAndCityInfo(String areaCode) {
        //取省市
        if (areaCode.length() >= 4) {
            AreaDivision provinceInfo = areaDivisionMapper.selectByCode(areaCode.substring(0, 2));
            AreaDivision cityInfo = areaDivisionMapper.selectByCode(areaCode.substring(0, 4) + "00000000");
            return LocationProvinceAndCityDTO.builder()
                    .province(provinceInfo.getName())
                    .city(cityInfo.getName())
                    .build();
        }

        //如果小于4且大于等于2则取省
        if (areaCode.length() >= 2) {
            AreaDivision provinceInfo = areaDivisionMapper.selectByCode(areaCode.substring(0, 2));
            return LocationProvinceAndCityDTO.builder().province(provinceInfo.getName()).build();
        }

        return LocationProvinceAndCityDTO.builder().build();
    }

    @Override
    public String completeAreaCode(String areaCode) {
        String fullCode = areaCode;
        if (AREA_CODE_LENGTH != fullCode.length()) {
            //当定位时取得的是6位城市编码
            fullCode = areaCode + "000000";
            LocationDetailDTO locationDTO = getLocationDetailByCode(fullCode);
            //如果是区是查不到的，查询上一级
            if (null == locationDTO) {
                fullCode = areaCode.substring(0, 4) + "00000000";
            }
        }
        return fullCode;
    }

}
