package com.bxm.localnews.base.service.impl;

import com.bxm.localnews.base.config.SpecialUrlConfig;
import com.bxm.localnews.base.service.BaseUrlFacadeService;
import com.bxm.localnews.base.service.ShortLinkService;
import com.bxm.localnews.common.constant.DomainScene;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.common.constant.ShortLinkTypeEnum;
import com.bxm.localnews.common.dto.PrivilegeShareShortLinkDTO;
import com.bxm.localnews.common.dto.ShortBaseUrl;
import com.bxm.localnews.common.param.GetAvailableDomainInfoParam;
import com.bxm.localnews.common.vo.BaseUrlInfo;
import com.bxm.localnews.common.vo.ShortLinkReq;
import com.bxm.localnews.thirdparty.param.RedirectToWechatMpAuthParam;
import com.bxm.localnews.thirdparty.service.WechatMpFacadeService;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.ShortUrlGenerator;
import com.bxm.newidea.component.tools.SpringContextHolder;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.util.Objects;

import static com.bxm.localnews.common.constant.ShortLinkTypeEnum.T;
import static com.bxm.localnews.common.rediskey.ShortUrlRedisKey.SHORT_URL;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.join;

/**
 * Created by Administrator on 2018/4/3.
 */
@Slf4j
@AllArgsConstructor
@Service("shortLinkService")
public class ShortLinkServiceImpl extends BaseService implements ShortLinkService {

    /**
     * 短链接host
     */
    private final static String SHORT = "t";

    private RedisStringAdapter redisStringAdapter;

    // private BizConfigProperties bizConfigProperties;

    private BaseUrlFacadeService baseUrlFacadeService;

    private SpecialUrlConfig specialUrlConfig;

    private WechatMpFacadeService getWechatMpFacadeService() {
        return SpringContextHolder.getBean(WechatMpFacadeService.class);
    }

    @Override
    public String getRegisterShortUrl(ShortLinkReq req) {
        StringBuilder url = new StringBuilder();
        //渠道地址 h5根路径/register.html?inviteUserId=
        String channelUrl = join(baseUrlFacadeService.getInnerH5BaseUrl(), specialUrlConfig.getChannel());
        //邀请注册地址
        url.append(channelUrl).append(req.getUserId());
        return this.getShortUrl(url.toString(), true);
    }

    @Override
    public String getDownloadShortUrl() {
        return getShortUrl(join(baseUrlFacadeService.getInnerH5BaseUrl(), specialUrlConfig.getDownload()), true);
    }

    @Override
    public String getFriendSweepShortUrl(ShortLinkReq req) {
        StringBuilder url = new StringBuilder();
        //渠道地址
        url.append(baseUrlFacadeService.getInnerH5BaseUrl())
                .append(specialUrlConfig.getFriendSweepUrl())
                .append(req.getUserId());
        return this.getShortUrl(url.toString(), true);
    }

    @Override
    public String getMachineShortLink(String machineId) {
        String shortCode = ShortUrlGenerator.getShortUrl(machineId)[0];
        KeyGenerator shortKey = RedisConfig.MACHINE_SHORT_URL_KEY.copy().setKey(shortCode);
        redisStringAdapter.set(shortKey, machineId);

        StringBuilder sortUrl = new StringBuilder();
        sortUrl.append(baseUrlFacadeService.getShortLinkBaseUrl()).append("m").append("/");
        sortUrl.append(shortCode);

        return sortUrl.toString();
    }

    @Override
    public String getPrivilegeShareShortLink(PrivilegeShareShortLinkDTO privilegeShareShortLinkDTO) {
        // 先拼接落地页的地址
        String redirectUrl = join(baseUrlFacadeService.getContentViewSceneBaseUrl(), specialUrlConfig.getShareDrawActivity());

        // 获取重定向地址
        RedirectToWechatMpAuthParam param = new RedirectToWechatMpAuthParam();
        param.setScene(Objects.toString(DomainScene.WECHAT_AUTH));
        param.setRedirectUrl(redirectUrl);
        param.setScope("snsapi_userinfo");
        param.setState("STATE");
        return getShortUrl(getWechatMpFacadeService().wechatMpAuthUrlGen(param), true);
    }

    @Override
    public String generateShortUrl(String url) {
        return getShortUrl(url, true);
    }

    @Override
    public String generateShortUrl(String url, ShortLinkTypeEnum shortLinkType) {
        return getShortUrl(url, shortLinkType, true);
    }

    /**
     * 生成默认的t短链
     *
     * @param url     url
     * @param expired 是否失效
     * @return 短链
     */
    private String getShortUrl(String url, boolean expired) {
        return getShortUrl(url, T, expired);
    }

    /**
     * 根据短链类型生成短链
     *
     * @param url           url
     * @param shortLinkType 短链类型
     * @param expired       是否失效
     * @return 短链
     */
    private String getShortUrl(String url, ShortLinkTypeEnum shortLinkType, boolean expired) {
        if (log.isDebugEnabled()) {
            logger.debug("生成短链: url: [{}] shortLinkType: {} expired: {}", url, shortLinkType, expired);
        }

        // 获取短连的base url
        ShortBaseUrl shortBaseUrl = getShortBaseUrl(shortLinkType);

        // 拼接
        StringBuilder sortUrl = new StringBuilder(shortBaseUrl.getBaseUrl());
        sortUrl.append(shortBaseUrl.getShortLinkType().getType())
                .append("/");

        // 得到短链code
        String[] keys = ShortUrlGenerator.getShortUrl(url);

        //防止短链冲突问题
        String key = keys[0];
        if (Objects.equals(this.redisStringAdapter.hasKey(SHORT_URL.copy().appendKey(key)), Boolean.TRUE)) {
            key = keys[1];
        }

        sortUrl.append(key);

        //缓存180天，超时，则跳转到注册界面
        if (expired) {
            this.redisStringAdapter.set(SHORT_URL.copy().appendKey(key), url, 3600 * 24 * 90);
        } else {
            this.redisStringAdapter.set(SHORT_URL.copy().appendKey(key), url);
        }

        return sortUrl.toString();
    }

    /**
     * 根据短链类型 获取base url
     *
     * @param shortLinkType 短链类型
     * @return base url
     */
    private ShortBaseUrl getShortBaseUrl(ShortLinkTypeEnum shortLinkType) {
        // 获取短链base url
        String baseUrl;
        ShortLinkTypeEnum finalType;
        if (Objects.equals(shortLinkType, T)) {
            // t/{key} 类型的短链
            baseUrl = baseUrlFacadeService.getShortLinkBaseUrl();
            finalType = T;
        } else {
            // 其他类型的
            GetAvailableDomainInfoParam param = new GetAvailableDomainInfoParam();
            param.setScene(Objects.toString(getSceneByShortLinkType(shortLinkType)));

            BaseUrlInfo baseUrlInfo = baseUrlFacadeService.getBaseUrlInfo(param);
            if (Objects.isNull(baseUrlInfo) || isBlank(baseUrlInfo.getBaseUrl())) {
                // 为空 则默认使用短链的
                baseUrl = baseUrlFacadeService.getShortLinkBaseUrl();
                finalType = T;
            } else {
                baseUrl = baseUrlInfo.getBaseUrl();
                finalType = shortLinkType;
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("源类型: {} 获取到的短链base url: {} finalType: {}", shortLinkType, baseUrl, finalType);
        }

        return ShortBaseUrl.builder().baseUrl(baseUrl).shortLinkType(finalType).build();
    }

    /**
     * 根据短链类型，获取对应的域名场景
     *
     * @param shortLinkType 短链类型
     * @return 域名场景
     */
    private DomainScene getSceneByShortLinkType(ShortLinkTypeEnum shortLinkType) {
        switch (shortLinkType) {
            case WX:
                // 微信短链
                return DomainScene.WECHAT_SHORT;
            case T:
            default:
                // 默认是短链类型
                return DomainScene.SHORT;
        }
    }

}
