package com.bxm.localnews.base.service.manage;

import com.bxm.localnews.base.domain.BaseDomainExtendMapper;
import com.bxm.localnews.base.domain.BaseDomainPublishExtendMapper;
import com.bxm.localnews.base.domain.WechatMpInfoExtendMapper;
import com.bxm.localnews.base.service.domain.mointer.MoniterService;
import com.bxm.localnews.common.constant.DomainScene;
import com.bxm.localnews.common.constant.DomainScene.DomainViewScene;
import com.bxm.localnews.common.entity.BaseDomainPublish;
import com.bxm.localnews.common.entity.DomainPublishInfo;
import com.bxm.localnews.common.entity.TBaseDomain;
import com.bxm.localnews.common.entity.WechatMpInfo;
import com.bxm.localnews.common.param.*;
import com.bxm.localnews.common.vo.DomainInfoVO;
import com.bxm.localnews.common.vo.DomainVO;
import com.bxm.localnews.mq.common.param.DingtalkMessage;
import com.bxm.localnews.msg.sender.MessageSender;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import com.bxm.newidea.component.vo.PageWarper;
import com.github.pagehelper.Page;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

import static com.alibaba.fastjson.JSON.toJSONString;
import static com.github.pagehelper.page.PageMethod.startPage;
import static java.util.Objects.isNull;
import static java.util.Objects.nonNull;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

/**
 * @author gonzo
 * @date 2020-09-02 19:37
 **/
@Slf4j
@Service
@AllArgsConstructor
public class DomainManageServiceImpl implements DomainManageService {

    private final BaseDomainPublishExtendMapper baseDomainPublishExtendMapper;

    private final BaseDomainExtendMapper baseDomainExtendMapper;

    private final SequenceCreater sequenceCreater;

    private final MoniterService moniterService;

    private final MessageSender messageSender;

    private final WechatMpInfoExtendMapper wechatMpInfoExtendMapper;

    @Override
    public List<DomainVO> listAll(DomainListParam param) {
        param.setStatus((byte) 1);
        return baseDomainExtendMapper.selectAllAvailable()
                .stream()
                .map(this::convert)
                .collect(Collectors.toList());
    }

    @Override
    public PageWarper<DomainVO> list(DomainListParam param) {

        Page<TBaseDomain> domains = startPage(param)
                .doSelectPage(() -> baseDomainExtendMapper.selectByParam(param));

        PageWarper pageWarper = new PageWarper(domains);
        pageWarper.setList(domains.stream()
                .map(this::convert)
                .collect(Collectors.toList()));
        return pageWarper;
    }

    private DomainVO convert(TBaseDomain domain) {
        DomainVO vo = new DomainVO();
        BeanUtils.copyProperties(domain, vo);
        vo.setStatusStr(statusStr(vo.getStatus()));
        return vo;
    }

    private String statusStr(byte status) {
        switch (status) {
            case 1:
                return "上架";
            case 2:
                return "微信被封";
            case 3:
                return "域名备份丢失";
            default:
                return "下架";
        }
    }

    @Override
    public Message add(AddBaseDomainParam param) {
        Date now = new Date();

        // 批量添加
        Arrays.stream(param.getDomain().split("\n"))
                .forEach(p -> {
                    // 避免重复添加
                    List<TBaseDomain> tBaseDomains = baseDomainExtendMapper.selectByDomain(p);
                    if (!CollectionUtils.isEmpty(tBaseDomains)) {
                        return;
                    }

                    TBaseDomain domain = new TBaseDomain();

                    domain.setId(sequenceCreater.nextLongId());
                    domain.setDomain(p);
                    domain.setStatus(param.getStatus());
                    domain.setCreateTime(now);
                    domain.setModifyTime(now);

                    baseDomainExtendMapper.insertSelective(domain);
                });

        return Message.build();
    }

    @Override
    public Message edit(EditBaseDomainParam param) {

        TBaseDomain update = new TBaseDomain();
        BeanUtils.copyProperties(param, update);
        update.setModifyTime(new Date());

        baseDomainExtendMapper.updateByPrimaryKeySelective(update);
        return Message.build();
    }

    @Override
    public Message deleteOneDomain(Long id) {
        baseDomainExtendMapper.deleteOne(id);
        return Message.build();
    }

    @Override
    public PageWarper<DomainInfoVO> publishList(DomainPublishListParam param) {
        Page<DomainPublishInfo> page = startPage(param)
                .doSelectPage(() -> baseDomainPublishExtendMapper.selectByParam(param));

        PageWarper pageWarper = new PageWarper(page);
        pageWarper.setList(page.stream()
                .map(this::build)
                .collect(Collectors.toList()));
        return pageWarper;
    }

    private DomainInfoVO build(DomainPublishInfo info) {
        DomainInfoVO vo = new DomainInfoVO();
        BeanUtils.copyProperties(info, vo);
        vo.setBaseUrl(StringUtils.join(info.getProtocol(), info.getDomain()));
        if (nonNull(info.getScene())) {
            Optional<DomainScene> domainScene = DomainScene.forScene(info.getScene());
            vo.setSceneStr(domainScene.isPresent() ? domainScene.get().getDesc() : "");
        }

        if (nonNull(info.getViewScene())) {
            Optional<DomainViewScene> domainViewScene = DomainViewScene.forViewScene(info.getViewScene());
            vo.setViewSceneStr(domainViewScene.isPresent() ? domainViewScene.get().getDesc() : "");
        }
        vo.setStatusStr(Objects.equals(vo.getStatus(), (byte) 1) ? "可用" : "不可用");

        vo.setWechatMpName("暂无");
        if (isNotBlank(info.getAppId())) {
            WechatMpInfo wechatMpInfo = wechatMpInfoExtendMapper.loadByAppId(info.getAppId());
            if (nonNull(wechatMpInfo)) {
                vo.setWechatMpName(wechatMpInfo.getName());
            }
        }

        return vo;
    }


    @Override
    public Message publishAdd(AddDomainPublishParam param) {

        if (Objects.equals(param.getScene(), DomainScene.OUTSIDE_SHARE_VIEW.getScene())
                && isNull(param.getViewScene())) {
            return Message.build(false, "如果投放为落地页，请选择具体的落地页场景");
        }

        // 域名
        TBaseDomain domain = baseDomainExtendMapper.selectByPrimaryKey(param.getDomainId());
        if (isNull(domain)) {
            return Message.build(false, "要绑定的域名不存在或不可用");
        }

        // 场景
        Optional<DomainScene> domainScene = DomainScene.forScene(param.getScene());
        if (!domainScene.isPresent()) {
            return Message.build(false, StringUtils.join("场景", param.getScene(), "不存在"));
        }

        // 校验域名 协议、域名和场景校验
        /*if (!moniterService.checkBaseUrlAvailable(StringUtils.join(param.getProtocol(), domain.getDomain()), domainScene.get())) {
            return Message.build(false, "域名校验不可用，请确认是否解析成功");
        }*/

        Date now = new Date();
        BaseDomainPublish publish = new BaseDomainPublish();
        publish.setId(sequenceCreater.nextLongId());
        publish.setDomainId(param.getDomainId());
        publish.setProtocol(param.getProtocol());
        publish.setScene(param.getScene());
        publish.setViewScene(param.getViewScene());
        publish.setAppId(param.getAppId());
        publish.setRemark(param.getRemark());
        publish.setStatus(param.getStatus());
        publish.setCreateTime(now);
        publish.setModifyTime(now);

        baseDomainPublishExtendMapper.insertSelective(publish);

        return Message.build();
    }

    @Override
    public Message publishEdit(EditDomainPublishParam param) {

        BaseDomainPublish publish = baseDomainPublishExtendMapper.selectByPrimaryKey(param.getId());
        if (isNull(publish)) {
            return Message.build(false, "投放记录不存在");
        }

        if (Objects.equals(param.getScene(), DomainScene.OUTSIDE_SHARE_VIEW.getScene())
                && isNull(param.getViewScene())) {
            return Message.build(false, "如果投放为落地页，请选择具体的落地页场景");
        }

        // 是否编辑了协议
        String protocol = isNotBlank(param.getProtocol()) ? param.getProtocol() : publish.getProtocol();

        if (isBlank(protocol)) {
            return Message.build(false, "协议不能为空");
        }

        // 编辑了域名
        TBaseDomain domain = baseDomainExtendMapper.selectByPrimaryKey(nonNull(param.getDomainId())
                // 如果编辑了域名 则使用编辑的域名查找
                ? param.getDomainId() : publish.getDomainId());

        if (isNull(domain)) {
            return Message.build(false, "要绑定的域名不存在或不可用");
        }

        // 是否编辑了场景
        byte scene = isNull(param.getScene()) ? publish.getScene() : param.getScene();

        // 最终的场景
        Optional<DomainScene> domainScene = DomainScene.forScene(scene);
        if (!domainScene.isPresent()) {
            return Message.build(false, StringUtils.join("场景", scene, "不存在"));
        }

        // 校验域名 使用原有的 or 编辑的协议、域名和场景校验
        /*if (!moniterService.checkBaseUrlAvailable(StringUtils.join(protocol, domain.getDomain()), domainScene.get())) {
            return Message.build(false, "域名校验不可用，请确认是否解析成功");
        }*/

        BaseDomainPublish update = new BaseDomainPublish();
        BeanUtils.copyProperties(param, update);
        update.setModifyTime(new Date());

        baseDomainPublishExtendMapper.updateByPrimaryKeySelective(update);

        return Message.build();
    }

    @Override
    public Message deleteOneDomainPublish(Long id) {
        baseDomainPublishExtendMapper.deleteOne(id);
        return Message.build();
    }

    /**
     * 定时处理已失效的域名
     */
    @Override
    public void scheduleCheckDomain() {
        Date now = new Date();
        List<DomainPublishInfo> infos = baseDomainPublishExtendMapper.selectAllAvailableDomain();
        infos.forEach(p -> {
            // 场景
            Optional<DomainScene> domainScene = DomainScene.forScene(p.getScene());
            if (!domainScene.isPresent()) {
                log.warn( "场景: {} 不存在", p.getScene());
                return;
            }

            log.info("校验域名: info {}", toJSONString(p));

            if (!moniterService.checkBaseUrlAvailable(StringUtils.join(p.getProtocol(), p.getDomain()), domainScene.get())) {
                log.info("域名: {} 类型: {} 校验不成功，下架域名", p.getDomain(), domainScene.get());

                // 如果校验失败 下架域名
                /*TBaseDomain domainInfo = new TBaseDomain();
                domainInfo.setId(p.getDomainId());
                // 已掉备案
                domainInfo.setStatus((byte) 3);
                domainInfo.setBlockedTime(now);
                domainInfo.setModifyTime(now);

                baseDomainExtendMapper.updateByPrimaryKeySelective(domainInfo);*/

                // 投放
                /*BaseDomainPublish publish = new BaseDomainPublish();
                publish.setId(p.getId());
                // 不可用
                publish.setStatus((byte) 0);
                publish.setModifyTime(now);
                baseDomainPublishExtendMapper.updateByPrimaryKeySelective(publish);*/

                // 发送叮叮消息
                messageSender.sendDingtalk(DingtalkMessage.builder()
                        .content(StringUtils.join("域名", p.getDomain(), "类型", domainScene.get(),
                                "检测不成功，将进行下架操作，请及时补充对应类型的域名！！！！"))
                        .build());
            }
        });
    }
}
