package com.bxm.localnews.base.service.webtoken.impl;

import com.bxm.localnews.base.service.webtoken.WebTokenService;
import com.bxm.localnews.common.vo.WebTokenVerifyVO;
import com.bxm.newidea.component.jwt.tools.JwtTokenUtil;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.vo.Message;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import me.chanjar.weixin.mp.api.WxMpService;
import me.chanjar.weixin.mp.bean.result.WxMpOAuth2AccessToken;
import org.springframework.stereotype.Service;

import java.util.Objects;

import static com.bxm.localnews.common.constant.RedisConfig.WEB_TOKEN_OPENID_CACHE;
import static org.apache.commons.lang3.StringUtils.isBlank;

/**
 * 获取站外WEB TOKEN服务
 *
 * @author wzy
 * @version 1.0
 * @date 2021/1/14 11:39 上午
 */
@Slf4j
@Service
@AllArgsConstructor
public class WebTokenServiceImpl implements WebTokenService {

    private final WxMpService wxMpService;

    private final RedisSetAdapter redisSetAdapter;

    /**
     * web token的过期时间
     */
    public static final Long WEB_TOKEN_EXPIRED_TIME = 180 * 24 * 60 * 60 * 1000L;

    @Override
    public Message getWebToken(String code) {
        //根据code获取openId
        WxMpOAuth2AccessToken wxMpOAuth2AccessToken = null;
        try {
            wxMpOAuth2AccessToken = wxMpService.oauth2getAccessToken(code);
        } catch (Exception e) {
            log.error("获取微信openid失败, code:{}", code);
            return Message.build(false, "获取微信认证信息失败");
        }

        //如果为空则返回获取失败
        if (Objects.isNull(wxMpOAuth2AccessToken)) {
            return Message.build(false, "获取微信认证信息失败");
        }

        //获取微信用户openid
        String openId = wxMpOAuth2AccessToken.getOpenId();

        //如果openid为空则返回获取失败
        if (isBlank(openId)) {
            return Message.build(false, "获取微信认证信息失败");
        }
        log.info("颁发webToken时，openId:{}", openId);

        //jwt生成token令牌
        String webToken = JwtTokenUtil.generateToken(openId, WEB_TOKEN_EXPIRED_TIME);
        log.info("颁发webToken时，webToken:{}", webToken);

        //记录颁发jwt Token所用到的openid
        redisSetAdapter.add(buildKey(), openId);

        return Message.build(true).addParam("webToken", webToken);
    }

    /**
     * 校验webToken是否合法
     *
     * @return 是否合法
     */
    @Override
    public WebTokenVerifyVO verifyWebToken(String token) {
        String openid = JwtTokenUtil.getWxAuthOpenidFromToken(token);

        log.info("验证webToken是否正确：{},{}", token, openid);

        if (isBlank(openid)) {
            return WebTokenVerifyVO.builder().expired(true).build();
        }
        //如果过期则返回token不合法
        if (JwtTokenUtil.isTokenExpired(token) || Boolean.FALSE.equals(redisSetAdapter.exists(buildKey(), openid))) {
            return WebTokenVerifyVO.builder().expired(true).build();
        }

        return WebTokenVerifyVO.builder().expired(false).openid(openid).build();
    }

    /**
     * 构建WEB TOKEN 缓存openid缓存key
     *
     * @return 缓存key
     */
    private KeyGenerator buildKey() {
        return WEB_TOKEN_OPENID_CACHE.copy();
    }
}