package com.bxm.localnews.base.service.impl;

import com.bxm.localnews.base.config.LocationProperties;
import com.bxm.localnews.base.domain.AreaInfoMapper;
import com.bxm.localnews.base.service.AreaInfoService;
import com.bxm.localnews.base.service.LocationService;
import com.bxm.localnews.common.config.BizConfigProperties;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.common.constant.SwitchType;
import com.bxm.localnews.common.dto.LocationDetailDTO;
import com.bxm.localnews.common.dto.LocationSwitchDTO;
import com.bxm.localnews.common.dto.UserForceChange;
import com.bxm.localnews.common.param.GetAreaInfoParam;
import com.bxm.localnews.common.param.LocationSwitchParam;
import com.bxm.localnews.common.vo.AreaInfo;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.dto.UserLocationInfoDTO;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.fasterxml.jackson.core.type.TypeReference;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Objects;
import java.util.Optional;

import static com.alibaba.fastjson.JSON.toJSON;
import static com.bxm.localnews.common.constant.RedisConfig.FORCE_CHANGE_USER_AREA_INFO;
import static com.gexin.fastjson.JSON.toJSONString;
import static java.util.Objects.isNull;
import static java.util.Objects.nonNull;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

@Slf4j
@Service
@AllArgsConstructor
public class AreaInfoServiceImpl extends BaseService implements AreaInfoService {

    private final AreaInfoMapper areaInfoMapper;

    private final RedisStringAdapter redisStringAdapter;

    private final BizConfigProperties bizConfigProperties;

    private final LocationService locationService;

    private final LocationProperties locationProperties;

    private final RedisHashMapAdapter redisHashMapAdapter;

    private final UserIntegrationService userIntegrationService;


    @Override
    public AreaInfo getAreaInfo(String code, BasicParam basicParam) {
        GetAreaInfoParam param = new GetAreaInfoParam();
        BeanUtils.copyProperties(basicParam, param);
        param.setCode(code);

        // 老版本接口 默认情况下 code传进来的当做最新code查询 且查询不到情况下，给默认城市
        return getAreaInfo(param, true, true);
    }

    @Override
    public AreaInfo getAreaInfo(GetAreaInfoParam param, boolean defaultInfo, boolean lastCodeQuery) {
        if (StringUtils.isBlank(param.getCode())) {
            logger.error("用户获取定位传参错误,code:[{}],参数:[{}]", param.getCode(), toJSON(param));
            return new AreaInfo();
        }

        // 临时的特殊处理
        String processedCode = temporaryProcess(param.getCode(), param.getUserId(), param.getDevcId());

        LocationDetailDTO locationDTO;
        if (defaultInfo)  {
            // 如果设置了查询不到需要返回默认的信息，则调用原有的方法
            locationDTO = locationService.getInternalLocationByGeocode(processedCode, param);
        } else {
            // 如果设置了查询不到返回空信息 则调用新增的方法
            locationDTO = locationService.getInternalLocationByGeocode(processedCode, param, false,
                    false, lastCodeQuery);
        }

        // 查询不到直接返回
        if (isNull(locationDTO)) {
            return new AreaInfo();
        }

        // 由于code可能是最新的编码，也可能是系统定义的编码，但是接下来使用的肯定都是系统的编码，所以这里将code进行重新赋值 使用系统编码进行处理
        processedCode = locationDTO.getCode();
        KeyGenerator areaInfoKey = RedisConfig.AREA_INFO;
        TypeReference<List<AreaInfo>> areaInfoTypeReference = new TypeReference<List<AreaInfo>>() {
        };
        AreaInfo areaInfo = null;
        List<AreaInfo> areaInfoList = redisStringAdapter.get(areaInfoKey, areaInfoTypeReference);
        if (areaInfoList != null) {
            String finalCode = processedCode;
            Optional<AreaInfo> areaInfoOptional = areaInfoList.stream().filter(x -> x.getCode().equals(finalCode)).findFirst();
            if (areaInfoOptional.isPresent()) {
                areaInfo = areaInfoOptional.get();
            }
        }

        if (null == areaInfo) {
            areaInfoList = areaInfoMapper.selectByModel(null);
            redisStringAdapter.set(areaInfoKey, areaInfoList);
            areaInfo = areaInfoMapper.selectByCode(processedCode);
        }

        if (null == areaInfo) {
            areaInfo = new AreaInfo();
        }

        areaInfo.setCode(processedCode);
        areaInfo.setName(locationDTO.getName());
        areaInfo.setEnablePaidPromote(locationDTO.getEnablePaidPromote());

        logger.info("根据code: {} 获取用户定位信息, 最终获取到的定位信息: {}, 请求参数信息: {} ", param.getCode(), toJSON(areaInfo),
                toJSON(param));
        return areaInfo;
    }

    @Override
    public LocationSwitchDTO locationSwitch(LocationSwitchParam param) {
        // TODO 可以抽取一下 重复逻辑过多
        if (log.isDebugEnabled()) {
            log.debug("用户: {} 获取定位信息的请求参数: {}", param.getUserId(), toJSONString(param));
        }

        LocationSwitchDTO switchDTO = new LocationSwitchDTO();
        AreaInfo areaInfo = new AreaInfo();

        // 优先查询用户的上一次定位
        UserLocationInfoDTO userLocationCodeById = userIntegrationService.getUserLocationCodeById(param.getUserId());
        if (log.isDebugEnabled()) {
            log.debug("用户: {} 的最后一次定位信息: {}", param.getUserId(), toJSONString(userLocationCodeById));
        }

        // 上一次定位为空
        if (isNull(userLocationCodeById) || isBlank(userLocationCodeById.getLocationCode())) {
            // 根据code获取要切换的城市信息 or弹出城市选择框
            return switchFromCode(param);
        }

        // 上次定位不为空 根据最后一次定位code查询定位信息
        LocationDetailDTO locationDTO = locationService.getInternalLocationByGeocode(userLocationCodeById.getLocationCode(), param);

        // 查看当前用户是否存在于强制切换的列表中 存在的话 则返回强制的列表信息
        locationDTO = processSpecialUser(param.getUserId(), locationDTO, param);
        // 特殊城市的强制切换
        locationDTO = forceSwitchIfNecessary(locationDTO, param);

        // 查询不到城市信息 则根据当前定位code查询
        if (isNull(locationDTO) || isBlank(locationDTO.getCode())) {
            // 避免客户端传一个空
            if (isBlank(param.getThirdpartyCode())) {
                logger.warn("用户获取定位传参错误, param: {}", toJSON(param));
                // 提示弹出城市选择
               return openSwitchReturn(param);
            }
            // code需要特殊处理一下
            locationDTO = locationService.getInternalLocationByGeocode(temporaryProcess(param.getThirdpartyCode(),
                    param.getUserId(), param.getDevcId()), param);

            // 查询不到城市信息 则控制客户端弹出选择城市弹窗
            if (isNull(locationDTO) || isBlank(locationDTO.getCode())) {
                // 提示弹出城市选择
                return openSwitchReturn(param);
            }
        }

        // 查询到了城市信息返回定位
        areaInfo.setCode(locationDTO.getCode());
        areaInfo.setName(locationDTO.getName());

        // 强制切换到指定城市（最后一次定位的信息）
        switchDTO.setSwitchType(SwitchType.FORCE_SWITCH);
        switchDTO.setAreaInfo(areaInfo);
        return logFinalInfo(switchDTO, param);
    }

    /**
     * 根据code获取要切换的城市信息 or弹出城市选择框
     * @param param param
     * @return 要切换的城市信息
     */
    private LocationSwitchDTO switchFromCode(LocationSwitchParam param) {
        LocationSwitchDTO switchDTO = new LocationSwitchDTO();
        AreaInfo areaInfo = new AreaInfo();

        // 避免客户端传一个空
        if (isBlank(param.getThirdpartyCode())) {
            logger.warn("用户获取定位传参错误, param: {}", toJSON(param));
            // 提示弹出城市选择
            return openSwitchReturn(param);
        }

        // 根据code查询城市信息 code需要先特殊处理一下
        LocationDetailDTO locationDTO = locationService.getInternalLocationByGeocode(temporaryProcess(param.getThirdpartyCode(),
                param.getUserId(), param.getDevcId()), param);

        // 查询不到城市信息 则控制客户端弹出选择城市弹窗
        if (isNull(locationDTO) || isBlank(locationDTO.getCode())) {
            // 提示弹出城市选择
            return openSwitchReturn(param);
        }

        // 查询到了城市信息返回定位
        areaInfo.setCode(locationDTO.getCode());
        areaInfo.setName(locationDTO.getName());

        // 强制切换到指定城市（根据code获取到的）
        switchDTO.setSwitchType(SwitchType.FORCE_SWITCH);
        switchDTO.setAreaInfo(areaInfo);
        return logFinalInfo(switchDTO, param);
    }

    /**
     * 弹出城市选择框
     * @param param param
     * @return 弹出城市选择框的选择信息
     */
    private LocationSwitchDTO openSwitchReturn(LocationSwitchParam param) {
        LocationSwitchDTO switchDTO = new LocationSwitchDTO();
        // 提示弹出城市选择
        switchDTO.setSwitchType(SwitchType.OPEN_CHOSE);
        switchDTO.setAreaInfo(null);

        return logFinalInfo(switchDTO, param);
    }

    /**
     * 查看最后的选择城市是需要强制切换的城市 比如：宣城的用户，就强制切换到广德
     *
     * @param locationDTO 要切换的城市信息
     * @param param param
     * @return 特殊城市处理后的城市信息
     */
    private LocationDetailDTO forceSwitchIfNecessary(LocationDetailDTO locationDTO, LocationSwitchParam param) {

        // 判断开关是否开启
        if (Objects.equals(locationProperties.getSpecialLocationForceSwitch(), Boolean.TRUE)
                // 获取到了城市信息
                && nonNull(locationDTO) && isNotBlank(locationDTO.getCode())
                && nonNull(locationProperties.getSpecialLocationSwitchInfo())) {
            // 判断用户是否是强制定位的区域 比如：宣城的用户，就强制切换到广德
            String specialSwitch = locationProperties.getSpecialLocationSwitchInfo().get(locationDTO.getCode());

            if (isNotBlank(specialSwitch)) {
                log.info("用户: {} 获取到的定位code: {} 强制切换到: {}", param.getUserId(), locationDTO.getCode(), specialSwitch);
                // 根据配置的Code查询城市信息
                return locationService.getInternalLocationByGeocode(specialSwitch,
                        param);
            }
        }

        return locationDTO;
    }

    /**
     * 记录一下日志
     * @param switchDTO 最终的切换城市的信息
     * @param param param
     * @return 最终的切换城市的信息
     */
    private LocationSwitchDTO logFinalInfo(LocationSwitchDTO switchDTO, LocationSwitchParam param) {
        log.info("用户: {} 请求参数: {} 最终获取到的定位信息为: {}", param.getUserId(), toJSONString(param), toJSONString(switchDTO));
        return switchDTO;
    }

    /**
     * 3.2.0.10 hotfix 的临时处理
     * 解决线上广德定位到宣城的强制处理
     */
    private String temporaryProcess(String code, Long userId, String deviceId) {

        if (Objects.equals(bizConfigProperties.getCodeProcessSwitch(), Boolean.FALSE)) {
            return code;
        }

        // 自动定位则杭州作额外处理
        if (code.startsWith("3301")) {
            logger.info("用户: {} 设备: {} code: {} 定位的特殊处理为杭州", userId, deviceId, code);
            return "330100000000";
        }

        // 2020-07-03 09:26:25 自动定位宣城(341800000000)则作为广德用户处理
        // code可能4位可能6位
        if (code.startsWith("3418") || code.startsWith("341800")) {
            logger.info("用户: {} 设备: {} code: {} 定位的特殊处理为广德", userId, deviceId, code);
            return "341822000000";
        }

        return code;
    }

    /**
     * 特殊用户的强制定位逻辑
     * 对于一些定位错乱的用户，如果确认其定位已出问题，则强制返回指定的位置
     * 客户端会手动提示其切换
     * @param userId userId
     * @param locationDTO 处理前，定位的最终信息
     * @return 如果需要切换，则是切换的code， 否则就是finalCode
     */
    private LocationDetailDTO processSpecialUser(Long userId, LocationDetailDTO locationDTO, BasicParam basicParam) {

        if (isNull(locationDTO)) {
            return null;
        }
        // 获取配置
        UserForceChange forceChange = redisHashMapAdapter.get(FORCE_CHANGE_USER_AREA_INFO, Objects.toString(userId), UserForceChange.class);
        if (log.isDebugEnabled()) {
            log.debug("用户: {} 原本的定位code: {} 获取到的强制切换信息为: {}", userId, locationDTO.getCode(), toJSONString(forceChange));
        }

        // 如果不存在对应的配置，或者配置的用户，已经不止指定要切换的地址的话，返回原本的获取到的地址
        if (isNull(forceChange) || !Objects.equals(forceChange.getIfFromCode(), locationDTO.getCode())
                // 或者已经修改过
                || Objects.equals(forceChange.getChanged(), 1)) {
            return locationDTO;
        }

        log.info("用户: {}符合强制切换的判断，将定位: {} 切换到: {}", userId, locationDTO.getCode(), forceChange.getThenChangeCode());
        // 否则 根据配置的code 查询对应的城市信息
        // 如果设置了查询不到需要返回默认的信息，则调用原有的方法
        return locationService.getInternalLocationByGeocode(forceChange.getThenChangeCode(), basicParam);
    }

}
