package com.bxm.localnews.base.service.impl;

import com.bxm.localnews.base.config.DomainConfig;
import com.bxm.localnews.base.config.SpecialUrlConfig;
import com.bxm.localnews.base.domain.BaseDomainPublishExtendMapper;
import com.bxm.localnews.base.service.BaseUrlFacadeService;
import com.bxm.localnews.base.service.domain.SceneBaseUrlJoiner;
import com.bxm.localnews.common.constant.DomainScene;
import com.bxm.localnews.common.constant.DomainScene.DomainViewScene;
import com.bxm.localnews.common.entity.DomainInfo;
import com.bxm.localnews.common.param.GetAvailableDomainInfoParam;
import com.bxm.localnews.common.param.GetViewSceneDomainInfoParam;
import com.bxm.localnews.common.param.SelectAvailableDomainBySceneParam;
import com.bxm.localnews.common.vo.BaseUrlInfo;
import com.bxm.localnews.common.vo.ViewSceneBaseUrl;
import com.bxm.localnews.common.vo.ViewSceneInfo;
import com.bxm.localnews.mq.common.param.DingtalkMessage;
import com.bxm.localnews.msg.sender.MessageSender;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.RandomUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import static com.gexin.fastjson.JSON.toJSONString;
import static org.apache.commons.lang3.StringUtils.join;

/**
 * @author Gonzo
 * @date 2020-07-30 17:21
 */
@Slf4j
@Service
public class BaseUrlServiceImpl implements BaseUrlService, BaseUrlFacadeService {

    private final BaseDomainPublishExtendMapper baseDomainPublishExtendMapper;

    private final SceneBaseUrlJoiner sceneBaseUrlJoiner;

    private final DomainConfig domainConfig;

    private final SpecialUrlConfig specialUrlConfig;

    private final MessageSender messageSender;

    private final SequenceCreater sequenceCreater;

    private LoadingCache<SelectAvailableDomainBySceneParam, List<DomainInfo>> CACHE;

    public BaseUrlServiceImpl(BaseDomainPublishExtendMapper baseDomainPublishExtendMapper, SceneBaseUrlJoiner sceneBaseUrlJoiner,
                              DomainConfig domainConfig, SpecialUrlConfig specialUrlConfig, MessageSender messageSender,
                              SequenceCreater sequenceCreater) {
        this.baseDomainPublishExtendMapper = baseDomainPublishExtendMapper;
        this.sceneBaseUrlJoiner = sceneBaseUrlJoiner;
        this.domainConfig = domainConfig;
        this.specialUrlConfig = specialUrlConfig;
        this.messageSender = messageSender;
        this.sequenceCreater = sequenceCreater;
    }

    @Override
    public BaseUrlInfo getBaseUrlInfo(GetAvailableDomainInfoParam param) {
        return getBaseUrlByScene(param)
                // 如果不存在 则使用配置数据
                .orElse(BaseUrlInfo.builder()
                        .baseUrl(domainConfig.getDefaultBaseUrl().get(param.getScene()))
                        .build());
    }

    @Override
    public ViewSceneBaseUrl getBaseUrlByViewScene(GetViewSceneDomainInfoParam param) {
        try {
            Optional<DomainViewScene> domainScene = DomainViewScene.forName(param.getViewScene());

            // 如果没有场景 则返回空
            if (!domainScene.isPresent()) {
                return new ViewSceneBaseUrl();
            }

            // 根据落地页场景 获取对应的落地页域名
            List<DomainInfo> domainInfos = selectAvailableDomainBySceneByCache(DomainScene.OUTSIDE_SHARE_VIEW.getScene(),
                    domainScene.get().getScene(), param.getAppId());

            // 剔除要过滤的落地页域名 主要是web.52huola.cn 这个域名不能当做落地页使用
            if (param.isExclusionViewScene()) {
                domainInfos.removeIf(p -> !CollectionUtils.isEmpty(domainConfig.getExclusionViewSceneDomains())
                        && domainConfig.getExclusionViewSceneDomains().contains(p.getDomain()));
            }

            // 随机获取
            Optional<DomainInfo> domainInfoOpt = getRandomInfo(domainInfos);
            if (!domainInfoOpt.isPresent()) {
                // 发送叮叮消息
                String msg = String.format("【紧急】类型: %s获取不到可用的落地页域名 事件id: %s ", param.getViewScene(), sequenceCreater.nextStringId());
                messageSender.sendDingtalk(DingtalkMessage.builder()
                        .content(msg)
                        .build());
                log.warn(msg + "param: {}", toJSONString(param));
                return new ViewSceneBaseUrl();
            }

            DomainInfo domainInfo = domainInfoOpt.get();

            ViewSceneBaseUrl domain = new ViewSceneBaseUrl();
            domain.setDomain(domainInfo.getDomain());
            domain.setBaseUrl(sceneBaseUrlJoiner.joinByScene(DomainScene.OUTSIDE_SHARE_VIEW, null,
                    StringUtils.join(domainInfo.getProtocol(),
                            domainInfo.getDomain())));
            domain.setViewScene(Objects.toString(domainScene));
            return domain;
        } catch (Exception e) {
            // 发送叮叮消息
            String msg = String.format("【紧急】类型: %s获取可用的落地页域名出错误了 事件id: %s ", param.getViewScene(), sequenceCreater.nextStringId());
            messageSender.sendDingtalk(DingtalkMessage.builder()
                    .content(msg)
                    .build());
            log.error(msg + "param: {}", toJSONString(param), e);
        }
        return new ViewSceneBaseUrl();

    }

    @Override
    public List<ViewSceneInfo> getViewSceneByDomain(String domain) {
        // 根据域名 获取对应的投放信息
        List<DomainInfo> domainInfos = baseDomainPublishExtendMapper.getViewSceneByDomain(domain);
        return domainInfos.stream().map(this::convert).collect(Collectors.toList());
    }

    @Override
    public String getShortLinkBaseUrl() {
        return getByScene(DomainScene.SHORT);
    }

    @Override
    public String getInnerH5BaseUrl() {
        return getByScene(DomainScene.INNER_H5);
    }

    /**
     * 获取分享的base url
     *
     * @param viewScene 分享落地页的类型
     * @return 分享的base url
     */
    @Override
    public String getOutSideShareBaseUrl(DomainScene.DomainViewScene viewScene) {
        GetAvailableDomainInfoParam param = new GetAvailableDomainInfoParam();
        param.setScene(Objects.toString(DomainScene.OUTSIDE_SHARE));
        param.setViewScene(Objects.toString(viewScene));

        Optional<BaseUrlInfo> baseUrlByScene = getBaseUrlByScene(param);
        return baseUrlByScene.isPresent() ? baseUrlByScene.get().getBaseUrl() : "";
    }

    @Override
    public String getDownloadYYBUrl() {
        return specialUrlConfig.getDownloadYYBUrl();
    }

    @Override
    public String getAppIconUrl() {
        return specialUrlConfig.getAppIconUrl();
    }

    @Override
    public String getContentViewSceneBaseUrl() {
        GetViewSceneDomainInfoParam param = new GetViewSceneDomainInfoParam();
        param.setViewScene(Objects.toString(DomainViewScene.CONTENT_VIEW));
        return getBaseUrlByViewScene(param).getBaseUrl();
    }

    @Override
    public String getDownloadUrl() {
        return join(getContentViewSceneBaseUrl(), specialUrlConfig.getDownload());
    }

    @Override
    public String getServerHostBaseUrl() {
        return getByScene(DomainScene.SERVER_HOST);
    }

    private Optional<BaseUrlInfo> getBaseUrlByScene(GetAvailableDomainInfoParam param) {
        try {
            Optional<DomainScene> domainScene = DomainScene.forName(param.getScene());
            // 如果没有场景 则返回空
            if (!domainScene.isPresent()) {
                log.warn("请求参数: {} 的场景值不存在 无法获取域名信息", toJSONString(param));
                return Optional.empty();
            }

            // 获取落地页场景
            Optional<DomainViewScene> domainViewScene = DomainViewScene.forName(param.getViewScene());
            if (Objects.equals(domainScene.get(), DomainScene.OUTSIDE_SHARE_VIEW) && !domainViewScene.isPresent()) {
                log.warn("请求参数: {} 的落地页场景值不存在 无法获取域名信息", toJSONString(param));
                return Optional.empty();
            }

            // 根据场景类型 查询可用的域名 这里不需要穿落地页类型
            List<DomainInfo> domainInfos = selectAvailableDomainBySceneByCache(domainScene.get().getScene(),
                    null,
                    param.getAppId());

            // 随机获取一个
            Optional<DomainInfo> domainInfoOpt = getRandomInfo(domainInfos);
            if (!domainInfoOpt.isPresent()) {
                // 发送叮叮消息
                String msg = String.format("【紧急】类型: %s获取不到可用的域名 事件id: %s ", param.getScene(), sequenceCreater.nextStringId());
                messageSender.sendDingtalk(DingtalkMessage.builder()
                        .content(msg)
                        .build());
                log.warn(msg + "param: {}", toJSONString(param));
                return Optional.empty();
            }

            DomainInfo domainInfo = domainInfoOpt.get();
            BaseUrlInfo info = new BaseUrlInfo();
            info.setDomain(domainInfo.getDomain());
            // 拼接
            info.setBaseUrl(sceneBaseUrlJoiner.joinByScene(domainScene.get(), domainViewScene.orElse(null),
                    StringUtils.join(domainInfo.getProtocol(),
                            domainInfo.getDomain())));

            if (log.isDebugEnabled()) {
                log.debug("param: {} 最终获取到的base url信息: {}", toJSONString(param), toJSONString(info));
            }

            return Optional.of(info);
        } catch (Exception e) {
            // 发送叮叮消息
            String msg = String.format("【紧急】类型: %s获取可用的域名出错误了 事件id: %s ", param.getScene(),
                    sequenceCreater.nextStringId());
            messageSender.sendDingtalk(DingtalkMessage.builder()
                    .content(msg)
                    .build());
            log.error(msg + "param: {}", toJSONString(param), e);
        }

        return Optional.empty();
    }

    /**
     * 随机获取一个域名
     *
     * @param domainInfos 多个域名
     * @return 随机的一个
     */
    private Optional<DomainInfo> getRandomInfo(List<DomainInfo> domainInfos) {
        if (domainInfos.isEmpty()) {
            return Optional.empty();
        }

        return Optional.of(domainInfos.get(RandomUtils.nextInt(0, domainInfos.size())));
    }

    private ViewSceneInfo convert(DomainInfo domainInfo) {
        ViewSceneInfo info = new ViewSceneInfo();
        Optional<DomainViewScene> domainViewScene = DomainViewScene.forViewScene(domainInfo.getViewScene());

        info.setDomain(domainInfo.getDomain());
        info.setProtocol(domainInfo.getProtocol());
        info.setViewScene(domainViewScene.isPresent() ? Objects.toString(domainViewScene.get()) : "");
        info.setAppId(domainInfo.getAppId());

        return info;
    }

    private String getByScene(DomainScene scene) {
        GetAvailableDomainInfoParam param = new GetAvailableDomainInfoParam();
        param.setScene(Objects.toString(scene));
        BaseUrlInfo domainInfo = getBaseUrlInfo(param);
        return domainInfo.getBaseUrl();
    }

    /**
     * 根据场景获取可用的域名信息列表 优先查询缓存
     *
     * @param scene     required 场景值
     * @param viewScene 落地页场景
     * @param appId     option 某些场景是在微信环境下的，所以选用app id对应的域名
     * @return 域名列表
     */
    private List<DomainInfo> selectAvailableDomainBySceneByCache(byte scene, Byte viewScene, String appId) {

        SelectAvailableDomainBySceneParam param = new SelectAvailableDomainBySceneParam();
        param.setScene(scene);
        param.setViewScene(viewScene);
        param.setAppId(appId);
        return selectAvailableDomainBySceneByCache(param);
    }

    private List<DomainInfo> selectAvailableDomainBySceneByCache(SelectAvailableDomainBySceneParam param) {

        if (Objects.isNull(CACHE)) {
            CACHE = CacheBuilder.newBuilder()
                    .maximumSize(100)
                    // 1h失效
                    .expireAfterWrite(1, TimeUnit.HOURS)
                    .build(new CacheLoader<SelectAvailableDomainBySceneParam, List<DomainInfo>>() {
                        @Override
                        public List<DomainInfo> load(SelectAvailableDomainBySceneParam sceneParam) throws Exception {
                            // 根据落地页场景 获取对应的落地页域名
                            return baseDomainPublishExtendMapper.selectAvailableDomainByScene(sceneParam.getScene(),
                                    sceneParam.getViewScene(), sceneParam.getAppId());
                        }
                    });
        }

        // 如果缓存开关未开启 则每次都刷新一下 请求最新的
        if (!Objects.equals(domainConfig.getCacheSwitch(), Boolean.TRUE)) {
            CACHE.refresh(param);
        }

        return CACHE.getUnchecked(param);
    }
}
