package com.bxm.localnews.base.service.impl;

import com.bxm.localnews.base.domain.DictMapper;
import com.bxm.localnews.base.service.DictService;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.common.dto.DictDto;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.stream.Collectors;

@Service
@Slf4j
public class DictServiceImpl implements DictService {

    private RedisHashMapAdapter redisHashMapAdapter;

    private final DictMapper dictMapper;

    private TypeReference<List<DictDto>> typeReference = new TypeReference<List<DictDto>>() {
    };

    @Autowired
    public DictServiceImpl(RedisHashMapAdapter redisHashMapAdapter, DictMapper dictMapper) {
        this.redisHashMapAdapter = redisHashMapAdapter;
        this.dictMapper = dictMapper;
    }

    @Override
    public List<DictDto> load(String group, Integer deep) {
        Preconditions.checkArgument(null != group);

        List<DictDto> result = redisHashMapAdapter.get(RedisConfig.DICT_CACHE, group, typeReference);
        if (null == result) {
            result = dictMapper.query(group);
            redisHashMapAdapter.put(RedisConfig.DICT_CACHE, group, result);
        }

        List<DictDto> finalResult = result.stream()
                .filter(item -> null == item.getParentId())
                .collect(Collectors.toList());

        if (deep == null || deep == 0) {
            deep = Integer.MAX_VALUE;
        }

        for (DictDto dict : finalResult) {
            lookup(dict, result, deep - 1);
        }

        return finalResult;
    }

    private void lookup(DictDto parent, List<DictDto> data, int deep) {
        if (deep == 0) {
            return;
        }

        data.forEach(item -> {
            if (StringUtils.equals(String.valueOf(item.getParentId()), parent.getCode())) {
                parent.addChild(item);
                lookup(item, data, deep - 1);
            }
        });
    }

    @Override
    public List<DictDto> children(String parentCode) {
        if (StringUtils.isBlank(parentCode)) {
            log.info("获取下级数据字典时，请求参数为空，返回空数组");
            return Lists.newArrayList();
        }

        List<DictDto> result = redisHashMapAdapter.get(RedisConfig.DICT_CACHE, parentCode, typeReference);
        if (result == null) {
            result = dictMapper.queryByParent(parentCode);
            redisHashMapAdapter.put(RedisConfig.DICT_CACHE, parentCode, result);
        }

        return result;
    }
}





















