package com.bxm.localnews.merchant.service.activity.impl;

import com.bxm.localnews.merchant.domain.ActivityUnicomInfoMapper;
import com.bxm.localnews.merchant.domain.ActivityUnicomPhoneInfoMapper;
import com.bxm.localnews.merchant.entity.UnicomInfo;
import com.bxm.localnews.merchant.entity.UnicomPhoneInfo;
import com.bxm.localnews.merchant.param.MerchantReportParam;
import com.bxm.localnews.merchant.param.activity.UnicomMatchPhoneParam;
import com.bxm.localnews.merchant.param.activity.UnicomParam;
import com.bxm.localnews.merchant.service.activity.UnicomService;
import com.bxm.localnews.merchant.service.count.MerchantPageCollectService;
import com.bxm.localnews.merchant.utils.ValidatorlUtils;
import com.bxm.localnews.msg.facade.SmsFacadeService;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.tools.Validater;
import com.bxm.newidea.component.vo.Message;
import com.bxm.newidea.component.vo.PageWarper;
import lombok.AllArgsConstructor;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.BeanUtils;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Objects;

/**
 * @author : pf.w
 * @date : 2020/4/30 14:50
 */
@Service
@Log4j2
@AllArgsConstructor
public class UnicomServiceImpl implements UnicomService {

    private ActivityUnicomInfoMapper activityUnicomInfoMapper;

    private ActivityUnicomPhoneInfoMapper activityUnicomPhoneInfoMapper;

    private SmsFacadeService smsFacadeService;

    private MerchantPageCollectService merchantPageCollectService;

    /**
     * 浙江省
     */
    private static String PROVINCE = "330000";

    @Override
    public Message saveUnicomUserInfo(UnicomParam unicomParam) {
        log.debug("unicomParam {}", unicomParam);

        Message checkResult = checkParam(unicomParam);

        if (!checkResult.isSuccess()) {
            return checkResult;
        }
        //录入大王卡活动信息
        saveUnicom(unicomParam);

        //异步执行后续处理（活动订单相关信息更新等）
        afterLogic(unicomParam);

        return Message.build().setSuccess(true);
    }

    @Async
    public void afterLogic(UnicomParam unicomParam) {
        MerchantReportParam merchantReportParam = new MerchantReportParam();
        merchantReportParam.setTraceId(unicomParam.getTraceId());
        merchantReportParam.setToken(unicomParam.getActorUserId());
        merchantReportParam.setShareUserId(unicomParam.getShareUserId());
        merchantPageCollectService.orderContent(merchantReportParam);
    }

    @Override
    public PageWarper<String> getPhoneList(UnicomMatchPhoneParam unicomMatchPhoneParam) {
        log.debug("unicomMatchPhoneParam {}", unicomMatchPhoneParam);
        List<String> phoneList = activityUnicomPhoneInfoMapper.getPhoneList(unicomMatchPhoneParam);
        return new PageWarper<>(phoneList);
    }

    private void saveUnicom(UnicomParam unicomParam) {
        UnicomInfo unicomInfo = new UnicomInfo();
        BeanUtils.copyProperties(unicomParam, unicomInfo);
        activityUnicomInfoMapper.saveUnicomInfo(unicomInfo);

        if (Objects.nonNull(unicomParam.getSelectedPhoneNumb())) {

            UnicomPhoneInfo unicomPhoneInfo = UnicomPhoneInfo.builder()
                    .idCard(unicomParam.getIdCard())
                    .usable(1)
                    .phoneNumb(unicomParam.getSelectedPhoneNumb().trim())
                    .unicomId(unicomInfo.getId())
                    .build();

            activityUnicomPhoneInfoMapper.updateUnicomPhoneInfo(unicomPhoneInfo);
        }
    }

    private Message checkParam(UnicomParam unicomParam) {
        if (!ValidatorlUtils.checkIdentityCode(unicomParam.getIdCard())) {
            return Message.build(false).setMessage("身份证格式错误");
        }

        if (!Validater.checkPhone(unicomParam.getPhone())) {
            return Message.build(false).setMessage("手机号格式错误");
        }

        if (!unicomParam.getOwnerProvince().contains(PROVINCE) && StringUtils.isNotBlank(unicomParam
                .getSelectedPhoneNumb())) {
            return Message.build(false).setMessage("非浙江省不能选择卡号!");
        }
        //校验验证码
        boolean checkCode = smsFacadeService.verifySmsCodeByType((byte) 7, unicomParam.getPhone(), unicomParam.getSmsCode());
        if (!checkCode) {
            return Message.build(false).setMessage("短信验证码错误");
        }
        return Message.build(true);
    }
}
