package com.bxm.localnews.merchant.service.activity.ticket;

import com.bxm.localnews.merchant.dto.activity.ActivityAdvertInfoDTO;
import lombok.extern.log4j.Log4j2;
import org.springframework.stereotype.Component;

import java.util.Objects;

/**
 * 责任链模式实现站外广告出券逻辑
 * <p>
 * （后续可预见的出券逻辑存在不确定性，该处理模式不一定囊概，后续可做调整）
 * <p>
 * 每个处理类需要在{@link com.bxm.localnews.merchant.common.constant.TicketLevelEnum} 枚举中定义处理优先级和类型
 * 处理级别不能重复，否则会导致重复链路丢失
 * (级别升序排列，顺序执行)
 *
 * @author pf.w
 * @date 2020/8/4 15:24
 **/
@Component
@Log4j2
public abstract class AbstractTicketHandler {

    private AbstractTicketHandler nextHandler;

    /**
     * 链路执行
     *
     * @return 出券信息
     */
    public final ActivityAdvertInfoDTO handlerMessage(TicketContext request) {

        ActivityAdvertInfoDTO activityAdvertInfoDTO = this.response(request);

        if (Objects.nonNull(activityAdvertInfoDTO)) {
            return activityAdvertInfoDTO;
        }

        if (this.nextHandler != null) {
            activityAdvertInfoDTO = this.nextHandler.handlerMessage(request);
        } else {
            activityAdvertInfoDTO = null;
        }
        return activityAdvertInfoDTO;
    }

    /**
     * 设置下一个请求处理者
     *
     * @param handler 处理类
     */
    public void setNextHandler(AbstractTicketHandler handler) {
        this.nextHandler = handler;
    }

    /**
     * 处理级别 (自定义优先级，级别不能和其他链路重复)
     * <p>
     * {@link com.bxm.localnews.merchant.common.constant.TicketLevelEnum}
     * <p>
     * (级别升序排列，顺序执行)
     *
     * @return 处理级别
     */
    protected abstract int getHandlerLevel();

    /**
     * 处理内容
     *
     * @param context 上下文内容
     * @return 券信息
     */
    protected abstract ActivityAdvertInfoDTO response(TicketContext context);


}
