package com.bxm.localnews.merchant.service.lottery.cache;

import com.bxm.localnews.merchant.common.config.RedisConfig;
import com.bxm.localnews.merchant.dto.activity.UserJoinCodeDTO;
import com.bxm.localnews.merchant.service.config.ActivityProperties;
import com.bxm.localnews.merchant.service.enums.LotteryJoinSourceEnum;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import static com.bxm.localnews.merchant.common.config.RedisConfig.*;

/**
 * 夺宝活动的相关缓存统一管理
 *
 * @author liujia
 * @date 2020-10-27 14:14
 **/
@Component
public class LotteryCacheManage {

    private RedisHashMapAdapter redisHashMapAdapter;

    private RedisStringAdapter redisStringAdapter;

    private final ActivityProperties activityProperties;

    private final RedisSetAdapter redisSetAdapter;

    private TypeReference<List<UserJoinCodeDTO>> joinTypeReference = new TypeReference<List<UserJoinCodeDTO>>() {
    };

    @Autowired
    public LotteryCacheManage(RedisHashMapAdapter redisHashMapAdapter,
                              RedisStringAdapter redisStringAdapter,
                              ActivityProperties activityProperties,
                              RedisSetAdapter redisSetAdapter) {
        this.redisHashMapAdapter = redisHashMapAdapter;
        this.redisStringAdapter = redisStringAdapter;
        this.activityProperties = activityProperties;
        this.redisSetAdapter = redisSetAdapter;
    }

    /**
     * 活动期数结束后进行相关缓存的清理
     *
     * @param phaseId 活动期ID
     */
    public void clearPhaseCache(Long phaseId) {
        redisStringAdapter.remove(buildJoinNumKey(phaseId));
    }

    private KeyGenerator buildJoinKey(Long userId) {
        return USER_JOIN_TIMES_MAP.copy().appendKey(userId);
    }

    /**
     * 判断用户是否有全局参与次数
     *
     * @param userId 用户ID
     * @return true表示用户可以参与站外的夺宝活动
     */
    public Boolean hasGlobalJoinTimes(Long userId) {
        return !redisSetAdapter.exists(buildTodayGlobalJoinUser(), userId);
    }

    /**
     * 记录用户的全局参与次数
     *
     * @param userId 用户ID
     */
    public void consumeGlobalJoinTimes(Long userId) {
        KeyGenerator key = buildTodayGlobalJoinUser();
        redisSetAdapter.add(key, userId);

        redisSetAdapter.expire(key, DateUtils.getCurSeconds());
    }

    private KeyGenerator buildTodayGlobalJoinUser() {
        return TODAY_GLOBAL_JOIN_USER.copy().appendKey(DateUtils.PATTERN_NO_DELIMITER_FORMAT.get().format(new Date()));
    }

    private KeyGenerator buildFreeJoinUser() {
        return LOTTERY_FREE_JOIN_USER.copy().appendKey(DateUtils.PATTERN_NO_DELIMITER_FORMAT.get().format(new Date()));
    }

    /**
     * 用户是否有免费参与的机会
     *
     * @param userId 参与夺宝活动的新用户
     * @return true表示用户还可以免费参与
     */
    public Boolean hasFreeJoinTime(Long userId) {
        // 用户是否有免费参与机会
        Long joinTimes = redisHashMapAdapter.getLong(buildFreeJoinUser(), userId.toString());
        return joinTimes < activityProperties.getFreeTimes();
    }

    /**
     * 消耗一次免费参与机会
     *
     * @param userId 参与夺宝活动的新用户
     */
    public void consumeFreeJoinTime(Long userId) {
        KeyGenerator key = buildFreeJoinUser();
        redisHashMapAdapter.increment(key, userId.toString(), 1);
        redisHashMapAdapter.expire(key, DateUtils.getCurSeconds());
    }

    /**
     * 用户参与站内夺宝的次数
     *
     * @param phaseId 活动期ID
     * @param userId  用户ID
     * @param maxType 参与次数限制类型 0: 整个活动生命周期的次数限制（默认） 1: 每日参与次数限制
     * @return 参与次数
     */
    public Long getAppJoinTimes(Long phaseId, Long userId, int maxType) {
        KeyGenerator key = buildJoinKey(userId);
        Date now = new Date();

        List<UserJoinCodeDTO> userJoinCodes = redisHashMapAdapter.get(key, String.valueOf(phaseId), joinTypeReference);

        if (null == userJoinCodes) {
            return 0L;
        }

        return userJoinCodes.stream().filter(joinCode -> {
            boolean isApp = Objects.equals(joinCode.getType(), LotteryJoinSourceEnum.APP.getCode());
            return maxType == 0 ? isApp : isApp
                    // 如果是每日参与次数限制 需要判断参与的时间是否是当天 同时需要注意joinTime是新版本才有的需要判空
                    && (Objects.nonNull(joinCode.getJoinTime()) && DateUtils.isSameDay(joinCode.getJoinTime(), now));
        }).count();
    }

    /**
     * 记录用户参与夺宝的夺宝券号码
     *
     * @param phaseId  活动期数ID
     * @param userId   参与的用户ID
     * @param joinCode 用户获得的夺宝券编码
     * @param source   夺宝券来源
     */
    public void addJoinCode(Long phaseId, Long userId, String joinCode, Integer source) {
        KeyGenerator key = buildJoinKey(userId);
        String subKey = String.valueOf(phaseId);

        List<UserJoinCodeDTO> userJoinCodes = redisHashMapAdapter.get(key, subKey, joinTypeReference);

        if (null == userJoinCodes) {
            userJoinCodes = Lists.newArrayList();
        }

        userJoinCodes.add(UserJoinCodeDTO.builder()
                .code(joinCode)
                .type(source)
                .joinTime(new Date())
                .build());

        redisHashMapAdapter.put(key, subKey, userJoinCodes);
    }

    /**
     * 获取用户持有的活动编码
     *
     * @param phaseIds 活动期数ID列表
     * @param userId   用户ID
     * @return 活动期数对应的持有的夺宝券码
     */
    public Map<Long, List<String>> getMyCodes(List<Long> phaseIds, Long userId) {
        if (CollectionUtils.isEmpty(phaseIds)) {
            return Maps.newHashMap();
        }

        KeyGenerator key = buildJoinKey(userId);
        Map<Long, List<String>> result = Maps.newHashMap();

        Map<String, List<UserJoinCodeDTO>> entries = redisHashMapAdapter.entries(key, joinTypeReference);

        entries.forEach((phaseIdStr, codes) -> {
            if (phaseIds.stream().anyMatch(phaseId -> String.valueOf(phaseId).equals(phaseIdStr))) {
                result.put(Long.valueOf(phaseIdStr), codes.stream()
                        .map(UserJoinCodeDTO::getCode).collect(Collectors.toList()));
            }
        });

        return result;
    }

    /**
     * 用户是否关注并发放优惠券
     *
     * @param userId   用户id
     * @param couponId 优惠券id
     * @return 是否要发放
     */
    public Boolean hasFollowGrantCoupon(Long userId, Long couponId) {
        // 之前已经通过关注领取过优惠券
        return redisSetAdapter.exists(buildUserFollowCouponKey(userId), couponId);
    }

    /**
     * 关注商户后发放优惠券进行记录
     */
    public void followGrantUserCoupon(Long userId, Long couponId) {
        redisSetAdapter.add(buildUserFollowCouponKey(userId), couponId);
    }


    /**
     * 构建用户关注领取优惠券，每张优惠券每个用户关注只能领取一次
     *
     * @param userId 用户id
     * @return redis key
     */
    private KeyGenerator buildUserFollowCouponKey(Long userId) {
        return RedisConfig.USER_FOLLOW_COUPON.copy().appendKey(userId);
    }

    private KeyGenerator buildJoinNumKey(Long phaseId) {
        return PHASE_JOIN_NUM.copy().appendKey(phaseId);
    }

    /**
     * 增加活动期的参与人数
     *
     * @param phaseId 活动期ID
     * @return 当前参与人数
     */
    public Long incrementJoinNum(Long phaseId) {
        return redisStringAdapter.increment(buildJoinNumKey(phaseId));
    }

    /**
     * 获取活动当前的参与人数
     *
     * @param phaseId 活动ID
     * @return 参与人数
     */
    public Long getJoinNum(Long phaseId) {
        return redisStringAdapter.getLong(buildJoinNumKey(phaseId));
    }

    /**
     * 扣除活动期的参与人数
     *
     * @param phaseId 活动期ID
     * @return 扣除后的参与人数
     */
    public void decrementJoinNum(Long phaseId) {
        redisStringAdapter.decrement(buildJoinNumKey(phaseId));
    }
}
