package com.bxm.localnews.merchant.service.lottery.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.merchant.domain.MerchantGoodsMapper;
import com.bxm.localnews.merchant.domain.lottery.ActivityAwardMapper;
import com.bxm.localnews.merchant.dto.activity.LotteryAwardDetailDTO;
import com.bxm.localnews.merchant.dto.activity.LotteryAwardOverviewDTO;
import com.bxm.localnews.merchant.entity.lottery.LotteryAwardEntityWithBLOBs;
import com.bxm.localnews.merchant.param.activity.LotteryAwardManagePageParam;
import com.bxm.localnews.merchant.param.activity.LotteryChangeStatusParam;
import com.bxm.localnews.merchant.service.enums.AwardTypeEnum;
import com.bxm.localnews.merchant.service.enums.LotteryStatusEnum;
import com.bxm.localnews.merchant.service.lottery.LotteryAwardService;
import com.bxm.localnews.merchant.service.lottery.LotteryService;
import com.bxm.localnews.merchant.vo.goods.MerchantGoodsVo;
import com.bxm.localnews.merchants.facade.service.MerchantInfoFacadeService;
import com.bxm.localnews.merchants.vo.MerchantInfo;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import com.bxm.newidea.component.vo.PageWarper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;

import static org.apache.commons.lang3.StringUtils.isNotBlank;

/**
 * @author liujia
 * @date 2020-10-26 17:29
 **/
@Slf4j
@Service
public class LotteryAwardServiceImpl implements LotteryAwardService {

    private final ActivityAwardMapper activityAwardMapper;

    private final MerchantGoodsMapper merchantGoodsMapper;

    private final SequenceCreater sequenceCreater;

    private final MerchantInfoFacadeService merchantInfoFacadeService;

    @Autowired

    public LotteryAwardServiceImpl(ActivityAwardMapper activityAwardMapper,
                                   SequenceCreater sequenceCreater,
                                   MerchantGoodsMapper merchantGoodsMapper,
                                   MerchantInfoFacadeService merchantInfoFacadeService) {
        this.activityAwardMapper = activityAwardMapper;
        this.sequenceCreater = sequenceCreater;
        this.merchantGoodsMapper = merchantGoodsMapper;
        this.merchantInfoFacadeService = merchantInfoFacadeService;
    }

    @Override
    public PageWarper<LotteryAwardOverviewDTO> query(LotteryAwardManagePageParam param) {
        return new PageWarper<>(activityAwardMapper.queryByPage(param));
    }

    @Override
    public LotteryAwardDetailDTO get(Long awardId) {
        LotteryAwardEntityWithBLOBs detailEntity = activityAwardMapper.selectByPrimaryKey(awardId);

        LotteryAwardDetailDTO detailDTO = new LotteryAwardDetailDTO();

        BeanUtils.copyProperties(detailEntity, detailDTO);
        detailDTO.setAwardId(detailEntity.getId());
        detailDTO.setAwardTitle(detailEntity.getTitle());
        detailDTO.setAwardType(detailEntity.getType());
        detailDTO.setReplay(detailEntity.getReplayContent());
        detailDTO.setMerchantId(detailEntity.getSponsor());

        detailDTO.setCarouseStr(detailEntity.getCarousel());

        //回显时展示奖品名称
        if (detailDTO.getGoodsId() != null) {
            //获取商品名称
            MerchantGoodsVo merchantGoodsInfo = merchantGoodsMapper.selectByPrimaryKey(detailDTO.getGoodsId());
            if (merchantGoodsInfo != null) {
                detailDTO.setGoodsName(merchantGoodsInfo.getName());
            }
        }
        return detailDTO;
    }

    @Override
    public Message saveOrUpdate(LotteryAwardDetailDTO awardInfo) {
        LotteryAwardEntityWithBLOBs entity = new LotteryAwardEntityWithBLOBs();

        BeanUtils.copyProperties(awardInfo, entity);
        entity.setId(awardInfo.getAwardId());
        entity.setTitle(awardInfo.getAwardTitle());

        //改为使用,分割的字符串
        entity.setCarousel(awardInfo.getCarouseStr());
        entity.setType(awardInfo.getAwardType());
        entity.setCreator(awardInfo.getCurrentUserId());
        entity.setReplayContent(awardInfo.getReplay());
        entity.setSponsor(awardInfo.getMerchantId());

        //对轮播图进行逗号分割
        List<String> carouselList = new ArrayList<>();
        //如果轮播图地址不为空
        if (isNotBlank(awardInfo.getCarouseStr())) {
            //分割字符串
            carouselList = Arrays.asList(awardInfo.getCarouseStr().split(","));
        }

        //如果轮播图列表不为空
        if (!CollectionUtils.isEmpty(carouselList)) {
            //取第一个图片为头图
            entity.setFirstImg(carouselList.get(0));
        }

        int result = 0;

        entity.setWelfareType(awardInfo.getWelfareType());
        entity.setWelfareId(awardInfo.getWelfareId());

        if (null == awardInfo.getAwardId()) {
            entity.setCreateTime(new Date());
            entity.setId(sequenceCreater.nextLongId());
            entity.setUseStock(0);
            //新增奖品时，当有赞助商家，判断赞助商家是否存在
            if (Objects.nonNull(entity.getMerchantId())) {
                MerchantInfo merchantInfo = merchantInfoFacadeService.getMerchantInfo(entity.getMerchantId());
                if (Objects.isNull(merchantInfo)) {
                    return Message.build(false, "赞助的商户不存在，请重新选择");
                }
            }

            result = activityAwardMapper.insert(entity);
        } else {
            Message message = beforeUpdate(entity);

            if (message.isSuccess()) {

                entity.setModifyTime(new Date());
                result = activityAwardMapper.updateByPrimaryKeySelective(entity);
            }
        }

        return Message.build(result);
    }

    /**
     * 夺宝奖品更新前进行库存相关的判断处理
     * 变更后的库存不可小于已使用的库存
     *
     * @param entity 奖品信息
     * @return 库存扣除结果
     */
    private Message beforeUpdate(LotteryAwardEntityWithBLOBs entity) {
        if (Objects.equals(entity.getType(), AwardTypeEnum.GOODS.getCode())) {
            LotteryAwardEntityWithBLOBs existsAward = activityAwardMapper.selectByPrimaryKey(entity.getId());

            if (existsAward.getUseStock() > entity.getStock()) {
                log.info("库存数量编辑错误，请求参数：{}", JSON.toJSONString(entity));
                return Message.build(false, "库存数量不能小于历史已使用库存");
            }
        }
        return Message.build();
    }

    @Override
    public List<LotteryAwardDetailDTO> searchAward(String awardTitle) {
        return activityAwardMapper.queryTopMatchAward(awardTitle);
    }

    @Override
    public List<Long> getExpiredLotteryWithAward() {
        return activityAwardMapper.queryExpiredLottery(LotteryStatusEnum.ACTIVED.getCode());
    }

    private LotteryService lotteryService;

    private LotteryService getLotteryService() {
        if (lotteryService == null) {
            lotteryService = SpringContextHolder.getBean(LotteryService.class);
        }
        return lotteryService;
    }

    @Override
    public Message changeStock(Long awardId, int num) {
        LotteryAwardEntityWithBLOBs awardInfo = activityAwardMapper.selectByPrimaryKey(awardId);

        if (num < 0) {
            // 如果是扣除库存的情况下，配置库存已经和使用库存一致，则认为库存已使用
            if (awardInfo.getStock().equals(awardInfo.getUseStock())) {
                log.info("奖品[{}]库存扣除完毕，更新奖品对应的活动状态，将其设置为下线", awardId);

                // 将奖品关联的活动进行下线操作
                List<Long> activeLotteries = activityAwardMapper.queryRelationActionLottery(awardId, LotteryStatusEnum.ACTIVED.getCode());

                for (Long activeLottery : activeLotteries) {
                    LotteryChangeStatusParam changeParam = new LotteryChangeStatusParam();
                    changeParam.setLotteryId(activeLottery);
                    changeParam.setEnable(false);

                    getLotteryService().changeStatus(changeParam, LotteryStatusEnum.STOCK_OVER);
                }
            }

            // 如果是扣除，并且已用库存已经小于设定库存，则扣除失败
            if (awardInfo.getStock() < awardInfo.getUseStock() + Math.abs(num)) {
                return Message.build(false, "库存不足");
            }
        }

        return Message.build(activityAwardMapper.updateStock(awardId, num));
    }
}



















