package com.bxm.localnews.merchant.service.lottery.impl;

import com.bxm.localnews.common.constant.PlatformEnum;
import com.bxm.localnews.merchant.domain.lottery.*;
import com.bxm.localnews.merchant.dto.UserInfoDTO;
import com.bxm.localnews.merchant.dto.activity.LotteryInviteInfoDTO;
import com.bxm.localnews.merchant.dto.activity.LotteryJoinStateDTO;
import com.bxm.localnews.merchant.dto.activity.LotteryParticipatorDTO;
import com.bxm.localnews.merchant.dto.activity.LotteryPhaseDetailDTO;
import com.bxm.localnews.merchant.entity.lottery.LotteryAwardEntityWithBLOBs;
import com.bxm.localnews.merchant.entity.lottery.LotteryEntity;
import com.bxm.localnews.merchant.integration.UserIntegrationService;
import com.bxm.localnews.merchant.param.activity.LotteryCheckJoinTimesParam;
import com.bxm.localnews.merchant.param.activity.LotteryPhaseDetailParam;
import com.bxm.localnews.merchant.service.lottery.LotteryDetailCompositeService;
import com.bxm.localnews.merchant.service.lottery.LotteryMerchantService;
import com.bxm.localnews.merchant.service.lottery.LotteryParticipatorService;
import com.bxm.localnews.merchant.service.lottery.LotteryStatisticsService;
import com.bxm.localnews.merchant.service.lottery.cache.LotteryCacheManage;
import com.bxm.localnews.merchant.service.lottery.event.LotteryShowEvent;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import static org.apache.commons.lang3.StringUtils.isBlank;

/**
 * @author wzy
 * @version 1.0
 * @date 2020/10/29 7:18 下午
 */
@Service
public class LotteryDetailCompositeServiceImpl implements LotteryDetailCompositeService {

    private final LotteryPhaseMapper lotteryPhaseMapper;

    private final LotteryCacheManage lotteryCacheManage;

    private final LotteryParticipatorMapper lotteryParticipatorMapper;

    private final LotteryParticipatorService lotteryParticipatorService;

    private final UserIntegrationService userIntegrationService;

    private final LotteryMapper lotteryMapper;

    private final ActivityAwardMapper activityAwardMapper;

    private final ActivityInviteHistoryMapper activityInviteHistoryMapper;

    private final LotteryMerchantService lotteryMerchantService;

    private final LotteryStatisticsService lotteryStatisticsService;

    public LotteryDetailCompositeServiceImpl(LotteryPhaseMapper lotteryPhaseMapper,
                                             LotteryCacheManage lotteryCacheManage,
                                             LotteryParticipatorService lotteryParticipatorService,
                                             UserIntegrationService userIntegrationService,
                                             LotteryMapper lotteryMapper,
                                             LotteryParticipatorMapper lotteryParticipatorMapper,
                                             ActivityAwardMapper activityAwardMapper,
                                             ActivityInviteHistoryMapper activityInviteHistoryMapper,
                                             LotteryMerchantService lotteryMerchantService,
                                             LotteryStatisticsService lotteryStatisticsService) {
        this.lotteryPhaseMapper = lotteryPhaseMapper;
        this.lotteryCacheManage = lotteryCacheManage;
        this.lotteryParticipatorService = lotteryParticipatorService;
        this.userIntegrationService = userIntegrationService;
        this.lotteryMapper = lotteryMapper;
        this.lotteryParticipatorMapper = lotteryParticipatorMapper;
        this.activityAwardMapper = activityAwardMapper;
        this.activityInviteHistoryMapper = activityInviteHistoryMapper;
        this.lotteryMerchantService = lotteryMerchantService;
        this.lotteryStatisticsService = lotteryStatisticsService;
    }


    @Override
    public LotteryPhaseDetailDTO getLotteryDetail(LotteryPhaseDetailParam param) {
        LotteryPhaseDetailDTO phaseDetailResult = lotteryPhaseMapper.getPhaseDetail(param);

        LotteryEntity lotteryEntity = lotteryMapper.selectByPrimaryKey(phaseDetailResult.getLotteryId());
        LotteryAwardEntityWithBLOBs awardInfo = activityAwardMapper.selectByPrimaryKey(lotteryEntity.getAwardId());

        //设置最后一期参与活动id
        phaseDetailResult.setLastPhaseId(getProceedLastPhaseId(lotteryEntity.getId()));

        //设置活动轮播图
        phaseDetailResult.setImgList(getCarouselList(awardInfo.getCarousel()));

        //设置近期参与人员
        phaseDetailResult.setRecentParticipatorList(getRecentParticipatorList(param.getPhaseId()));

        //用户未登录且是站外请求
        if (param.getUserId() == null && PlatformEnum.WEB.getCode() == param.getPlatform()) {
            // 站外可以免费参与1次
            phaseDetailResult.setExpendGold(0);
            phaseDetailResult.setJoinTimes(0);
            phaseDetailResult.setMyCodes(Collections.emptyList());
            if (awardInfo.getMerchantId() != null) {
                phaseDetailResult.setMerchantInfo(lotteryMerchantService.get(awardInfo.getMerchantId()));
            }
            //夺宝码和邀请数据均为null
            return phaseDetailResult;
        }

        //如果是站内/站外已经登录
        LotteryJoinStateDTO joinInfo = getJoinInfo(lotteryEntity.getMaxTimes(), lotteryEntity.getMaxType(), param.getPhaseId(), param.getUserId(), param.getPlatform());
        // 设置可参与次数
        phaseDetailResult.setJoinTimes(joinInfo.getJoinTimes());
        // 如果是免费则消耗0朵小红花，否则则消耗固定数量小红花
        phaseDetailResult.setExpendGold(joinInfo.getFree() ? 0 : lotteryEntity.getExpendGold());
        // 如果赞助商家 不为null
        if (awardInfo.getMerchantId() != null) {
            phaseDetailResult.setMerchantInfo(lotteryMerchantService.get(awardInfo.getMerchantId(), param.getUserId()));
            // 商家一句话描述可能为空，这里使用在夺宝后台配置的一句话简介
            phaseDetailResult.getMerchantInfo().setDescription(awardInfo.getIntro());
        }
        //设置我的夺宝码
        List<String> myCodeList = getMyCodeList(param.getPhaseId(), param.getUserId());
        phaseDetailResult.setMyCodes(myCodeList);

        // 只有为站内的时候才返回邀请数据，减少站外请求时间
        if (PlatformEnum.WEB.getCode() != param.getPlatform()) {
            //设置邀请数据，只有站内有用
            phaseDetailResult.setInviteInfo(getInviteInfo(param.getPhaseId(), param.getUserId()));
        }

        // 记录详情曝光
        lotteryStatisticsService.post(new LotteryShowEvent(lotteryEntity.getId(), param.getPhaseId()));

        return phaseDetailResult;
    }

    /**
     * 获取用户的参与信息
     *
     * @param maxTimes 当前活动最大参与次数（站内）
     * @param phaseId  期数Id
     * @param maxType 参与次数限制类型 0: 整个活动生命周期的次数限制（默认） 1: 每日参与次数限制
     * @param userId   用户id
     * @param platform 平台类型
     * @return 用户的可参与数据
     */
    private LotteryJoinStateDTO getJoinInfo(Integer maxTimes, Integer maxType, Long phaseId, Long userId, Integer platform) {
        UserInfoDTO userCache = userIntegrationService.getUserFromRedisDB(userId);

        LotteryCheckJoinTimesParam param = LotteryCheckJoinTimesParam.builder()
                .phaseId(phaseId)
                .platform(platform)
                .userId(userId)
                .userCreateTime(userCache.getCreateTime())
                .maxTimes(maxTimes)
                .maxType(maxType)
                .build();
        return lotteryParticipatorService.getJoinInfo(param);
    }


    /**
     * 获取用户的夺宝码信息
     *
     * @param phaseId 期数id
     * @param userId  用户id
     * @return 我的夺宝码
     */
    public List<String> getMyCodeList(Long phaseId, Long userId) {
        Map<Long, List<String>> codeMap =
                lotteryCacheManage.getMyCodes(Collections.singletonList(phaseId), userId);

        List<String> myCodeList = codeMap.get(phaseId);
        if (CollectionUtils.isEmpty(myCodeList)) {
            return Collections.emptyList();
        }
        return myCodeList;
    }

    /**
     * 获取最新正在进行中的期数id
     *
     * @param lotteryId 活动id
     * @return 正在进行中的最新的期数id
     */
    public Long getProceedLastPhaseId(Long lotteryId) {
        return lotteryPhaseMapper.getProceedLastPhaseId(lotteryId);
    }

    /**
     * 获取奖品轮播图列表
     *
     * @param carousel 轮播图字符串
     * @return 轮播图列表
     */
    public List<String> getCarouselList(String carousel) {
        if (isBlank(carousel)) {
            return Collections.emptyList();
        }
        return Arrays.asList(carousel.split(","));
    }

    /**
     * 获取最近参与者列表【详情中的弹幕用】
     *
     * @param phaseId 期数id
     * @return 近期参与者列表
     */
    public List<LotteryParticipatorDTO> getRecentParticipatorList(Long phaseId) {
        return lotteryParticipatorMapper.getParticipatorListByPhase(phaseId);
    }

    /**
     * 获取用户的邀请信息
     *
     * @param phaseId 期数id
     * @param userId  用户id
     * @return 用户的邀请信息
     */
    public LotteryInviteInfoDTO getInviteInfo(Long phaseId, Long userId) {

        LotteryInviteInfoDTO resultDTO = new LotteryInviteInfoDTO();

        int inviteNum = activityInviteHistoryMapper.countUserInvite(userId, phaseId);
        List<LotteryParticipatorDTO> userInviteList = activityInviteHistoryMapper.getUserInvite(userId, phaseId);

        resultDTO.setInviteNum(inviteNum);
        resultDTO.setInviteUsers(userInviteList);

        return resultDTO;
    }
}