package com.bxm.localnews.merchant.service.lottery.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.merchant.domain.lottery.LotteryParticipatorMapper;
import com.bxm.localnews.merchant.domain.lottery.LotteryPhaseMapper;
import com.bxm.localnews.merchant.domain.lottery.LotteryWinnerMapper;
import com.bxm.localnews.merchant.dto.LocationDTO;
import com.bxm.localnews.merchant.dto.UserInfoDTO;
import com.bxm.localnews.merchant.dto.activity.LotteryAwardDetailDTO;
import com.bxm.localnews.merchant.dto.activity.LotteryPhaseCalculateDTO;
import com.bxm.localnews.merchant.dto.activity.LotteryPhaseCalculatePartDTO;
import com.bxm.localnews.merchant.dto.activity.LotteryPhaseCoreInfoDTO;
import com.bxm.localnews.merchant.entity.lottery.ActivityPrizeEntity;
import com.bxm.localnews.merchant.entity.lottery.LotteryParticipatorEntity;
import com.bxm.localnews.merchant.entity.lottery.LotteryWinnerEntity;
import com.bxm.localnews.merchant.integration.LocationIntegrationService;
import com.bxm.localnews.merchant.integration.UserIntegrationService;
import com.bxm.localnews.merchant.param.activity.LotteryPhaseDetailParam;
import com.bxm.localnews.merchant.param.activity.LotteryPhaseJoinParam;
import com.bxm.localnews.merchant.service.config.ActivityProperties;
import com.bxm.localnews.merchant.service.enums.*;
import com.bxm.localnews.merchant.service.lottery.*;
import com.bxm.localnews.merchant.service.lottery.context.LotteryJoinContext;
import com.bxm.localnews.merchant.service.lottery.strategy.JoinStrategyManage;
import com.bxm.localnews.merchants.facade.service.MerchantInfoFacadeService;
import com.bxm.localnews.merchants.vo.MerchantInfo;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.vo.Message;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.math.NumberUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author liujia
 * @date 2020-10-28 10:45
 **/
@Service
@Slf4j
@AllArgsConstructor(onConstructor = @__(@Autowired))
public class LotteryDrawServiceImpl implements LotteryDrawService {

    private final JoinStrategyManage joinStrategyManage;

    private final LotteryParticipatorMapper lotteryParticipatorMapper;

    private final static Integer CAL_NUM = 50;

    private final ActivityProperties activityProperties;

    private final UserIntegrationService userIntegrationService;

    private final LotteryPhaseService lotteryPhaseService;

    private final LotteryWinnerService lotteryWinnerService;

    private final LotteryPrizeService lotteryPrizeService;

    private final LotteryAwardService lotteryAwardService;

    private final LotteryPhaseMapper lotteryPhaseMapper;

    private final LotteryWinnerMapper lotteryWinnerMapper;

    private final LotteryPushService lotteryPushService;

    private final MerchantInfoFacadeService merchantInfoFacadeService;

    private final LocationIntegrationService locationIntegrationService;

    private final static ThreadLocal<DateFormat> CUSTOM_FORMAT = ThreadLocal.withInitial(() ->
            new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS"));

    private LotteryParticipatorEntity getVirtualUser(Date lastJoinTime) {
        List<UserInfoDTO> virtualUserList = userIntegrationService.getVirtualUserList(1);
        UserInfoDTO userInfoDTO = virtualUserList.get(0);

        LotteryParticipatorEntity entity = new LotteryParticipatorEntity();
        entity.setUserId(userInfoDTO.getId());
        entity.setNickName(userInfoDTO.getNickname());
        entity.setHeadImg(userInfoDTO.getHeadImg());
        entity.setCreateTime(DateUtils.addField(lastJoinTime, Calendar.SECOND, 1));
        return entity;
    }

    @Override
    public void execAsyncDraw(LotteryJoinContext context) {
        try {
            TimeUnit.SECONDS.sleep(1);
        } catch (InterruptedException e) {
            //do nothing
        }

        log.info("活动开奖，请求参数：{}", JSON.toJSONString(context));

        LotteryPhaseCoreInfoDTO phaseInfo = context.getPhaseInfo();

        // 获取最后49个参与人，补充一个虚拟用户，构成开奖需要的50个人
        List<LotteryParticipatorEntity> lastJoinUsers = lotteryParticipatorMapper.getLastJoinUser(phaseInfo.getPhaseId(), CAL_NUM - 1);

        if (lastJoinUsers.size() < CAL_NUM - 1) {
            log.error("活动[{}]开奖失败，人数不足", phaseInfo.getPhaseId());
            abandonPhase(context);
            return;
        }

        // 最后一位参与者的参与时间，以此作为基点进行偏差值的增加，作为最后一个马甲号参与的时间
        Date lastUserJoinTime = lastJoinUsers.get(0).getCreateTime();
        LotteryParticipatorEntity virtualUser = getVirtualUser(lastUserJoinTime);

        // 判断是真实用户中奖还是虚拟用户中奖，随机获取一个中奖者
        LotteryParticipatorEntity winner;
        if (Objects.equals(phaseInfo.getActualFlag(), 1)) {
            winner = lotteryParticipatorMapper.getRandomParticipator(phaseInfo.getPhaseId(), LotteryUserTypeEnum.ACTURAL.getCode());
        } else {
            winner = lotteryParticipatorMapper.getRandomParticipator(phaseInfo.getPhaseId(), LotteryUserTypeEnum.VIRTURAL.getCode());
        }

        if (winner == null) {
            // 如果活动设置的虚拟用户开奖，并且没有找到虚拟用户，则最后一个虚拟用户作为中奖人
            if (Objects.equals(phaseInfo.getActualFlag(), 0)) {
                log.info("活动[{}]设定为虚拟用户开奖，但是没有足够的虚拟用户，设置为最后一位虚拟用户直接中奖", phaseInfo.getPhaseId());

                virtualUser.setCode(String.valueOf(activityProperties.getJoinCodeStartIndex() + phaseInfo.getTargetNum()));
                winner = virtualUser;
            } else {
                log.error("活动[{}]没有符合条件的中奖人员", phaseInfo.getPhaseId());
                abandonPhase(context);
                return;
            }
        }

        lastJoinUsers.add(virtualUser);

        // 进行计算，处理偏差值，将最后一个虚拟用户保存到参与人员信息中
        long offset = getOffset(lastJoinUsers, context, winner);
        virtualUser.setCreateTime(DateUtils.addField(virtualUser.getCreateTime(), Calendar.MILLISECOND, (int) -offset));

        // 抽奖结束后的相关处理
        afterDraw(context, virtualUser, winner);
    }

    /**
     * 废弃活动期数
     *
     * @param context 开奖参数
     */
    private void abandonPhase(LotteryJoinContext context) {
        log.error("强制终止活动，并尝试重新开启，相关参数：{}", JSON.toJSONString(context));

        LotteryPhaseCoreInfoDTO phaseInfo = context.getPhaseInfo();
        lotteryPhaseService.cancel(phaseInfo.getLotteryId(), phaseInfo.getPhaseId(), true);

        lotteryPhaseService.create(phaseInfo.getLotteryId());
    }

    /**
     * 处理中奖结果
     */
    private void afterDraw(LotteryJoinContext context, LotteryParticipatorEntity virtualUser, LotteryParticipatorEntity winner) {
        LotteryPhaseCoreInfoDTO phaseInfo = context.getPhaseInfo();
        // 添加最后一位虚拟参与者
        addLastJoinVirtual(context, virtualUser);

        // 关闭活动期
        Message message = lotteryPhaseService.close(phaseInfo.getLotteryId(), phaseInfo.getPhaseId());
        if (message.isSuccess()) {
            // 保存获奖者信息
            addWinner(context, winner, virtualUser.getCreateTime());

            // 开启下一期活动
            lotteryPhaseService.create(phaseInfo.getLotteryId());

            // 开奖后的相关推送，给参与者推送开奖通知
            List<Long> joinUserIds = lotteryParticipatorMapper.getJoinUserIds(phaseInfo.getPhaseId(),
                    LotteryUserTypeEnum.ACTURAL.getCode());

            //公众号推送需要商户的经纬度和areaname
            LocationDTO locationByGeocode = locationIntegrationService.getLocationByGeocode(context.getParam().getAreaCode());
            MerchantInfo merchantInfo = null;
            if (phaseInfo.getMerchantId() != null) {
                merchantInfo = merchantInfoFacadeService.getMerchantInfo(phaseInfo.getMerchantId());
            }
            for (Long userId : joinUserIds) {
                lotteryPushService.lotteryDraw(userId, phaseInfo.getLotteryTitle());
                //给用户推送公众号消息
                lotteryPushService.lotteryDrawOfficeAccountMsg(merchantInfo, context, phaseInfo.getLotteryTitle(), userId, locationByGeocode.getName());
            }
        }
    }

    /**
     * 添加中奖者信息
     *
     * @param context 开奖信息上下文
     * @param winner  获奖人员
     */
    private void addWinner(LotteryJoinContext context, LotteryParticipatorEntity winner, Date winnerTime) {
        LotteryPhaseCoreInfoDTO phaseInfo = context.getPhaseInfo();

        UserInfoDTO cacheUser = userIntegrationService.getUserFromRedisDB(winner.getUserId());

        LotteryWinnerEntity entity = new LotteryWinnerEntity();
        entity.setAwardId(phaseInfo.getAwardId());
        entity.setCode(Integer.valueOf(winner.getCode()));
        entity.setCreateTime(winnerTime);
        entity.setUserId(winner.getUserId());
        entity.setHeadImg(winner.getHeadImg());
        entity.setNickName(winner.getNickName());
        entity.setPhone(cacheUser.getPhone());
        entity.setPhaseId(phaseInfo.getPhaseId());
        entity.setLotteryId(phaseInfo.getLotteryId());
        entity.setStatus(LotteryWinnerStatusEnum.UN_RECEIVE.getCode());

        Message message = lotteryWinnerService.saveWinner(entity);

        if (message.isSuccess()) {
            // 保存奖品信息
            LotteryAwardDetailDTO awardInfo = lotteryAwardService.get(entity.getAwardId());

            ActivityPrizeEntity prizeEntity = new ActivityPrizeEntity();
            prizeEntity.setBizId(entity.getId());
            prizeEntity.setCreateTime(winnerTime);

            prizeEntity.setType(awardInfo.getAwardType());

            //商品默认是已发奖，其他是未发奖,需要运营手工发奖
            if (Objects.equals(AwardTypeEnum.GOODS.getCode(), awardInfo.getAwardType())) {
                prizeEntity.setStatus(ActivityPrizeStatusEnum.GRANT.getCode());
            } else {
                prizeEntity.setStatus(ActivityPrizeStatusEnum.UN_GRANT.getCode());
            }

            lotteryPrizeService.savePrizeInfo(prizeEntity);
        }
    }

    /**
     * 将最后一个虚拟用户插入到参与用户中
     *
     * @param context     开奖参数上下文
     * @param virtualUser 最后一个补位的虚拟用户
     */
    private void addLastJoinVirtual(LotteryJoinContext context, LotteryParticipatorEntity virtualUser) {
        LotteryPhaseJoinParam param = new LotteryPhaseJoinParam();
        param.setUserId(virtualUser.getUserId());
        param.setPhaseId(context.getPhaseInfo().getPhaseId());

        LotteryJoinContext joinContext = LotteryJoinContext.builder()
                .param(param)
                .joinSource(LotteryJoinSourceEnum.VIRTUAL)
                .skipTriggerDraw(true)
                .joinTime(virtualUser.getCreateTime())
                .build();

        joinStrategyManage.execute(joinContext);
    }

    /**
     * 获取偏差值，用于手工设定最后一个中奖人员的毫秒数
     *
     * @return 中奖人员与计算结果的偏差值
     */
    private long getOffset(List<LotteryParticipatorEntity> lastJoinUsers,
                           LotteryJoinContext context,
                           LotteryParticipatorEntity winner) {
        long sum = lastJoinUsers.stream().mapToLong(joinUser -> convert(joinUser.getCreateTime())).sum();

        Integer targetNum = context.getPhaseInfo().getTargetNum();

        long winnerNum = (sum % targetNum) + 1;
        long winnerCode = Long.valueOf(winner.getCode()) - activityProperties.getJoinCodeStartIndex();

        return winnerNum - winnerCode;
    }

    /**
     * 将日期转换为整形，如：2020-02-10 10:01:11.123
     * 转换结果为：100111123
     * 取时分秒毫秒组成字符串，然后转换为Long型
     *
     * @param date 原始日期数据
     * @return 转化结果
     */
    private Long convert(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);

        int hour = calendar.get(Calendar.HOUR_OF_DAY);
        int minute = calendar.get(Calendar.MINUTE);
        int second = calendar.get(Calendar.SECOND);
        int millisecond = calendar.get(Calendar.MILLISECOND);

        String finalNumStr = hour +
                String.format("%02d", minute) +
                String.format("%02d", second) +
                String.format("%03d", millisecond);
        return NumberUtils.toLong(finalNumStr);
    }

    @Override
    public LotteryPhaseCalculateDTO get(LotteryPhaseDetailParam param) {
        LotteryPhaseCoreInfoDTO corePhaseInfo = lotteryPhaseMapper.getCorePhaseInfo(param.getPhaseId());

        if (null == corePhaseInfo || !Objects.equals(corePhaseInfo.getStatus(), LotteryPhaseStatusEnum.FINISH.getCode())) {
            if (log.isDebugEnabled()) {
                log.debug("活动[{}]还未结束，不显示计算过程", param.getPhaseId());
            }
            return LotteryPhaseCalculateDTO.builder().build();
        }

        List<LotteryParticipatorEntity> lastJoinUserList = lotteryParticipatorMapper.getLastJoinUser(param.getPhaseId(), CAL_NUM);
        LotteryWinnerEntity winner = lotteryWinnerMapper.selectByPhaseId(param.getPhaseId());

        long total = lastJoinUserList.stream().mapToLong(user -> convert(user.getCreateTime())).sum();

        List<LotteryPhaseCalculatePartDTO> partList = lastJoinUserList.stream().map(user -> {
            return LotteryPhaseCalculatePartDTO.builder()
                    .joinTime(CUSTOM_FORMAT.get().format(user.getCreateTime()))
                    .partJoinTime(String.valueOf(convert(user.getCreateTime())))
                    .build();
        }).collect(Collectors.toList());

        return LotteryPhaseCalculateDTO.builder()
                .totalNum(corePhaseInfo.getTargetNum())
                .total(total)
                .partList(partList)
                .result(String.valueOf(winner.getCode()))
                .build();
    }
}
