package com.bxm.localnews.merchant.service.lottery.impl;

import com.bxm.localnews.merchant.domain.lottery.*;
import com.bxm.localnews.merchant.dto.activity.*;
import com.bxm.localnews.merchant.entity.lottery.LotteryAwardEntityWithBLOBs;
import com.bxm.localnews.merchant.entity.lottery.LotteryCountNumEntity;
import com.bxm.localnews.merchant.entity.lottery.LotteryCounterEntity;
import com.bxm.localnews.merchant.entity.lottery.LotteryPhaseEntity;
import com.bxm.localnews.merchant.param.activity.*;
import com.bxm.localnews.merchant.service.enums.*;
import com.bxm.localnews.merchant.service.lottery.LotteryListCompositeService;
import com.bxm.localnews.merchant.service.lottery.cache.LotteryCacheManage;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.PageWarper;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static com.github.pagehelper.page.PageMethod.startPage;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

/**
 * @author wzy
 * @version 1.0
 * @date 2020/10/27 3:56 下午
 */
@Service
public class LotteryListCompositeServiceImpl implements LotteryListCompositeService {

    private final LotteryMapper lotteryMapper;
    private final ActivityAwardMapper activityAwardMapper;
    private final LotteryCounterMapper lotteryCounterMapper;
    private final LotteryPhaseMapper lotteryPhaseMapper;
    private final LotteryWinnerMapper lotteryWinnerMapper;
    private final LotteryParticipatorMapper lotteryParticipatorMapper;
    private final LotteryCacheManage lotteryCacheManage;

    public LotteryListCompositeServiceImpl(LotteryMapper lotteryMapper,
                                           LotteryCounterMapper lotteryCounterMapper,
                                           LotteryPhaseMapper lotteryPhaseMapper,
                                           LotteryCacheManage lotteryCacheManage,
                                           LotteryParticipatorMapper lotteryParticipatorMapper,
                                           LotteryWinnerMapper lotteryWinnerMapper,
                                           ActivityAwardMapper activityAwardMapper) {
        this.lotteryMapper = lotteryMapper;
        this.lotteryCounterMapper = lotteryCounterMapper;
        this.lotteryPhaseMapper = lotteryPhaseMapper;
        this.lotteryCacheManage = lotteryCacheManage;
        this.lotteryParticipatorMapper = lotteryParticipatorMapper;
        this.lotteryWinnerMapper = lotteryWinnerMapper;
        this.activityAwardMapper = activityAwardMapper;
    }

    @Override
    public PageWarper<MerchantLotteryAwardDTO> queryMerchantLotteryList(MerchantLotteryAwardPageParam param) {
        // 如果是已结束进行特殊处理,这里以期数为主体进行展示
        if (LotteryMerchantTabEnum.UNDERWAY.getCode().equals(param.getStatus())
                || LotteryMerchantTabEnum.END.getCode().equals(param.getStatus())) {
            startPage(param.getPageNum(), param.getPageSize());
            PageWarper<MerchantLotteryAwardDTO> merchantPhasePageInfo = new PageWarper<>(lotteryPhaseMapper.queryMerchantPhaseListByStatus(param));

            merchantPhasePageInfo.getList().forEach(item -> {
                //对开奖奖品使用状态进行转换,需求更改，这里只在已结束情况下才会对状态进行处理
                if (LotteryMerchantTabEnum.END.getCode().equals(param.getStatus())) {
                    convertMerchantPhaseItem(item);
                }
                //设置数据统计信息，根据期数进行统计的曝光等信息
                LotteryCounterEntity lotteryCounterEntity = lotteryCounterMapper.countPhaseNum(item.getCurrentPhaseId());
                if (lotteryCounterEntity == null) {
                    item.setFunsNum(0);
                    item.setShowNum(0);
                    item.setJoinNum(0);
                } else {
                    item.setFunsNum(lotteryCounterEntity.getFunsNum());
                    item.setShowNum(lotteryCounterEntity.getShowNum());
                    item.setJoinNum(lotteryCounterEntity.getJoinNum());
                }
            });
            return merchantPhasePageInfo;
        } else {
            // 如果是进行中tab，以活动为主体进行查询
            startPage(param.getPageNum(), param.getPageSize());
            PageWarper<MerchantLotteryAwardDTO> merchantLotteryAwardPageInfo = new PageWarper<>(lotteryMapper.queryMerchantLotteryListByStatus(param));

            merchantLotteryAwardPageInfo.getList().forEach(item -> {
                //获取每个活动的统计信息
                LotteryCountNumEntity lotteryCountNumEntity = lotteryCounterMapper.countLotteryNum(item.getLotteryId());

                if (lotteryCountNumEntity == null) {
                    item.setFunsNum(0);
                    item.setShowNum(0);
                    item.setJoinNum(0);
                } else {
                    item.setFunsNum(lotteryCountNumEntity.getFunsNum());
                    item.setShowNum(lotteryCountNumEntity.getShowNum());
                    item.setJoinNum(lotteryCountNumEntity.getJoinNum());
                }
            });
            return merchantLotteryAwardPageInfo;
        }
    }

    @Override
    public List<MerchantLotteryCountDTO> awardListCount(MerchantLotteryAwardCountParam param) {
        List<MerchantLotteryCountDTO> resultList = new ArrayList<>();

        //获取进行中的数据
        int startCount = lotteryPhaseMapper.queryMerchantPhaseCountByStatus(
                param.getAreaCode(), param.getMerchantId(), LotteryMerchantTabEnum.UNDERWAY.getCode());
        //获取正在进行中的数量
        MerchantLotteryCountDTO startResult = new MerchantLotteryCountDTO();
        startResult.setStatus(LotteryMerchantTabEnum.UNDERWAY.getCode());
        startResult.setCount(startCount);
        resultList.add(startResult);

        //获取未开始的数据
        int notStartCount = lotteryMapper.queryMerchantLotteryCountByStatus(
                param.getAreaCode(), param.getMerchantId(), LotteryStatusEnum.UNDER_PLANNING.getCode());
        MerchantLotteryCountDTO notStartResult = new MerchantLotteryCountDTO();
        notStartResult.setStatus(LotteryMerchantTabEnum.NOT_START.getCode());
        notStartResult.setCount(notStartCount);
        resultList.add(notStartResult);

        //当前结束和下架的期数
        int offlineAndEndCount = lotteryPhaseMapper.queryMerchantPhaseCountByStatus(param.getAreaCode(), param.getMerchantId(), LotteryMerchantTabEnum.END.getCode());
        //返回已结束的tab数量,将结果进行合并
        MerchantLotteryCountDTO endResult = new MerchantLotteryCountDTO();
        endResult.setStatus(LotteryMerchantTabEnum.END.getCode());
        endResult.setCount(offlineAndEndCount);
        resultList.add(endResult);

        return resultList;
    }

    @Override
    public PageWarper<LotteryPhaseOverviewDTO> historyPublishList(LotteryParam param) {
        startPage(param.getPageNum(), param.getPageSize());
        PageWarper<LotteryPhaseOverviewDTO> historyPublishPhasePageInfo = new PageWarper<>(lotteryPhaseMapper.queryHistoryPublishList(param));
        fillJoinTimes(historyPublishPhasePageInfo.getList(), param.getUserId());

        // 给手机号码打码
        for (LotteryPhaseOverviewDTO dto : historyPublishPhasePageInfo.getList()) {
            maskDtoPhone(dto);
        }

        return historyPublishPhasePageInfo;
    }

    @Override
    public PageWarper<LotteryPhaseOverviewDTO> underwayList(LotteryBasePageParam param) {
        startPage(param.getPageNum(), param.getPageSize());
        PageWarper<LotteryPhaseOverviewDTO> underwayPageInfo = new PageWarper<>(lotteryPhaseMapper.queryUnderwayList(param));
        fillJoinTimes(underwayPageInfo.getList(), param.getUserId());

        return underwayPageInfo;
    }

    @Override
    public PageWarper<LotteryPhaseOverviewDTO> recentList(LotteryBasePageParam param) {
        startPage(param.getPageNum(), param.getPageSize());
        PageWarper<LotteryPhaseOverviewDTO> recentPageInfo = new PageWarper<>(lotteryPhaseMapper.queryRecentList(param));
        fillJoinTimes(recentPageInfo.getList(), param.getUserId());

        for (LotteryPhaseOverviewDTO dto : recentPageInfo.getList()) {
            maskDtoPhone(dto);
        }

        return recentPageInfo;
    }

    @Override
    public List<LotteryPhaseOverviewDTO> historyList(LotteryParam param) {
        List<LotteryPhaseOverviewDTO> historyList = lotteryPhaseMapper.queryHistoryList(param);
        for (LotteryPhaseOverviewDTO dto : historyList) {
            maskDtoPhone(dto);
        }
        return historyList;
    }

    @Override
    public PageWarper<LotteryPhaseOverviewDTO> myJoinPhaseList(LotteryBasePageParam param) {

        startPage(param);
        PageWarper<LotteryPhaseOverviewDTO> myJoinPhasePageInfo = new PageWarper<>(lotteryParticipatorMapper.queryMyJoinPhaseQueryList(param.getAreaCode(),
                param.getUserId()));

        fillJoinTimes(myJoinPhasePageInfo.getList(), param.getUserId());
        for (LotteryPhaseOverviewDTO dto : myJoinPhasePageInfo.getList()) {
            maskDtoPhone(dto);
        }

        return myJoinPhasePageInfo;
    }

    @Override
    public PageWarper<LotteryPhaseOverviewDTO> recommendList(LotteryMorePhaseParam param) {
        startPage(param.getPageNum(), param.getPageSize());
        Long currentPhaseId = param.getPhaseId();

        //获取当前商户id，让当前详情页的商户的夺宝信息排在最前面
        if (currentPhaseId != null) {
            LotteryPhaseEntity phaseInfo = lotteryPhaseMapper.selectById(currentPhaseId);
            Long awardId = phaseInfo.getAwardId();

            LotteryAwardEntityWithBLOBs awardInfo = activityAwardMapper.selectByPrimaryKey(awardId);
            if (awardInfo != null) {
                param.setMerchantId(awardInfo.getMerchantId());
            }
        }

        return new PageWarper<>(lotteryPhaseMapper.queryMerchantPhaseList(param));
    }

    @Override
    public LotteryWinnerCarouselDTO lotteryCarousel(LotteryBaseParam param) {
        //1.查询当前地区近30天中奖的用户信息
        List<LotteryCarouselDTO> recentWinnerInfoList = lotteryWinnerMapper.selectRecentWinnerInfo(param);

        //2.获取当前地区中奖人数
        int winnerNum = lotteryWinnerMapper.countWinner(param);

        LotteryWinnerCarouselDTO resultDTO = new LotteryWinnerCarouselDTO();
        resultDTO.setWinnerNum(winnerNum);
        if (CollectionUtils.isEmpty(recentWinnerInfoList)) {
            resultDTO.setCarouselList(Collections.emptyList());
        } else {
            resultDTO.setCarouselList(recentWinnerInfoList);
        }
        return resultDTO;
    }

    @Override
    public PageWarper<LotteryParticipatorDTO> joinRecordList(LotteryParticipatorPageParam param) {
        startPage(param.getPageNum(), param.getPageSize());

        return new PageWarper<>(lotteryParticipatorMapper.joinRecordList(param));
    }

    @Override
    public PageWarper<LotteryWinnerOverviewDTO> myWinnerPhaseList(LotteryBasePageParam param) {
        startPage(param.getPageNum(), param.getPageSize());

        return new PageWarper<>(lotteryWinnerMapper.myWinnerPhaseList(param));
    }

    @Override
    public List<LotteryWinnerInfoDTO> queryWinnerList(LotteryBaseParam param) {
        return lotteryWinnerMapper.queryWinnerList(param);
    }

    /**
     * 填充期数列表项中的当前用户参与次数
     *
     * @param currentUserId        当前用户id
     * @param phaseOverviewDTOList 期数列表
     */
    public void fillJoinTimes(List<LotteryPhaseOverviewDTO> phaseOverviewDTOList, Long currentUserId) {
        List<Long> phaseIds = phaseOverviewDTOList.stream().map(LotteryPhaseBaseInfoDTO::getPhaseId)
                .collect(Collectors.toList());

        //当前用户参与活动的夺宝码信息
        Map<Long, List<String>> myCodes = lotteryCacheManage.getMyCodes(phaseIds, currentUserId);

        phaseOverviewDTOList.forEach(item -> {
            if (myCodes != null && myCodes.get(item.getPhaseId()) != null) {
                List<String> codeList = myCodes.get(item.getPhaseId());
                item.setJoinTimes(codeList.size());
            } else {
                item.setJoinTimes(0);
            }
        });
    }

    /**
     * 对夺宝期数列表项中的状态进行转换（针对已结束列表）
     *
     * @param item 夺宝吸粉已结束列表项
     */
    private void convertMerchantPhaseItem(MerchantLotteryAwardDTO item) {
        //如果期数状态为已下架
        if (LotteryPhaseStatusEnum.CANCEL.getCode().equals(item.getStatus())) {
            item.setUseStatus(UseStatusEnum.OFFLINE.getCode());
        } else {
            // 如果奖品为商家商品则判断是否被核销
            if (AwardTypeEnum.GOODS.getCode().equals(item.getType())) {
                // 如果是商家商品中奖状态为已核销则为已核销状态，其余都为未核销
                if (ActivityPrizeStatusEnum.CHECKED.getCode().equals(item.getCheckStatus())) {
                    item.setUseStatus(UseStatusEnum.HAVE_CHECK.getCode());
                } else {
                    item.setUseStatus(UseStatusEnum.NOT_CHECK.getCode());
                }
            } else {
                //如果是其他奖品则判断是否发奖
                if (ActivityPrizeStatusEnum.GRANT.getCode().equals(item.getCheckStatus())) {
                    item.setUseStatus(UseStatusEnum.HAVE_AWARD.getCode());
                } else if (ActivityPrizeStatusEnum.UN_GRANT.getCode().equals(item.getAwardStatus())) {
                    item.setUseStatus(UseStatusEnum.NOT_AWARD.getCode());
                }
            }
        }
    }

    /**
     * 如果存在中奖用户且手机号码不为空，则给手机号码打码
     *
     * @param phaseDTO 期数dto
     */
    private void maskDtoPhone(LotteryPhaseOverviewDTO phaseDTO) {
        if (phaseDTO.getWinner() != null) {
            LotteryWinnerInfoDTO winnerInfoDTO = phaseDTO.getWinner();
            if (isNotBlank(winnerInfoDTO.getPhone())) {
                winnerInfoDTO.setPhone(StringUtils.hideMobile(winnerInfoDTO.getPhone()));
            }
        }
    }
}