package com.bxm.localnews.merchant.service.lottery.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.common.param.PointReportParam;
import com.bxm.localnews.merchant.coupon.facade.service.CouponFacadeService;
import com.bxm.localnews.merchant.domain.MerchantGoodsMapper;
import com.bxm.localnews.merchant.domain.lottery.ActivityAwardMapper;
import com.bxm.localnews.merchant.domain.lottery.LotteryPhaseMapper;
import com.bxm.localnews.merchant.dto.UserInfoDTO;
import com.bxm.localnews.merchant.dto.activity.*;
import com.bxm.localnews.merchant.entity.coupon.MerchantCouponInfoEntity;
import com.bxm.localnews.merchant.entity.lottery.LotteryAwardEntityWithBLOBs;
import com.bxm.localnews.merchant.entity.lottery.LotteryPhaseEntity;
import com.bxm.localnews.merchant.integration.BizLogIntegrationService;
import com.bxm.localnews.merchant.integration.ImIntegrationService;
import com.bxm.localnews.merchant.integration.UserIntegrationService;
import com.bxm.localnews.merchant.param.BatchMessageParam;
import com.bxm.localnews.merchant.param.activity.LotteryMerchantFollowParam;
import com.bxm.localnews.merchant.service.enums.WelfareTypeEnum;
import com.bxm.localnews.merchant.service.lottery.LotteryMerchantService;
import com.bxm.localnews.merchant.service.lottery.LotteryStatisticsService;
import com.bxm.localnews.merchant.service.lottery.cache.LotteryCacheManage;
import com.bxm.localnews.merchant.service.lottery.event.LotteryFollowEvent;
import com.bxm.localnews.merchant.utils.GoodsPriceRelatedUtils;
import com.bxm.localnews.merchant.vo.goods.MerchantGoodsVo;
import com.bxm.localnews.merchants.facade.service.MerchantInfoFacadeService;
import com.bxm.localnews.merchants.vo.MerchantInfo;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import static java.util.Objects.nonNull;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

/**
 * @author liujia
 * @date 2020-10-28 11:46
 **/
@Slf4j
@Service
public class LotteryMerchantServiceImpl implements LotteryMerchantService {

    private final ActivityAwardMapper activityAwardMapper;

    private final LotteryPhaseMapper lotteryPhaseMapper;

    private final UserIntegrationService userIntegrationService;

    private final MerchantInfoFacadeService merchantInfoFacadeService;

    private final ImIntegrationService imIntegrationService;

    private final CouponFacadeService couponInfoService;

    private final LotteryCacheManage lotteryCacheManage;

    private final LotteryStatisticsService lotteryStatisticsService;

    private final BizLogIntegrationService bizLogIntegrationService;

    private final MerchantGoodsMapper merchantGoodsMapper;

    @Autowired
    public LotteryMerchantServiceImpl(UserIntegrationService userIntegrationService,
                                      MerchantInfoFacadeService merchantInfoFacadeService,
                                      ImIntegrationService imIntegrationService,
                                      LotteryPhaseMapper lotteryPhaseMapper,
                                      ActivityAwardMapper activityAwardMapper,
                                      CouponFacadeService couponInfoService,
                                      LotteryCacheManage lotteryCacheManage,
                                      LotteryStatisticsService lotteryStatisticsService,
                                      BizLogIntegrationService bizLogIntegrationService,
                                      MerchantGoodsMapper merchantGoodsMapper) {
        this.userIntegrationService = userIntegrationService;
        this.merchantInfoFacadeService = merchantInfoFacadeService;
        this.imIntegrationService = imIntegrationService;
        this.lotteryPhaseMapper = lotteryPhaseMapper;
        this.activityAwardMapper = activityAwardMapper;
        this.couponInfoService = couponInfoService;
        this.lotteryStatisticsService = lotteryStatisticsService;
        this.bizLogIntegrationService = bizLogIntegrationService;
        this.lotteryCacheManage = lotteryCacheManage;
        this.merchantGoodsMapper = merchantGoodsMapper;
    }

    @Override
    @Async
    public void execAsyncFollow(LotteryMerchantFollowParam param) {
        //如果商家自己关注自己返回
        if (Objects.equals(param.getMerchantUserId(), param.getUserId())) {
            return;
        }

        // 接口做一下兼容，因为前端在站外下单成功之后调用这个接口，merchantUserId传的是merchantId
        MerchantInfo merchantInfoFrom = merchantInfoFacadeService.getMerchantInfo(param.getMerchantUserId());
        if (nonNull(merchantInfoFrom)) {
            param.setMerchantUserId(merchantInfoFrom.getUserId());
        }

        // 用户关注商家
        userIntegrationService.follow(param.getUserId(), param.getMerchantUserId());

        Long phaseId = param.getPhaseId();
        LotteryPhaseEntity lotteryPhaseEntity = lotteryPhaseMapper.selectById(phaseId);
        if (lotteryPhaseEntity == null) {
            log.warn("当前期数信息不存在：{}", param);
            return;
        }

        // 吸粉数量事件
        lotteryStatisticsService.post(new LotteryFollowEvent(lotteryPhaseEntity.getLotteryId(), lotteryPhaseEntity.getId()));
        // 商家吸粉成功
        bizLogIntegrationService.point(PointReportParam.build()
                .e("3034")
                .ev("129." + param.getPhaseId())
                .put("uid", String.valueOf(param.getMerchantUserId()))
        );

        LotteryAwardEntityWithBLOBs awardInfo = activityAwardMapper.selectByPrimaryKey(lotteryPhaseEntity.getAwardId());

        Long merchantId = awardInfo.getMerchantId();

        //获取商家店铺名称
        MerchantInfo merchantInfo = merchantInfoFacadeService.getMerchantInfo(merchantId);

        //发送关注商家后的消息，如果关注之后商家回复信息不为空
        if (isNotBlank(awardInfo.getReplayContent())) {
            sendFollowMsg(param.getMerchantUserId(), param.getUserId(), awardInfo.getReplayContent());
        }

        // 福利类型为0，则无福利
        if (Objects.equals(awardInfo.getWelfareType(), WelfareTypeEnum.NO_WELFARE.getType())) {
            return;
        }

        // 福利类型为1，则优惠券福利
        if (Objects.equals(awardInfo.getWelfareType(), WelfareTypeEnum.COUPON.getType())) {
            Long couponId = Long.valueOf(awardInfo.getWelfareId());
            //判断当前用户是否还能领取优惠券,如果标记不存在则发放优惠券消息
            if (Boolean.FALSE.equals(lotteryCacheManage.hasFollowGrantCoupon(param.getUserId(), couponId))) {
                sendCouponMsg(param.getUserId(), couponId, merchantInfo);
            }
        }
        // 福利类型为2，则商家团购福利
        if (Objects.equals(awardInfo.getWelfareType(), WelfareTypeEnum.GROUP_BUY.getType())) {
            sendGroupBuyMsg(param.getUserId(), Long.valueOf(awardInfo.getWelfareId()), merchantInfo);
        }

    }

    private void sendGroupBuyMsg(Long userId, Long goodsId, MerchantInfo merchantInfo) {
        ImGroupProductDTO imGroupProductDTO = fixGroupProductInfo(goodsId);
        BatchMessageParam messageParam = new BatchMessageParam();
        messageParam.setContent(JSON.toJSONString(imGroupProductDTO));
        messageParam.setObjectName("LN:GroupProduct");
        messageParam.setFromUserId(merchantInfo.getUserId());
        messageParam.setToUserId(Collections.singletonList(userId));
        //发送im消息
        imIntegrationService.sendMsg(messageParam);
    }


    private ImGroupProductDTO fixGroupProductInfo(Long goodsId) {
        //商品在点击时候才做判断是否下架之类的信息，发送消息直接发送
        MerchantGoodsVo merchantGoodsVo = merchantGoodsMapper.getGoodsInfoById(goodsId);
        if (merchantGoodsVo == null) {
            return ImGroupProductDTO.builder().build();
        }
        // 计算后的vip折扣价格
        BigDecimal vipPrice = GoodsPriceRelatedUtils.getVipPrice(merchantGoodsVo.getPrice(), merchantGoodsVo.getVipDiscount());
        // 折扣价格
        BigDecimal discount = GoodsPriceRelatedUtils.getDiscount(merchantGoodsVo.getPrice(), merchantGoodsVo.getVipDiscount(),
                merchantGoodsVo.getOriginalPrice(), vipPrice);

        String firstImg = getFirstImg(merchantGoodsVo.getHeadPics());
        return ImGroupProductDTO.builder()
                .imgUrl(firstImg)
                .title(merchantGoodsVo.getName())
                .price(merchantGoodsVo.getOriginalPrice().toString())
                .productId(goodsId.toString())
                .realPrice(vipPrice.toString())
                .tag(discount.stripTrailingZeros().toPlainString() + "折")
                .build();
    }

    /**
     * 发送优惠券im消息
     *
     * @param toUserId     目标用户id
     * @param couponId     优惠券id
     * @param merchantInfo 商户信息
     */
    private void sendCouponMsg(Long toUserId, Long couponId, MerchantInfo merchantInfo) {
        ImCouponDTO couponDTO = fillCouponImMsg(merchantInfo, couponId);
        BatchMessageParam messageParam = new BatchMessageParam();
        messageParam.setContent(JSON.toJSONString(couponDTO));
        messageParam.setObjectName("LN:Coupon");
        messageParam.setFromUserId(merchantInfo.getUserId());
        messageParam.setToUserId(Collections.singletonList(toUserId));

        //当用户第一次通过关注领取一张优惠券做一个标记，再次关注不会在发放优惠券
        lotteryCacheManage.followGrantUserCoupon(toUserId, couponId);
        //发送im消息
        imIntegrationService.sendMsg(messageParam);
    }


    /**
     * 给用户发送关注回复消息
     *
     * @param merchantUserId 商户用户id
     * @param userId         用户id
     * @param replyContent   回复消息内容
     */
    private void sendFollowMsg(Long merchantUserId, Long userId, String replyContent) {
        BatchMessageParam messageParam = new BatchMessageParam();
        messageParam.setContent(replyContent);
        messageParam.setObjectName("RC:TxtMsg");
        messageParam.setFromUserId(merchantUserId);
        messageParam.setToUserId(Collections.singletonList(userId));

        //发送im消息
        imIntegrationService.sendMsg(messageParam);
    }

    /**
     * 组装填充要发送到im的消息
     *
     * @param merchantInfo 商户信息
     * @param couponId     优惠券id
     */
    private ImCouponDTO fillCouponImMsg(MerchantInfo merchantInfo, Long couponId) {
        MerchantCouponInfoEntity couponDetail = couponInfoService.getCouponDetail(couponId);
        if (couponDetail == null) {
            log.warn("当前优惠券信息为空：{}", couponId);
            return new ImCouponDTO();
        }
        ImCouponDTO couponMsg = new ImCouponDTO();
        couponMsg.setCouponId(couponId);

        BigDecimal discount = couponDetail.getDiscount();
        if (Objects.nonNull(discount)) {
            couponMsg.setAmount(discount.setScale(2, BigDecimal.ROUND_DOWN));
        }

        Integer conditionAmount = couponDetail.getConditionAmount();
        //判断是否有门槛
        if (conditionAmount == 0) {
            couponMsg.setLimit("无门槛");
        } else if (conditionAmount == 1) {
            BigDecimal totalAmount = couponDetail.getTotalAmount();
            if (Objects.nonNull(totalAmount)) {
                couponMsg.setLimit(String.format("满%s元可用", totalAmount.setScale(2, BigDecimal.ROUND_DOWN)));
            }
        }

        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
        Date usableStartTime = couponDetail.getUsableStartTime();
        Date usableEndTime = couponDetail.getUsableEndTime();

        String usableDate = sdf.format(usableStartTime) + "-" + sdf.format(usableEndTime);
        couponMsg.setDate(usableDate);

        couponMsg.setStore(merchantInfo.getMerchantName());

        return couponMsg;
    }

    @Override
    public LotteryMerchantInfoDTO get(Long merchantId) {
        LotteryMerchantInfoDTO resultDTO = new LotteryMerchantInfoDTO();
        MerchantInfo merchantInfo = merchantInfoFacadeService.getMerchantInfo(merchantId);
        if (Objects.nonNull(merchantInfo)) {
            convert2LotteryMerchantInfoDTO(merchantInfo, resultDTO);
        }

        return resultDTO;
    }


    @Override
    public LotteryMerchantInfoDTO get(Long merchantId, Long currentUserId) {
        LotteryMerchantInfoDTO resultDTO = get(merchantId);
        //设置用户是否关注商家
        if (null != currentUserId) {
            resultDTO.setFollow(userIntegrationService.isFollow(currentUserId, resultDTO.getMerchantUserId()));
        }

        return resultDTO;
    }

    @Override
    public MerchantLastPhaseDTO getMerchantHasProceedPhase(Long merchantId) {
        return lotteryPhaseMapper.queryMerchantLastProceedPhaseId(merchantId);
    }

    private void convert2LotteryMerchantInfoDTO(MerchantInfo merchantInfo, LotteryMerchantInfoDTO resultDTO) {
        resultDTO.setMerchantId(merchantInfo.getId());
        resultDTO.setMerchantName(merchantInfo.getMerchantName());
        resultDTO.setAddress(merchantInfo.getAddress());
        // 如果logo为空的话，使用商家头图第一张
        if (StringUtils.isNotBlank(merchantInfo.getLogo())) {
            resultDTO.setLogoImg(merchantInfo.getLogo());
        } else {
            String headPics = merchantInfo.getHeadPics();
            if (StringUtils.isNotBlank(headPics)) {
                String[] firstHeadPic = headPics.split(",");
                resultDTO.setLogoImg(firstHeadPic[0]);
            }
        }
        resultDTO.setDescription(merchantInfo.getDes());
        resultDTO.setMerchantUserId(merchantInfo.getUserId());
        UserInfoDTO userCache = userIntegrationService.getUserFromRedisDB(merchantInfo.getUserId());
        if (userCache != null) {
            resultDTO.setMerchantUserHeadImg(userCache.getHeadImg());
            resultDTO.setMerchantUserName(userCache.getNickname());
        }
        resultDTO.setLat(merchantInfo.getLat());
        resultDTO.setLng(merchantInfo.getLng());
    }

    @Override
    public List<MerchantGoodsListDTO> getMerchantGoods(Long merchantId) {
        List<MerchantGoodsVo> merchantGoodsVos = merchantGoodsMapper.selectByMerchantId(merchantId, 2);
        return merchantGoodsVos.stream()
                .map(merchantGoodsVo -> {
                    MerchantGoodsListDTO goodsListDTO = new MerchantGoodsListDTO();
                    if (nonNull(merchantGoodsVo)) {
                        BeanUtils.copyProperties(merchantGoodsVo, goodsListDTO);
                        if (org.apache.commons.lang.StringUtils.isNotBlank(merchantGoodsVo.getHeadPics())) {
                            goodsListDTO.setImg(getFirstImg(merchantGoodsVo.getHeadPics()));
                        }
                        // 计算后的vip折扣价格
                        BigDecimal vipPrice = GoodsPriceRelatedUtils.getVipPrice(merchantGoodsVo.getPrice(), goodsListDTO.getVipDiscount());
                        // 折扣价格
                        BigDecimal discount = GoodsPriceRelatedUtils.getDiscount(merchantGoodsVo.getPrice(), goodsListDTO.getVipDiscount(), merchantGoodsVo.getOriginalPrice(),
                                vipPrice);

                        goodsListDTO.setVipPrice(vipPrice);
                        goodsListDTO.setVipDiscount(merchantGoodsVo.getVipDiscount());
                        goodsListDTO.setDiscount(discount);
                        goodsListDTO.setGoodsId(merchantGoodsVo.getId());
                        goodsListDTO.setGazePeopleNum(merchantGoodsVo.getGazePeopleNum());
                    }
                    return goodsListDTO;
                }).collect(Collectors.toList());
    }

    private String getFirstImg(String goodsImg) {
        if (org.apache.commons.lang.StringUtils.isNotBlank(goodsImg) && goodsImg.contains(",")) {
            String[] imgs = goodsImg.split(",");
            if (imgs.length > 0) {
                return imgs[0];
            }
        }
        return goodsImg;
    }
}
