package com.bxm.localnews.merchant.service.lottery.impl;

import com.bxm.localnews.common.constant.PlatformEnum;
import com.bxm.localnews.merchant.domain.lottery.LotteryParticipatorMapper;
import com.bxm.localnews.merchant.domain.lottery.LotteryPhaseMapper;
import com.bxm.localnews.merchant.dto.UserInfoDTO;
import com.bxm.localnews.merchant.dto.activity.LotteryJoinResultDTO;
import com.bxm.localnews.merchant.dto.activity.LotteryJoinStateDTO;
import com.bxm.localnews.merchant.dto.activity.LotteryPhaseCoreInfoDTO;
import com.bxm.localnews.merchant.entity.lottery.LotteryParticipatorEntity;
import com.bxm.localnews.merchant.integration.UserIntegrationService;
import com.bxm.localnews.merchant.param.activity.LotteryAddVirtualParam;
import com.bxm.localnews.merchant.param.activity.LotteryCheckJoinTimesParam;
import com.bxm.localnews.merchant.param.activity.LotteryPhaseJoinParam;
import com.bxm.localnews.merchant.service.config.ActivityProperties;
import com.bxm.localnews.merchant.service.enums.LotteryJoinSourceEnum;
import com.bxm.localnews.merchant.service.enums.LotteryUserTypeEnum;
import com.bxm.localnews.merchant.service.lottery.LotteryParticipatorService;
import com.bxm.localnews.merchant.service.lottery.cache.LotteryCacheManage;
import com.bxm.localnews.merchant.service.lottery.context.LotteryJoinContext;
import com.bxm.localnews.merchant.service.lottery.strategy.JoinStrategyManage;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.RandomUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import static com.bxm.localnews.merchant.common.config.RedisConfig.LOTTERY_TEMP_VIRTUAL_KEY;

/**
 * @author liujia
 * @date 2020-10-27 15:01
 **/
@Service
@Slf4j
public class LotteryParticipatorServiceImpl implements LotteryParticipatorService {

    private final JoinStrategyManage joinStrategyManage;

    private final LotteryParticipatorMapper lotteryParticipatorMapper;

    private final SequenceCreater sequenceCreater;

    private final LotteryCacheManage lotteryCacheManage;

    private final UserIntegrationService userIntegrationService;

    private final LotteryPhaseMapper lotteryPhaseMapper;

    private final RedisStringAdapter redisStringAdapter;

    private final ActivityProperties activityProperties;

    @Autowired
    public LotteryParticipatorServiceImpl(JoinStrategyManage joinStrategyManage,
                                          LotteryParticipatorMapper lotteryParticipatorMapper,
                                          SequenceCreater sequenceCreater,
                                          LotteryCacheManage lotteryCacheManage,
                                          UserIntegrationService userIntegrationService,
                                          LotteryPhaseMapper lotteryPhaseMapper,
                                          RedisStringAdapter redisStringAdapter,
                                          ActivityProperties activityProperties) {
        this.joinStrategyManage = joinStrategyManage;
        this.lotteryParticipatorMapper = lotteryParticipatorMapper;
        this.sequenceCreater = sequenceCreater;
        this.lotteryCacheManage = lotteryCacheManage;
        this.userIntegrationService = userIntegrationService;
        this.lotteryPhaseMapper = lotteryPhaseMapper;
        this.redisStringAdapter = redisStringAdapter;
        this.activityProperties = activityProperties;
    }

    @Override
    public LotteryJoinResultDTO join(LotteryPhaseJoinParam param) {
        return joinStrategyManage.execute(LotteryJoinContext.builder().param(param).build());
    }

    @Override
    public Message save(LotteryParticipatorEntity entity) {
        entity.setId(sequenceCreater.nextLongId());
        return Message.build(lotteryParticipatorMapper.insert(entity));
    }

    @Override
    public int countUserType(Long phaseId, LotteryUserTypeEnum userType) {
        return lotteryParticipatorMapper.countUserType(phaseId, userType.getCode());
    }

    @Override
    public void addVirtual(Long phaseId) {
        KeyGenerator key = LOTTERY_TEMP_VIRTUAL_KEY.copy().appendKey(phaseId).appendKey(getHalfHourKey());

        List<LotteryParticipatorEntity> virtualUsers;

        if (redisStringAdapter.hasKey(key)) {
            TypeReference<List<LotteryParticipatorEntity>> typeReference = new TypeReference<List<LotteryParticipatorEntity>>() {
            };
            virtualUsers = redisStringAdapter.get(key, typeReference);
        } else {
            //创建半个小时以后的虚拟用户,剩余分钟数，进行取数权重的降低
            int currentMin = 30 - (LocalDateTime.now().getMinute() % 30);

            int randomNum = RandomUtils.nextInt(activityProperties.getHalfHourMinNum() * currentMin / 30,
                    activityProperties.getHalfHourMaxNum() * currentMin / 30);

            if (randomNum < 1) {
                randomNum = 1;
            }

            List<UserInfoDTO> virtualUserList = userIntegrationService.getVirtualUserList(randomNum);

            virtualUsers = virtualUserList.stream().map(user -> {
                return LotteryParticipatorEntity.builder()
                        .createTime(getRandomDate())
                        .headImg(user.getHeadImg())
                        .nickName(user.getNickname())
                        .userId(user.getId())
                        .build();
            }).collect(Collectors.toList());

            //根据时间排序
            virtualUsers.sort((u1, u2) -> {
                return (int) (u1.getCreateTime().getTime() - u2.getCreateTime().getTime());
            });

            redisStringAdapter.set(key, virtualUsers, 60 * 30);
        }


        List<LotteryParticipatorEntity> afterJoin = Lists.newArrayList();
        afterJoin.addAll(virtualUsers);

        //将小于当前时间的虚拟数据添加到参与用户中
        virtualUsers.forEach(user -> {
            if (DateUtils.before(user.getCreateTime())) {
                addVirtualJoin(user.getUserId(), phaseId, null, false);

                afterJoin.remove(user);
            }
        });

        redisStringAdapter.set(key, afterJoin, 60 * 30);
    }

    @Override
    public void addVirtual(LotteryAddVirtualParam param) {
        if (null == param.getNum() || null == param.getPhaseId()) {
            return;
        }

        List<UserInfoDTO> virtualUserList = userIntegrationService.getVirtualUserList(param.getNum());

        int seconds = 1;
        for (UserInfoDTO userInfoDTO : virtualUserList) {
            Date createTime = new Date();
            createTime = DateUtils.addField(createTime, Calendar.SECOND, seconds);
            createTime = DateUtils.addField(createTime, Calendar.MILLISECOND, RandomUtils.nextInt(0, 1000));

            addVirtualJoin(userInfoDTO.getId(), param.getPhaseId(), createTime, true);

            seconds += RandomUtils.nextInt(1, param.getSeconds());
        }
    }

    private void addVirtualJoin(Long userId, Long phaseId, Date createTime, Boolean skipLimit) {
        LotteryPhaseJoinParam param = new LotteryPhaseJoinParam();
        param.setUserId(userId);
        param.setPhaseId(phaseId);

        if (createTime == null) {
            createTime = new Date();
        }

        DateUtils.setField(createTime, Calendar.MILLISECOND, RandomUtils.nextInt(0, 1000));

        LotteryJoinContext joinContext = LotteryJoinContext.builder()
                .param(param)
                .joinSource(LotteryJoinSourceEnum.VIRTUAL)
                .joinTime(createTime)
                .skipActualLimit(skipLimit)
                .build();

        joinStrategyManage.execute(joinContext);
    }

    /**
     * 获取当前时间到下个整点半小时的随机时间
     */
    private Date getRandomDate() {
        Date now = new Date();

        Calendar calendar = Calendar.getInstance();
        calendar.setTime(now);

        Calendar afterHalfHour = Calendar.getInstance();
        afterHalfHour.setTime(now);
        afterHalfHour.set(Calendar.SECOND, 0);
        afterHalfHour.set(Calendar.MILLISECOND, 0);

        int minute = afterHalfHour.get(Calendar.MINUTE);

        if (minute >= 30) {
            afterHalfHour.set(Calendar.MINUTE, 0);
            afterHalfHour.add(Calendar.HOUR, 1);
        } else {
            afterHalfHour.set(Calendar.MINUTE, 30);
        }

        int diffMillSeconds = (int) (afterHalfHour.getTimeInMillis() - calendar.getTimeInMillis());

        calendar.add(Calendar.MILLISECOND, RandomUtils.nextInt(0, diffMillSeconds));

        return calendar.getTime();
    }

    private String getHalfHourKey() {
        LocalDateTime now = LocalDateTime.now();
        int minute = now.getMinute();

        String key = now.getDayOfMonth() + "" + now.getHour();
        if (minute >= 30) {
            key += "30";
        } else {
            key += "00";
        }

        return key;
    }

    @Override
    public LotteryJoinStateDTO getJoinInfo(LotteryCheckJoinTimesParam param) {
        LotteryJoinStateDTO stateDTO = new LotteryJoinStateDTO();

        stateDTO.setNewbie(false);
        stateDTO.setFree(false);

        // 如果是站外
        if (Objects.equals(PlatformEnum.WEB.getCode(), param.getPlatform())) {
            if (lotteryCacheManage.hasGlobalJoinTimes(param.getUserId())) {
                stateDTO.setJoinTimes(1);
            } else {
                stateDTO.setJoinTimes(0);
            }
            stateDTO.setFree(true);
        } else {
            stateDTO.setNewbie(isNewbie(param));

            if (stateDTO.getNewbie()) {
                stateDTO.setFree(lotteryCacheManage.hasFreeJoinTime(param.getUserId()));
            }
            // 设置参与限制
            setMaxTimesAndMaxType(param);

            stateDTO.setJoinTimes((int) (param.getMaxTimes() -
                    lotteryCacheManage.getAppJoinTimes(param.getPhaseId(), param.getUserId(), param.getMaxType())));
        }

        return stateDTO;
    }

    private void setMaxTimesAndMaxType(LotteryCheckJoinTimesParam param) {
        if (null == param.getMaxTimes() || Objects.isNull(param.getMaxType())) {
            LotteryPhaseCoreInfoDTO phaseInfo = lotteryPhaseMapper.getCorePhaseInfo(param.getPhaseId());
            param.setMaxTimes(phaseInfo.getMaxTimes());
            param.setMaxType(phaseInfo.getMaxType());
        }
    }

    private boolean isNewbie(LotteryCheckJoinTimesParam param) {
        Date userCreateTime = param.getUserCreateTime();
        if (userCreateTime == null) {
            UserInfoDTO cacheUser = userIntegrationService.getUserFromRedisDB(param.getUserId());
            userCreateTime = cacheUser.getCreateTime();
        }

        return DateUtils.isSameDay(userCreateTime, new Date());
    }

}
